/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#include "samhain.h"
#include "sh_error.h"
#include "sh_utils.h"
#include "sh_mem.h"

#undef  FIL__
#define FIL__  _("sh_mem.c")
static int eblock = 0;

#ifdef MEM_DEBUG

#define CHECKBYTE 0x7F

/* Memory alignment; should be 16 bytes on 64 bit machines.
 * -> 32 bytes overhead/allocation 
 */
#define SH_MEMMULT 16

typedef struct mem_struct {
  struct mem_struct *next;        /* link to next struct    */
  char * real_address;            /* address assigned       */
  char * address;                 /* address returned       */
  unsigned long size;             /* size allocated         */
  char file[20];                  /* Allocation file name   */
  int line;                       /* Allocation line number */
} memlist_t;

memlist_t   * memlist       = NULL;

int           Free_Count  = 0, Alloc_Count = 0;
unsigned long Mem_Current = 0, Mem_Max = 0;


void sh_mem_stat ()
{
  memlist_t   * this;

  if (Alloc_Count == Free_Count) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_MSTAMP,
		       Mem_Max, Mem_Current);
      return;
    }
    
  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, 0, MSG_MSTAMP2,
		   Alloc_Count, Free_Count);
  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, 0, MSG_MSTAMP,
		   Mem_Max, Mem_Current);

  this = memlist;

  while (this != NULL) 
    {
      sh_error_handle (SH_ERR_WARN, FIL__, __LINE__, 0, MSG_E_NOTFREE,
		       this->size, this->file, this->line);
      this = this->next;
    }

  return;
}

void sh_mem_check ()
{
  memlist_t * this;

  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_MSTAMP,
		   Mem_Max, Mem_Current);

  this = memlist;

  while (this != NULL) 
    {
      if ( this->address[this->size]        != CHECKBYTE )
	sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MOVER,
			 this->file, this->line, FIL__, __LINE__);

      if ( this->real_address[SH_MEMMULT-1] != CHECKBYTE )
	sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MUNDER,
			 this->file, this->line, FIL__, __LINE__);

      this = this->next;
    }

  return;
}

void * sh_mem_malloc (unsigned long size, char * file, int line)
{
  void      * the_realAddress;
  void      * theAddress;
  memlist_t * this;

  the_realAddress = my_malloc(size + 2 * SH_MEMMULT);
  
  if ( the_realAddress  == NULL ) 
    {
      if (eblock == 0)
	{
	  eblock = 1;
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MMEM,
			   file, line);
	  eblock = 0;
	}
      /* use _exit() rather than exit() - we malloc() in atexit() functions 
       */
      _exit (EXIT_FAILURE);
    }
  
  /* --- Set check bytes. --- 
   */
  theAddress = ((char *) the_realAddress + SH_MEMMULT);

  memset(the_realAddress, CHECKBYTE, SH_MEMMULT);
  memset(theAddress,      CHECKBYTE, size + 1);
  memset(theAddress,      0,         1);

  ++Alloc_Count;

  Mem_Current += size;
  Mem_Max = ( (Mem_Current > Mem_Max) ? Mem_Current : Mem_Max);

  this = (memlist_t *) my_malloc (sizeof(memlist_t));

  if ( this == NULL) 
    {
      if (eblock == 0)
	{
	  eblock = 1;
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MNULL,
			   file, line);
	  eblock = 0;
	}
      return the_realAddress;
    }

  /* make list entry */

  this->real_address = the_realAddress;
  this->address      = theAddress;
  this->size         = size;
  this->line         = line;
  sl_strlcpy(this->file, file, 20);

  this->next = memlist;
  memlist = this;

  return theAddress;
}


void sh_mem_free (void * a, char * file, int line)
{
  memlist_t * this   = memlist;
  memlist_t * before = memlist;
  unsigned long size = 0;

  if ( a == NULL ) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MNULL,
		       file, line);
      return;
    }
    
  /* -- Find record. -- 
   */
  while (this != NULL) 
    {
      if (this->address == a) 
	break;
      before = this;
      this   = this->next;
    }

  if (this == NULL) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MREC,
		       file, line);
      return;
    } 
  else 
    {

      a = this->real_address;

      if ( this->address[this->size]        != CHECKBYTE )
	sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MOVER,
			 this->file, this->line, file, line);
      if ( this->real_address[SH_MEMMULT-1] != CHECKBYTE )
	sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MUNDER,
			 this->file, this->line, file, line);

      size = this->size;

      if (this == memlist) 
	memlist = this->next;
      else 
	before->next = this->next;
    }


  my_free(a);
  if (this)
    my_free(this);
  ++Free_Count;
  Mem_Current -= size;
  return;
}

#else

void sh_mem_free (void * a)
{
  
  if ( a == NULL ) 
    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MNULL);
  else
    {
      my_free(a);
      a = NULL;
    }
  return;
}

void * sh_mem_malloc (unsigned long size)
{
  void * theAddress;

  theAddress = my_malloc(size);

  if ( theAddress == NULL ) 
    {
      if (eblock == 0)
	{
	  eblock = 1;
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_MMEM);
	  eblock = 0;
	}
      /* use _exit() rather than exit() - we malloc() in atexit()  
       */
      _exit (EXIT_FAILURE);
    }
  return theAddress;
}
#endif
