/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2001 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#ifdef SH_USE_KERN
#define __KERNEL__
#include <linux/module.h>
#undef __KERNEL__
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>


#ifdef SH_USE_KERN

#undef  FIL__
#define FIL__  _("sh_kern.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_modules.h"
#include "sh_kern.h"
#include "sh_ks.h"



sh_rconf sh_kern_table[] = {
  {
    N_("severitykernel"),
    sh_kern_set_severity
  },
  {
    N_("kernelcheckactive"),
    sh_kern_set_activate
  },
  {
    N_("kernelcheckinterval"),
    sh_kern_set_timer
  },
  {
    NULL,
    NULL
  },
};


static time_t  lastcheck;
static int     ShKernActive   = S_TRUE;
static int     ShKernInterval = 300;
static int     ShKernSeverity = SH_ERR_SEVERE;




int query_module(const char *name, int which, void *buf, size_t bufsize,
                 size_t *ret);

struct new_module_symbol
{
  unsigned long value;
  unsigned long name;
};

#define QM_SYMBOLS     4


long my_address (char *sym_name)
{
  struct new_module_symbol *buf;
  size_t ret, j, size;
  long   retval;
  int    status;
  char   *p;

  buf = (struct new_module_symbol *) 
    my_malloc (sizeof(struct new_module_symbol));

  if (buf == NULL)
    {
      TPT((0, FIL__, __LINE__, _("msg=<Out of memory>\n")));
      return -1;
    }

  size = sizeof(struct new_module_symbol);

  while (0 != query_module(NULL, QM_SYMBOLS, buf, 
			   size,
			   &ret))
    {
      if (errno == ENOSPC)
	{
	  my_free(buf);
	  size = ret;
	  buf  = (struct new_module_symbol *) my_malloc (size);
	  if (buf == NULL)
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Out of memory>\n")));
	      return -1;
	    }
	}
      else
	{
	  status = errno;
	  TPT((0, FIL__, __LINE__, _("msg=<query_module: %s>\n"), 
	       sh_error_message(status)));
	  my_free(buf);
	  return -1;
	}
    }

  
  for (j = 0; j < ret; ++j)
    {
      p = (char *) buf; p += buf[j].name;
      if(strstr(p, sym_name))
	{
	  retval = (long) buf[j].value;
	  my_free(buf);
	  return retval;
	}
    }

  TPT((0, FIL__, __LINE__, _("msg=<%s kmem offset not found>\n"), sym_name));
  my_free(buf);
  return -1;
}


int sh_kern_check_internal ()
{
  int kd;
  int i, status = 0;
  unsigned int kaddr;
  unsigned long kmem_call_table[512];
#ifdef SH_USE_LKM
  static int check_getdents      = 0;
#endif

  kaddr = my_address (_("sys_call_table"));

  if (kaddr == (unsigned int) -1)
    {
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error finding sys_call_table"),
		       _("kern_check_internal") );
      return (-1);
    }
  
  kd = aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/kmem"), O_RDONLY, 0);
  
  if (kd < 0)
    {
      status = errno;
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error opening /dev/kmem"),
		       _("kern_check_internal") );
      return (-1);
    }

  if(lseek(kd, (off_t)kaddr, SEEK_SET) == -1)
    {
      status = -1;
    }

  if(status == 0)
    {
      if (sizeof(kmem_call_table) != 
	  read(kd, &kmem_call_table, sizeof(kmem_call_table)))
	{
	  status = -2;
	}
    }

  close(kd);

  if ( status < 0)
    {
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error reading from /dev/kmem"),
		       _("kern_check_internal") );
      return (-1);
    }
  
  for (i = 0; i < SH_MAXCALLS; ++i) 
    {
      if (sh_syscalls[i].name == NULL || sh_syscalls[i].addr == 0UL)
	break;

      /*
      fprintf(stderr, "Kernel: %#lx Map: %#lx  %s\n", 
	      kmem_call_table[i], 
	      sh_syscalls[i].addr, 
	      sh_syscalls[i].name);
      */
      
#ifdef SH_USE_LKM
      if (sh_syscalls[i].addr != kmem_call_table[i])
	{
	  if (check_getdents == 0 && 
	      0 == strcmp(_(sh_syscalls[i].name), _("sys_getdents")))
	    {
	      check_getdents = 1;
	      sh_error_handle (SH_ERR_WARN, FIL__, __LINE__, 
			       status, MSG_E_SUBGEN,
			       _("Modified kernel syscall (expected)."),
			       _(sh_syscalls[i].name) );
	    }
	  else
	    {
	      sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			       status, MSG_KERN_POLICY,
			       kmem_call_table[i],
			       sh_syscalls[i].addr,
			       _(sh_syscalls[i].name)
			       );
	    }
	  sh_syscalls[i].addr = kmem_call_table[i];
	}
#else
      if (sh_syscalls[i].addr != kmem_call_table[i])
	{
	  sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			   status, MSG_KERN_POLICY,
			   kmem_call_table[i],
			   sh_syscalls[i].addr,
			   _(sh_syscalls[i].name)
			   );
	  sh_syscalls[i].addr = kmem_call_table[i];
	}
#endif
    }

  return 0;
}

/*************
 *
 * module init
 *
 *************/
int sh_kern_init ()
{
  if (ShKernActive == S_FALSE)
    return (-1);

  lastcheck  = time (NULL);
  sh_kern_check_internal ();
  return (0);
}

/*************
 *
 * module cleanup
 *
 *************/
int sh_kern_end ()
{
  return (0);
}


/*************
 *
 * module timer
 *
 *************/
int sh_kern_timer (unsigned long tcurrent)
{
  if ((time_t) (tcurrent - lastcheck) > ShKernInterval)
    {
      lastcheck  = tcurrent;
      return (-1);
    }
  return 0;
}

/*************
 *
 * module check
 *
 *************/
int sh_kern_check ()
{
  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, EINVAL, MSG_E_SUBGEN,
		   _("Checking kernel syscalls"),
		   _("kern_check") );
  return (sh_kern_check_internal ());
}

/*************
 *
 * module setup
 *
 *************/

int sh_kern_set_severity  (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShKernSeverity);
  return 0;
}

int sh_kern_set_timer (char * c)
{
  long val;

  val = strtol (c, (char **)NULL, 10);
  if (val <= 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("kern timer"), c);

  val = (val <= 0 ? 60 : val);

  ShKernInterval = (time_t) val;
  return 0;
}

int sh_kern_set_activate (char * c)
{
  return sh_util_flagval(c, &ShKernActive);
}



#endif


/* #ifdef SH_USE_UTMP */
#endif



