/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

/* make sure state changes of a file are always reported, even
 *  with reportonlyonce=true
 */
#define REPLACE_OLD

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>


#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)

#include "sh_hash.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_tiger.h"
#include "sh_gpg.h"
#include "sh_unix.h"

#if defined(SH_WITH_CLIENT)
#include "sh_forward.h"
#endif

#undef  FIL__
#define FIL__  _("sh_hash.c")

/* MAX_PATH_STORE must be >= KEY_LEN
 */
#define MAX_PATH_STORE SH_BUFSIZE+1

typedef struct store_info {
  unsigned short   mark;
  /*
  char             fullpath[MAX_PATH_STORE+2];
  char             linkpath[MAX_PATH_STORE+2];
  */
  unsigned int     mode;
  unsigned int     linkmode;
  char             c_mode[11];
#if defined(__linux__)
  unsigned long    attributes;
  char             c_attributes[16];
#endif
  long             owner;
  char             c_owner[USER_MAX+2];
  long             group;
  char             c_group[GROUP_MAX+2];
  unsigned long    hardlinks;
  unsigned long    ino;
  unsigned long    size;
  unsigned long    atime;
  unsigned long    mtime;
  unsigned long    ctime;
  char             checksum[KEY_LEN+1];
} sh_filestore_t;
  
typedef struct file_info {
  sh_filestore_t   theFile;
  char           * fullpath;
  char           * linkpath;
  int              visited;
  int              reported;
  int              allignore;
  unsigned long    modi_mask;
  struct           file_info * next;
} sh_file_t;

  static char  *policy[] = {
    N_("[]"),
    N_("[ReadOnly]"),
    N_("[LogFiles]"),
    N_("[GrowingLogs]"),
    N_("[IgnoreNone]"),
    N_("[IgnoreAll]"),
    N_("[Attributes]"),
    NULL
  };

/* For added files not in database
 */
typedef struct _list_added {
  char               * name;
  int                  visited;
  int                  class;
  struct _list_added * next;
} list_added;

list_added * addList = NULL;


/**********************************
 *
 * hash table functions
 *
 **********************************
 */

#include "sh_hash.h"

/* must fit an int              */
#define TABSIZE 2048  

/* must fit an unsigned short   */
/* changed for V0.8, as the     */
/* database format has changed  */

/* changed again for V0.9       */
#define REC_MAGIC 19

static sh_file_t * tab[TABSIZE];

/**************************************************************
 *
 * compute hash function
 *
 **************************************************************/
static int hashfunc(char *s) 
{
  unsigned n = 0; 

  for ( ; *s; s++) 
    n = 31 * n + *s; 
  return n % TABSIZE; 
} 


/**************************************************************
 *
 * search for files not visited, and check whether they exist
 *
 **************************************************************/
static void hash_unvisited (sh_file_t *p, ShErrLevel level)
{
  struct stat buf;
  int i;
  char * tmp;

  if (p->next != NULL)
    hash_unvisited (p->next, level);

  if (p->visited == S_FALSE && p->reported == S_FALSE 
      && p->allignore == S_FALSE)
    {
      sl_set_suid(); 
      i = retry_lstat(p->fullpath, &buf);
      sl_unset_suid();

      if (0 != i)
	{
	  tmp = sh_util_safe_name(p->fullpath);
	  sh_error_handle (level, FIL__, __LINE__, 0, MSG_FI_MISS, tmp);
	  SH_FREE(tmp);
	  if (sh.flag.reportonce == S_TRUE)
	    p->reported = S_TRUE;
	}
    }

  else if (p->visited == S_TRUE && p->reported == S_TRUE 
	   && p->allignore == S_FALSE)
    {
      tmp = sh_util_safe_name(p->fullpath);
      sh_error_handle (level, FIL__, __LINE__, 0, MSG_FI_ADD, tmp);
      SH_FREE(tmp);

      p->reported = S_FALSE;
    }

  if (sh.flag.reportonce == S_FALSE)
    p->reported = S_FALSE;

  p->visited = S_FALSE;
  return;
}


/*********************************************************************
 *
 * Search for files in the database that have been deleted from disk.
 *
 *********************************************************************/
void sh_hash_unvisited (ShErrLevel level)
{
  int i;
  char * tmp;
  list_added * addCheck = addList;
  list_added * addPrev  = addList;
  list_added * addCurr  = NULL;

  for (i = 0; i < TABSIZE; ++i)
    {
      if (tab[i] != NULL) 
	hash_unvisited (tab[i], level);
    }

  while (addCheck)
    {
      if (addCheck->visited == S_FALSE)
	{
	  addCurr       = addCheck;
	  if (addPrev == addList)
	    {
	      addList = addCheck->next;
	      addPrev = addCheck->next;
	    }
	  else
	    addPrev->next = addCheck->next;
	  addCheck      = addCheck->next;

	  tmp = sh_util_safe_name(addCurr->name);
	  sh_error_handle (ShDFLevel[addCurr->class], FIL__, __LINE__, 0, 
			   MSG_FI_MISS, 
			   tmp);
	  SH_FREE(tmp);

	  SH_FREE(addCurr->name);
	  SH_FREE(addCurr);
	}
      else
	{
	  /* Reset the flag.
	   */
	  addCheck->visited = S_FALSE;
	  addPrev           = addCheck;
	  addCheck          = addCheck->next;
	}
    }

  return;
}


/**********************************************************************
 *
 * delete hash array
 *
 **********************************************************************/
static void hash_kill (sh_file_t *p)
{

  if (p == NULL)
    return;

  if (p->next != NULL)
    hash_kill (p->next);

  if (p->fullpath)
    {
      SH_FREE(p->fullpath);
      p->fullpath = NULL;
    }
  if (p->linkpath)
    {
      SH_FREE(p->linkpath);
      p->linkpath = NULL;
    }
  SH_FREE(p);
  p = NULL;
  return;
}


/***********************************************************************
 *
 * get info out of hash array
 *
 ***********************************************************************/
static sh_file_t * hashsearch (char * s) 
{
  sh_file_t * p;

  if (!s)
    return NULL;

  for (p = tab[hashfunc(s)]; p; p = p->next) 
    if (sl_strcmp(s, p->fullpath) == 0) 
      return p; 
  return NULL;
} 


/***********************************************************************
 *
 * insert into hash array
 *
 ***********************************************************************/
static void hashinsert (sh_file_t * s) 
{
  sh_file_t * p;

  int key = hashfunc(s->fullpath);

  if (tab[key] == NULL) 
    {
      tab[key] = s;
      tab[key]->next = NULL;
      return;
    } 
  else 
    {
      p = tab[key];
      while (1) 
	{
	  if (p->next == NULL) 
	    {
	      p->next = s;
	      p->next->next = NULL;
	      return;
	    }
	  p = p->next;
	}
    }
  /* notreached */
}


/******************************************************************
 *
 * ------- Check functions -------
 *
 ******************************************************************/

static int IsInit = 0;


/******************************************************************
 *
 * Fast forward to start of data
 *
 ******************************************************************/
int sh_hash_setdataent (SL_TICKET fd, char * line)
{
  long i;

  sl_rewind (fd);

  while (1) 
    {
      i =  sh_unix_getline (fd, line, SH_BUFSIZE);
      if (i < 0 ) 
	{
	  SH_FREE(line);
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_P_NODATA,
			   file_path('D', 'R'));
	  aud_exit (FIL__, __LINE__, EXIT_FAILURE);
	}

#if defined(SH_STEALTH)
      if (0 == sl_strncmp (line, N_("[SOF]"), 5)) 
#else
      if (0 == sl_strncmp (line, _("[SOF]"),  5)) 
#endif
	break;
    }
  return 1;
}

/******************************************************************
 *
 * Read next record
 *
 ******************************************************************/
sh_file_t *  sh_hash_getdataent (SL_TICKET fd, char * line)
{
  sh_file_t * p;
  sh_filestore_t ft;
  long i;
  char * fullpath;
  char * linkpath;

  /* Read next record -- Part One 
   */
  p = SH_ALLOC(sizeof(sh_file_t));

  i = sl_read(fd, &ft, sizeof(sh_filestore_t));
  if ( SL_ISERROR(i) || i == 0) 
    {
      SH_FREE(p);
      return NULL;
    }
      
  if (ft.mark != REC_MAGIC)
    {
      SH_FREE(p);
      return NULL;
    }

  /* Read next record -- Part Two -- Fullpath
   */
  i =  sh_unix_getline (fd, line, SH_BUFSIZE);
  if (i < 0 ) 
    {
      SH_FREE(line);
      SH_FREE(p);
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_P_NODATA,
		       file_path('D', 'R'));
      aud_exit (FIL__, __LINE__,EXIT_FAILURE);
    }

  i = sl_strlen(line)+1;
  fullpath = SH_ALLOC(i);
  sl_strlcpy (fullpath, line, i);
  if (fullpath[i-2] == '\n')
    fullpath[i-2] = '\0';

  /* Read next record -- Part Three -- Linkpath
   */
  i =  sh_unix_getline (fd, line, SH_BUFSIZE);
  if (i < 0 ) 
    {
      SH_FREE(line);
      SH_FREE(fullpath);
      SH_FREE(p);
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_P_NODATA,
		       file_path('D', 'R'));
      aud_exit (FIL__, __LINE__,EXIT_FAILURE);
    }

  i = sl_strlen(line)+1;
  linkpath = SH_ALLOC(i);
  sl_strlcpy (linkpath, line, i);
  if (linkpath[i-2] == '\n')
    linkpath[i-2] = '\0';

  /* Read next record -- Part Four -- Decode
   */
#if defined(SH_STEALTH)
  sh_do_decode(fullpath,    sl_strlen(fullpath));
  
#if defined(__linux__)
  sh_do_decode(ft.c_attributes,   sl_strlen(ft.c_attributes));
#endif
  
  sh_do_decode(ft.c_mode,   sl_strlen(ft.c_mode));
  sh_do_decode(ft.c_owner,  sl_strlen(ft.c_owner));
  sh_do_decode(ft.c_group,  sl_strlen(ft.c_group));
  sh_do_decode(ft.checksum, sl_strlen(ft.checksum));
  
  
  if (ft.c_mode[0] == 'l') 
    {  
      sh_do_decode(linkpath, sl_strlen(linkpath));
    }
#endif

  memcpy( &(*p).theFile, &ft, sizeof(sh_filestore_t) );
  p->visited   = S_FALSE;
  p->reported  = S_FALSE;
  p->allignore = S_FALSE;
  p->modi_mask = 0L;
  p->fullpath  = fullpath;
  p->linkpath  = linkpath;

  /* set to an invalid value 
   */
  ft.mark = (REC_MAGIC + 5);

  return p;
}


/******************************************************************
 *
 * Initialize
 *
 ******************************************************************/
void sh_hash_init ()
{
  sh_file_t * p;
  SL_TICKET fd    = (-1);
  long i;
  int count = 0;
  char * line = NULL;

#if defined(WITH_GPG) || defined(WITH_PGP)
  char * buf  = NULL;
  int    bufc;
  int    flag_pgp = S_FALSE;
  SL_TICKET fdTmp = (-1);
  SL_TICKET open_tmp (void);
#endif


  if (IsInit == 1) 
    return;

  for (i = 0; i < TABSIZE; ++i) 
    tab[i] = NULL;

#if defined(SH_WITH_CLIENT)

  /* Data file from Server
   */

  if (fd == (-1) && 0 == sl_strcmp(file_path('D', 'R'), _("REQ_FROM_SERVER")))
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_D_DSTART);
      fd = sh_forward_req_file(_("DATA"));
      if (SL_ISERROR(fd))
	aud_exit (FIL__, __LINE__, EXIT_FAILURE);

      sl_rewind (fd);

      tiger_fd = fd;
      sl_strlcpy (sh.data.hash, 
		  sh_tiger_hash (file_path('C', 'R'), 
				 TIGER_FD, 0),
		  KEY_LEN+1);
    }
  else 
#endif
    /* Local data file
     */

    if (fd == (-1))
      {
	if ( SL_ISERROR(fd = sl_open_read(file_path('D', 'R'), SL_YESPRIV))) 
	  {
	    TPT(( 0, FIL__, __LINE__, _("msg=<Error opening: %s>\n"), 
		  file_path('D', 'R')));
	    sh_error_handle ((-1), FIL__, __LINE__, fd, MSG_EXIT_ABORT1, 
			     sh.prg_name);
	    aud_exit (FIL__, __LINE__, EXIT_FAILURE);
	  }
	
	TPT(( 0, FIL__, __LINE__, _("msg=<Opened database: %s>\n"), 
	      file_path('D', 'R')));

	tiger_fd = fd;
	if (0 != sl_strncmp(sh.data.hash, 
			    sh_tiger_hash (file_path('D', 'R'), TIGER_FD, 0),
			    KEY_LEN)
	    && sh.flag.checkSum != SH_CHECK_INIT) 
	  {
	    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_AUTH,
			     file_path('D', 'R'));
	    aud_exit (FIL__, __LINE__, EXIT_FAILURE);
	  }
	sl_rewind (fd);

#if defined(WITH_GPG) || defined(WITH_PGP)
	/* extract the data and copy to temporary file
	 */
	fdTmp = open_tmp();
	buf = (char *) SH_ALLOC(4096);
	while ( (bufc = sh_unix_getline (fd, buf, 4095)) > 0) 
	  {
	    if (flag_pgp == S_FALSE &&
		(0 == sl_strcmp(buf, _("-----BEGIN PGP SIGNED MESSAGE-----"))||
		 0 == sl_strcmp(buf, _("-----BEGIN PGP MESSAGE-----")))
		 )
	      flag_pgp = S_TRUE;
	    
	    if (flag_pgp == S_TRUE && buf[0] == '\n')
	      {
		sl_write(fdTmp, buf, 1);
	      }
	    else if (flag_pgp == S_TRUE)
	      {
		sl_write_line(fdTmp, buf, bufc);
	      }

	    if (flag_pgp == S_TRUE && 
		0 == sl_strcmp(buf, _("-----END PGP SIGNATURE-----")))
	      break;
	  }
	SH_FREE(buf);
	sl_close(fd);
	fd = fdTmp;
	sl_rewind (fd);

	/* Validate signature of open file.
	 */
	if (0 != sh_gpg_check_sign (0, fd, 2))
	  aud_exit (FIL__, __LINE__, EXIT_FAILURE);
	sl_rewind (fd);
#endif
      }

  line = SH_ALLOC(SH_BUFSIZE+1);

  /* fast forward to start of data
   */
  sh_hash_setdataent(fd, line);

  while (1) 
    {
      p = sh_hash_getdataent (fd, line);
      if (p != NULL)
	{
	  hashinsert (p); 
	  ++count;
	}
      else
	break;
    }

  if (line != NULL)
    SH_FREE(line);

  /* Always keep db in memory, so we have no open file
   */
  sl_close (fd);
  fd = -1;

  /* --- Initialization done successfully. ---
   */
  IsInit = 1;
  return;
}
  
/*****************************************************************
 *
 * delete hash array
 *
 *****************************************************************/
void sh_hash_hashdelete ()
{
  int i;

  if (IsInit == 0) 
    return;
  for (i = 0; i < TABSIZE; ++i) 
    if (tab[i] != NULL)
      { 
	hash_kill (tab[i]);
	tab[i] = NULL;
      }
  IsInit = 0;
  return;
}

/******************************************************************
 *
 * Insert a file into the database.
 *
 ******************************************************************/ 
void sh_hash_pushdata (file_type * buf, char * fileHash)
{
  sh_file_t * pp;
  static long p_count = 0;
  long        pi;

  unsigned short eo_m;

  SL_TICKET fd;
  sh_filestore_t p;
  static int isfirst = 1;

  struct stat sbuf;

  char fullpath[MAX_PATH_STORE+1];
  char linkpath[MAX_PATH_STORE+1];

  char * line = NULL;


  linkpath[0] =  '-'; 
  linkpath[1] = '\0'; 
  fullpath[0] =  '-'; 
  fullpath[1] = '\0'; 

  if (NULL == file_path('D', 'W') || 
      0 == sl_strcmp(file_path('D', 'W'), _("REQ_FROM_SERVER")))
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORTS,
		      sh.prg_name, 
		      _("No local path for database initialisation"), 
		      _("sh_hash_pushdata"));
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }


  if (isfirst == 1)  
    {
      if (0 == retry_lstat(file_path('D', 'W'), &sbuf))
	sh_error_handle((-1), FIL__, __LINE__, 0, MSG_FI_DBEX,
			file_path('D', 'W'));
    }


  if (sh.flag.update == S_FALSE)
    {
      if ( SL_ISERROR(fd = sl_open_write(file_path('D', 'W'), SL_YESPRIV)))
	return;
      if ( SL_ISERROR(sl_forward(fd)))
	return;
    }
  else
    {
      if ( SL_ISERROR(fd = sl_open_rdwr(file_path('D', 'W'), SL_YESPRIV)))
	return;
      line = SH_ALLOC(SH_BUFSIZE+1);
      if (SL_ISERROR(sh_hash_setdataent (fd, line)))
	{
	  SH_FREE(line);
	  return;
	}
      for (pi = 0; pi < p_count; ++pi)
	{
	  pp = sh_hash_getdataent(fd, line);
	  if (pp)
	    {
	      if (pp->fullpath)
		SH_FREE(pp->fullpath);
	      if (pp->linkpath)
		SH_FREE(pp->linkpath);
	      SH_FREE(pp);
	    }
	}
      SH_FREE(line);
    }
	  


  if (sl_strlen(buf->fullpath) <= MAX_PATH_STORE) 
    {
      sl_strlcpy(fullpath, buf->fullpath, MAX_PATH_STORE+1);
      if (sl_strlen(buf->fullpath) < (MAX_PATH_STORE-3))
	{
	  fullpath[MAX_PATH_STORE-2] = '\0';
	  fullpath[MAX_PATH_STORE-1] = '\n';
	}
    } 
  else 
    {
      sl_strlcpy(fullpath, 
		 sh_tiger_hash (buf->fullpath,
				TIGER_DATA,sl_strlen(buf->fullpath)), 
		 KEY_LEN+1);
    }

#if defined(SH_STEALTH)
  sh_do_encode(fullpath, sl_strlen(fullpath));
#endif

  if (buf->c_mode[0] == 'l') 
    {  
      if (sl_strlen(buf->linkpath) <= MAX_PATH_STORE) 
	{
	  sl_strlcpy(linkpath, buf->linkpath, MAX_PATH_STORE+1);  
	} 
      else 
	{
	  sl_strlcpy(linkpath, 
		     sh_tiger_hash (buf->linkpath,
				    TIGER_DATA,sl_strlen(buf->linkpath)),
		     KEY_LEN+1);
	}

#if defined(SH_STEALTH)
      sh_do_encode(linkpath, sl_strlen(linkpath));
#endif
    }

  p.mark = REC_MAGIC;
  sl_strlcpy(p.c_mode,   buf->c_mode,   11);
  sl_strlcpy(p.c_group,  buf->c_group,  GROUP_MAX+1);
  sl_strlcpy(p.c_owner,  buf->c_owner,  USER_MAX+1);
  sl_strlcpy(p.checksum, fileHash,      KEY_LEN+1);
#if defined(__linux__)
  sl_strlcpy(p.c_attributes, buf->c_attributes, 13);
#endif

#if defined(SH_STEALTH)
  sh_do_encode(p.c_mode,   sl_strlen(p.c_mode));
  sh_do_encode(p.c_owner,  sl_strlen(p.c_owner));
  sh_do_encode(p.c_group,  sl_strlen(p.c_group));
  sh_do_encode(p.checksum, sl_strlen(p.checksum));
#if defined(__linux__)
  sh_do_encode(p.c_attributes,   sl_strlen(p.c_attributes));
#endif
#endif

#if defined(__linux__)
  p.attributes  = buf->attributes;
#endif
  p.linkmode    = buf->linkmode;
  p.hardlinks   = buf->hardlinks;
  p.mode  = buf->mode;
  p.ino   = buf->ino;
  p.size  = buf->size;
  p.mtime = buf->mtime;
  p.atime = buf->atime;
  p.ctime = buf->ctime;
  p.owner = buf->owner;
  p.group = buf->group;

  /* write the start marker 
   */
  if (isfirst == 1) 
    {
      if (sh.flag.update == S_FALSE)
	{
#if defined(SH_STEALTH)
	  sl_write      (fd,        "\n", 1);
	  sl_write_line (fd, N_("[SOF]"), 5);
#else
	  sl_write_line (fd, _("\n[SOF]"),  6);
#endif
	}
      isfirst = 0;
    }
      
  sl_write      (fd,       &p, sizeof(sh_filestore_t));
  sl_write_line (fd, fullpath,    sl_strlen(fullpath));
  sl_write_line (fd, linkpath,    sl_strlen(linkpath));

  if (sh.flag.update == S_TRUE)
    {
      eo_m = REC_MAGIC - 5;
      sl_write      (fd,    &eo_m, sizeof(unsigned short));
    }

  ++p_count;

  sl_close (fd);
  return;
}

/*********************************************************************
 *
 * Check whether a file is present in the database.
 *
 *********************************************************************/
int sh_hash_have_it (char * newname)
{
  sh_file_t * p;

  if (newname == NULL)
    return (-1);

  if (IsInit != 1) 
    sh_hash_init();
  if (sl_strlen(newname) <= MAX_PATH_STORE) 
    p = hashsearch(newname);
  else 
    p = hashsearch ( sh_tiger_hash(newname, TIGER_DATA, sl_strlen(newname)) );
  if (p == NULL) 
    return (-1);

  if (p->allignore == S_FALSE && 
      (p->modi_mask & MODI_CHK) != 0 &&
      (p->modi_mask & MODI_MOD) != 0)
    return (1);
  return (0);
}

/*****************************************************************
 *
 * Set a file's status to 'visited'. This is required for
 * files that should be ignored, and may be present in the
 * database, but not on disk.
 *
 *****************************************************************/
static int sh_hash_set_visited_int (char * newname, int flag)
{
  sh_file_t * p;

  if (newname == NULL)
    return (-1);
  if (IsInit != 1) 
    sh_hash_init();

  if (sl_strlen(newname) <= MAX_PATH_STORE) 
    p = hashsearch(newname);
  else 
    p = hashsearch (sh_tiger_hash(newname, TIGER_DATA, sl_strlen(newname)));
  
  if (p == NULL) 
    return (-1);
  p->visited  = S_TRUE;
  p->reported = flag;

  return 0;
}


int sh_hash_set_visited (char * newname)
{
  return(sh_hash_set_visited_int(newname, S_TRUE));
}

int sh_hash_set_visited_true (char * newname)
{
  return(sh_hash_set_visited_int(newname, S_FALSE));
}


/*****************************************************************
 *
 * Compare a file with the database status.
 *
 *****************************************************************/
void sh_hash_compdata (int class, file_type * theFile, char * fileHash)
{
  list_added * addCheck = addList;
  int addLen;
  char * msg;
  sh_file_t * p;
  char * tmp;
  char * tmp_path;
  char * tmp_lnk;
  char * tmp_lnk_old;
  char timstr1c[32];
  char timstr2c[32];
  char timstr1a[32];
  char timstr2a[32];
  char timstr1m[32];
  char timstr2m[32];
  char linkHash[KEY_LEN+1];
  int  maxcomp;

  unsigned long modi_mask = 0;

  if (IsInit != 1) sh_hash_init();

  /* --------  find the entry for the file ----------------       */

  if (sl_strlen(theFile->fullpath) <= MAX_PATH_STORE) 
    p = hashsearch(theFile->fullpath);
  else 
    p = hashsearch( sh_tiger_hash(theFile->fullpath, 
				  TIGER_DATA, 
				  sl_strlen(theFile->fullpath))
		    );


  /* --------- Not found in database. ------------
   */

  if (p == NULL) 
    {
      if (sh.flag.reportonce == S_TRUE)
	{
	  while (addCheck)
	    {
	      if (0 == sl_strcmp(theFile->fullpath, addCheck->name))
		{
		  addCheck->visited = S_TRUE;
		  return;
		}
	      addCheck = addCheck->next;
	    }
	  
	  addLen = sl_strlen(theFile->fullpath) + 1;
	  
	  addCheck = (list_added *) SH_ALLOC (sizeof(list_added));
	  addCheck->name = (char *) SH_ALLOC (addLen);

	  sl_strlcpy(addCheck->name, theFile->fullpath, addLen);
	  addCheck->class   = class;
	  addCheck->visited = S_TRUE;
	  addCheck->next    = addList; /* NULL if first */
	  addList           = addCheck;
	}

      tmp = sh_util_safe_name(theFile->fullpath);
      sh_error_handle (ShDFLevel[class], FIL__, __LINE__, 0, MSG_FI_ADD, 
		       tmp);
      SH_FREE(tmp);

      if (sh.flag.reportonce == S_TRUE)
	theFile->reported = S_TRUE;
	
      return;
    }
  else
    {
      p->modi_mask = theFile->check_mask;
    }



  TPT ((0, FIL__, __LINE__, _("file=<%s>, cs_old=<%s>, cs_new=<%s>\n"),
	theFile->fullpath, fileHash, p->theFile.checksum));

  if ( sl_strncmp (fileHash, p->theFile.checksum, 50) != 0 && 
       (theFile->check_mask & MODI_CHK) != 0)
    {
      modi_mask |= MODI_CHK;
      TPT ((0, FIL__, __LINE__, _("mod=<checksum>")));
    } 

  if (p->theFile.c_mode[0] == 'l') 
    {
      if (sl_strlen(theFile->linkpath) >= MAX_PATH_STORE) 
	{
	  sl_strlcpy(linkHash, 
		     sh_tiger_hash(theFile->linkpath, 
				   TIGER_DATA,
				   sl_strlen(theFile->linkpath)), 
		     MAX_PATH_STORE+1);
	  maxcomp = MAX_PATH_STORE;
	} 
      else 
	{
	  sl_strlcpy(linkHash, theFile->linkpath, KEY_LEN + 1);
	  maxcomp = KEY_LEN;
	}

    if ( sl_strncmp (linkHash, p->linkpath, maxcomp) != 0 &&
	 (theFile->check_mask & MODI_LNK) != 0)
      {
	modi_mask |= MODI_LNK;
	TPT ((0, FIL__, __LINE__, _("mod=<link>")));
      } 
    }
    
  if ( theFile->ino != p->theFile.ino  &&
       (theFile->check_mask & MODI_INO) != 0)
    {
      modi_mask |= MODI_INO;
      TPT ((0, FIL__, __LINE__, _("mod=<inode>")));
    } 
    
  if ( theFile->hardlinks != p->theFile.hardlinks &&
       (theFile->check_mask & MODI_HLN) != 0)
    {
      modi_mask |= MODI_HLN;
      TPT ((0, FIL__, __LINE__, _("mod=<hardlink>")));
    } 
    
  if ( (  (theFile->mode != p->theFile.mode)
#if defined(__linux__)
	  || (theFile->attributes != p->theFile.attributes)
#endif
	)
       && (theFile->check_mask & MODI_MOD) != 0)
    {
      modi_mask |= MODI_MOD;
      TPT ((0, FIL__, __LINE__, _("mod=<mode>")));
    } 

  if ( theFile->owner != p->theFile.owner &&
       (theFile->check_mask & MODI_USR) != 0)
    {
      modi_mask |= MODI_USR;
      TPT ((0, FIL__, __LINE__, _("mod=<user>")));
    } 

  if ( theFile->group != p->theFile.group &&
       (theFile->check_mask & MODI_GRP) != 0)
    {
      modi_mask |= MODI_GRP;
      TPT ((0, FIL__, __LINE__, _("mod=<group>")));
    } 

  
  if ( theFile->mtime != p->theFile.mtime &&
       (theFile->check_mask & MODI_MTM) != 0)
    {
      modi_mask |= MODI_MTM;
      TPT ((0, FIL__, __LINE__, _("mod=<mtime>")));
    } 
  
  if ( theFile->atime != p->theFile.atime &&
       (theFile->check_mask & MODI_ATM) != 0)
    {
      modi_mask |= MODI_ATM;
      TPT ((0, FIL__, __LINE__, _("mod=<atime>")));
    } 

  
  /* Resetting the access time will set a new ctime. Thus, either we ignore
   * the access time or the ctime for NOIGNORE
   */
  if ( theFile->ctime != p->theFile.ctime &&
       (theFile->check_mask & MODI_CTM) != 0)
    {
      modi_mask |= MODI_CTM;
      TPT ((0, FIL__, __LINE__, _("mod=<ctime>")));
    } 

  if ( theFile->size != p->theFile.size &&
       (theFile->check_mask & MODI_SIZ) != 0)
    {
      if (class == SH_LEVEL_LOGGROW && theFile->size < p->theFile.size)
	{
	  modi_mask |= MODI_SIZ;
	  TPT ((0, FIL__, __LINE__, _("mod=<size>")));
	} 
      else if (class != SH_LEVEL_LOGGROW)
	{ 
	  modi_mask |= MODI_SIZ;
	  TPT ((0, FIL__, __LINE__, _("mod=<size>")));
	} 
    }

  /* --- Report full details. ---
   */
  if (modi_mask != 0 && sh.flag.fulldetail == S_TRUE)
    {
      if ((theFile->check_mask & MODI_ATM) == 0)
	modi_mask = mask_READONLY;
      else
	modi_mask = mask_NOIGNORE;
    }

  /* --- Report on modified files. ---
   */
  if (modi_mask != 0 && p->reported == S_FALSE)
    { 
      tmp = SH_ALLOC(SH_BUFSIZE);
      msg = SH_ALLOC(SH_BUFSIZE);
      msg[0] = '\0';

      if ((modi_mask & MODI_MOD) != 0)
	{
#if defined(__linux__)
	  sl_snprintf(tmp, SH_BUFSIZE, 
		      _("mode_old=<%s>, mode_new=<%s>, attr_old=<%s>, attr_new=<%s>, "),
		      p->theFile.c_mode, theFile->c_mode,
		      p->theFile.c_attributes, theFile->c_attributes);
#else
	  sl_snprintf(tmp, SH_BUFSIZE, _("mode_old=<%s>, mode_new=<%s>, "),
		      p->theFile.c_mode, theFile->c_mode);
#endif
	  sl_strlcat(msg, tmp, SH_BUFSIZE);
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    {
	      sl_strlcpy(p->theFile.c_mode, theFile->c_mode, 11);
	      p->theFile.mode = theFile->mode;
#if defined(__linux__)
	      sl_strlcpy(p->theFile.c_attributes, theFile->c_attributes, 16);
	      p->theFile.attributes = theFile->attributes;
#endif
	    }
#endif
	}

      if ((modi_mask & MODI_HLN) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, 
		      _("hardlinks_old=<%ld>, hardlinks_new=<%ld>, "),
		      p->theFile.hardlinks, theFile->hardlinks);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.hardlinks = theFile->hardlinks;
#endif
	}

      if ((modi_mask & MODI_INO) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, 
		      _("inode_old=<%ld>, inode_new=<%ld>, "),
		      p->theFile.ino, theFile->ino);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.ino = theFile->ino;
#endif
	}

      if ((modi_mask & MODI_USR) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, _("owner_old=<%s>, owner_new=<%s>, "),
		      p->theFile.c_owner, theFile->c_owner);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    {
	      sl_strlcpy(p->theFile.c_owner, theFile->c_owner, USER_MAX+2);
	      p->theFile.owner = theFile->owner;
	    }
#endif
	}

      if ((modi_mask & MODI_GRP) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, _("group_old=<%s>, group_new=<%s>, "),
		      p->theFile.c_group, theFile->c_group);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    {
	      sl_strlcpy(p->theFile.c_group, theFile->c_group, GROUP_MAX+2);
	      p->theFile.group = theFile->group;
	    }
#endif
	}

      if ((modi_mask & MODI_SIZ) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, _("size_old=<%ld>, size_new=<%ld>, "),
		      p->theFile.size, theFile->size);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.size = theFile->size;
#endif
	}

      if ((modi_mask & MODI_CTM) != 0)
	{
	  sl_strlcpy (timstr1c, sh_unix_time (p->theFile.ctime), 32);
	  sl_strlcpy (timstr2c, sh_unix_time (theFile->ctime),   32);
	  sl_snprintf(tmp, SH_BUFSIZE, _("ctime_old=<%s>, ctime_new=<%s>, "),
		      timstr1c, timstr2c);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.ctime = theFile->ctime;
#endif
	}

      if ((modi_mask & MODI_ATM) != 0)
	{
	  sl_strlcpy (timstr1a, sh_unix_time (p->theFile.atime), 32);
	  sl_strlcpy (timstr2a, sh_unix_time (theFile->atime),   32);
	  sl_snprintf(tmp, SH_BUFSIZE, _("atime_old=<%s>, atime_new=<%s>, "),
		      timstr1a, timstr2a);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.atime = theFile->atime;
#endif
	}

      if ((modi_mask & MODI_MTM) != 0)
	{
	  sl_strlcpy (timstr1m, sh_unix_time (p->theFile.mtime), 32);
	  sl_strlcpy (timstr2m, sh_unix_time (theFile->mtime),   32);
	  sl_snprintf(tmp, SH_BUFSIZE, _("mtime_old=<%s>, mtime_new=<%s>, "),
		      timstr1m, timstr2m);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    p->theFile.mtime = theFile->mtime;
#endif
	}


      if ((modi_mask & MODI_CHK) != 0)
	{
	  sl_snprintf(tmp, SH_BUFSIZE, 
		      _("chksum_old=<%s>, chksum_new=<%s>, "),
		      p->theFile.checksum, fileHash);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    sl_strlcpy(p->theFile.checksum, fileHash, KEY_LEN+1);
#endif
	}


      if ((modi_mask & MODI_LNK) != 0 && theFile->c_mode[0] == 'l')
	{
	  tmp_lnk     = sh_util_safe_name(theFile->linkpath);
	  tmp_lnk_old = sh_util_safe_name(p->linkpath);
	  sl_snprintf(tmp, SH_BUFSIZE, _("link_old=<%s>, link_new=<%s>"),
		      tmp_lnk_old, tmp_lnk);
	  SH_FREE(tmp_lnk);
	  SH_FREE(tmp_lnk_old);
	  sl_strlcat(msg, tmp, SH_BUFSIZE); 
#ifdef REPLACE_OLD
	  if (sh.flag.reportonce == S_TRUE)
	    {
	      if (p->linkpath != NULL)
		SH_FREE(p->linkpath);
	      p->linkpath = (char*)SH_ALLOC (sl_strlen(theFile->linkpath) + 1);
	      sl_strlcpy(p->linkpath, theFile->linkpath, 
			 sl_strlen(theFile->linkpath) + 1);
	    }
#endif
	}


      tmp_path = sh_util_safe_name(theFile->fullpath);
      sh_error_handle (ShDFLevel[class], FIL__, __LINE__, 0, MSG_FI_CHAN,
		       _(policy[class]), tmp_path, msg);

      SH_FREE(tmp_path);
      SH_FREE(tmp);
      SH_FREE(msg);

#ifndef REPLACE_OLD
      p->reported = S_TRUE;
#endif

    }

  p->visited = S_TRUE;

  return;
}

int hash_full_tree () 
{
  sh_file_t * p;
  int         i;

  if (IsInit != 1) 
    return(0);

  for (i = 0; i < TABSIZE; ++i)
    {
      for (p = tab[i]; p; p = p->next)
	p->allignore  = S_FALSE;
    }
  return (0);
} 


int hash_remove_tree (char * s) 
{
  sh_file_t * p;
  int         len;
  int         i;

  if (!s)
    return (-1);
  else
    len = sl_strlen(s);

  if (IsInit != 1) 
    sh_hash_init();

  for (i = 0; i < TABSIZE; ++i)
    {
      for (p = tab[i]; p; p = p->next)
	{
	  if (sl_strncmp(s, p->fullpath, len) == 0)
	    { 
	      /* fprintf(stderr, "FIXME: remove %s\n", p->fullpath); */
	      p->allignore  = S_TRUE;
	    }
	}
    }
  return (0);
} 



/* if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE) */

#endif
