/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <errno.h>


#include "samhain.h"
#include "sh_error.h"
#include "sh_getopt.h"
#include "sh_files.h"
#include "sh_utils.h"
#include "sh_mail.h"
#include "sh_forward.h"

#if defined(WITH_EXTERNAL)
#include "sh_extern.h"
#endif



#define HAS_ARG_NO  0
#define HAS_ARG_YES 1
#define DROP_PRIV_NO  0
#define DROP_PRIV_YES 1


typedef struct options {
  char * longopt;
  const char   shortopt;
  char * usage;
  int          hasArg;
  int (*func)(char * opt);
} opttable_t;

static int sh_getopt_usage (char * dummy);
static int sh_getopt_copyright (char * dummy);


opttable_t table[] = {

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
  { N_("set-checksum-test"),  
    't', 
    N_("Set checksum testing to 'init', 'update', or 'check'"),  
    HAS_ARG_YES, 
    sh_util_setchecksum },
#endif
#ifdef SH_WITH_SERVER
  { N_("server"),  
    'S', 
    N_("Run as log server"),  
    HAS_ARG_NO, 
    sh_util_setserver },
  { N_("qualified"),  
    'q', 
    N_("Log fully qualified name of client host"),  
    HAS_ARG_NO, 
    sh_forward_set_strip },
#endif
  { N_("deamon"),  
    'D', 
    N_("Run as daemon"),  
    HAS_ARG_NO, 
    sh_unix_setdeamon },
#ifdef SH_WITH_CLIENT
  { N_("set-export-severity"),  
    'e', 
    N_("Set severity threshold for export to remote log server"),  
    HAS_ARG_YES, 
    sh_error_setexport },
#endif
  { N_("set-syslog-severity"),  
    's', 
    N_("Set severity threshold for syslog"),  
    HAS_ARG_YES, 
    sh_error_set_syslog },
#ifdef WITH_EXTERNAL
  { N_("set-extern-severity"),  
    'x', 
    N_("Set severity threshold for logging by external program(s)"),  
    HAS_ARG_YES, 
    sh_error_set_external },
#endif
  { N_("set-log-severity"),  
    'l', 
    N_("Set severity threshold for logfile"),  
    HAS_ARG_YES, 
    sh_error_setlog },
  { N_("set-mail-severity"),  
    'm', 
    N_("Set severitythreshold  for e-mail"),  
    HAS_ARG_YES, 
    sh_error_setseverity },
  { N_("set-print-severity"),  
    'p', 
    N_("Set the severity threshold for terminal/console log"),  
    HAS_ARG_YES, 
    sh_error_setprint },
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  { N_("recursion"),  
    'r', 
    N_("Set recursion level for directories"),  
    HAS_ARG_YES, 
    sh_files_setrecursion },
#endif
  { N_("verify-log"),  
    'L', 
    N_("Verify the audit trail"),  
    HAS_ARG_YES, 
    sh_error_logverify },
#if defined(SH_WITH_MAIL)
  { N_("verify-mail"),  
    'M', 
    N_("Verify the mailbox"),  
    HAS_ARG_YES, 
    sh_mail_sigverify 
  },
#endif
  { N_("hash-string"),  
    'H', 
    N_("Print the hash of a string"),  
    HAS_ARG_YES, 
    sh_error_verify },
#if defined (SH_WITH_SERVER) 
  { N_("password"),  
    'P', 
    N_("Compute a client registry entry for password"),  
    HAS_ARG_YES, 
    sh_forward_make_client },
  { N_("gen-password"),  
    'G', 
    N_("Generate a random password"),  
    HAS_ARG_NO, 
    sh_forward_create_password },
#endif
#if 0
  /* ----------------------------------------------------------- */
  /* don't allow this from CL -- may leak privileged information */
  /* #if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)     */
  /* ----------------------------------------------------------- */
  { N_("check-dir-readonly"),  
    'F', 
    N_("Check a directory, ignore access time"),  
    HAS_ARG_YES, 
    sh_files_pushdir_ro },
  { N_("check-file-readonly"),  
    'f', 
    N_("Check a file, ignore access time"),  
    HAS_ARG_YES, 
    sh_files_pushfile_ro },
  { N_("check-file-noignore"),  
    'n', 
    N_("Check a file, report all modifications"),  
    HAS_ARG_YES,
    sh_files_pushfile_noig },
  { N_("check-file-loggrow"),  
    'g', 
    N_("Check a file, ignore timestamp, growing size, signature"),  
    HAS_ARG_YES, 
    sh_files_pushfile_glog },
#endif
  { N_("copyright"),  
    'c', 
    N_("Print copyright information"),  
    HAS_ARG_NO, 
    sh_getopt_copyright },
  { N_("help"),  
    'h', 
    N_("Print usage information"),  
    HAS_ARG_NO, 
    sh_getopt_usage },
#if defined(WITH_TPT) && !defined(WITH_TRACE)
  { N_("tracelevel"),  
    'z', 
    N_("Set > 0 for debug"),  
    HAS_ARG_YES, 
    set_ebene },
#elif defined(WITH_TRACE)
  { N_("milestone"),  
    'i', 
    N_("Set milestone for trace/debug (all: -1)"),  
    HAS_ARG_YES, 
    set_stein },
  { N_("tracelevel"),  
    'z', 
    N_("Set level for trace/debug"),  
    HAS_ARG_YES, 
    set_ebene },
#endif
  /* last entry -- required !! -- */
  { NULL, 
    '\0',     
    NULL,  
    HAS_ARG_NO, 
    NULL }
};

static int sh_getopt_copyright (char * dummy)
{
  fprintf (stdout, 
	   _("Copyright (C) 1999, 2000 Rainer Wichmann"\
	     " (http://la-samhna.de).\n\n"));

  fprintf (stdout, 
	   _("This program is free software; "\
	     "you can redistribute it and/or modify\n"));
  fprintf (stdout, _("it under the terms of "\
		     "the GNU General Public License as published by\n"));
  fprintf (stdout, _("the Free Software Foundation; "\
		     "either version 2 of the License, or\n"));
  fprintf (stdout, _("(at your option) any later version.\n\n"));

  fprintf (stdout, _("This program is distributed in the hope "\
		     "that it will be useful,\n"));
  fprintf (stdout, _("but WITHOUT ANY WARRANTY; "\
		     "without even the implied warranty of\n"));
  fprintf (stdout, _("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."\
		     " See the\n"));
  fprintf (stdout, _("GNU General Public License for more details.\n\n"));

  fprintf (stdout, _("You should have received a copy of the "\
		     "GNU General Public License\n"));
  fprintf (stdout, _("along with this program; "\
		     "if not, write to the Free Software\n"));
  fprintf (stdout, _("Foundation, Inc., 59 Temple Place - Suite 330, "\
		     "Boston, MA  02111-1307, USA.\n\n"));

  fprintf (stdout, _("This product makes use of the reference implementation "\
		     "of the TIGER message\n"));
  fprintf (stdout, _("digest algorithm. This code is copyright Eli Biham "\
		     "(biham@cs.technion.ac.il)\n"));
  fprintf (stdout, _("and Ross Anderson (rja14@cl.cam.ac.uk). It can be used "\
		     "freely without any\n"));
  fprintf (stdout, _("restrictions.\n"));
#ifdef USE_SRP_PROTOCOL
  fprintf (stdout, _("This product makes use of the 'bignum' library by "\
		     "Henrik Johansson\n"));
  fprintf (stdout, _("(Henrik.Johansson@Nexus.Comm.SE). If you are including "\
		     "this library\n"));
  fprintf (stdout, _("in a commercial product, be sure to distribute all of"\
		     " it with\n"));
  fprintf (stdout, _("the product.\n"));
  fprintf (stdout, _("This product uses the 'Secure Remote Password' "\
		     "cryptographic\n"));
  fprintf (stdout, _("authentication system developed by Tom Wu "\
		     "(tjw@CS.Stanford.EDU).\n"));
#endif
  fprintf (stdout, _("\nThis product may use the LZO real-time data compression library, which\n"));
  fprintf (stdout, _("is released under the terms of the GNU General Public License.\n"));
  fprintf (stdout, _("Copyright (C) 1996, 1997, 1998, 1999 Markus Franz Xaver Johannes Oberhumer\n"));
  fprintf (stdout, _("<markus.oberhumer@jk.uni-linz.ac.at>\n"));
  fprintf (stdout, _("http://wildsau.idv.uni-linz.ac.at/mfx/lzo.html\n"));

  if (dummy)
    _exit (EXIT_SUCCESS);
  else
    _exit (EXIT_SUCCESS);
  return 0; /* make compilers happy */
}
  
static int sh_getopt_usage (char * dummy)
{
  int  i;
  char fmt[64];

  char opts[64] = { '\0' };


  fprintf (stdout,
	   _("This is samhain (%s), "\
	     "(c) 1999, 2000 Rainer Wichmann (http://la-samhna.de).\n"),
	   VERSION);
  fprintf (stdout, _("This software comes with ABSOLUTELY NO WARRANTY. "));
  fprintf (stdout, _("Use at own risk.\n"));

  fprintf (stdout, _("Usage:\n\n"));

  for (i = 0; table[i].shortopt != '\0'; ++i) {

    if (strchr(opts, table[i].shortopt) != NULL)
      fprintf (stdout, _("Short option char collision !\n"));
    opts[i] = table[i].shortopt;


    if (table[i].hasArg == HAS_ARG_NO) {
      if (sl_strlen(table[i].longopt) < 10) 
	strcpy(fmt, _("-%c,        --%-s,\t\t\t %s\n")); /* known to fit  */
      else if (sl_strlen(table[i].longopt) < 17)
	strcpy(fmt, _("-%c,        --%-s,\t\t %s\n"));   /* known to fit  */
      else 
	strcpy(fmt, _("-%c,        --%-s,\t %s\n"));     /* known to fit  */
      fprintf (stdout,
	       fmt,
	       table[i].shortopt,
	       _(table[i].longopt),
	       _(table[i].usage));
    } else {
      if (sl_strlen(table[i].longopt) < 12) 
	strcpy(fmt,                                      /* known to fit  */
	       _("-%c <arg>,  --%-s=<arg>,\t\t %s\n"));  
      else 
	strcpy(fmt,                                      /* known to fit  */ 
	       _("-%c <arg>,  --%-s=<arg>,\t %s\n"));   
      fprintf (stdout,
	       fmt,
	       table[i].shortopt,
	       _(table[i].longopt),
	       _(table[i].usage));
    }
  }

  fprintf (stdout, 
	   _("\nPlease report bugs to bugs@la-samhna.de.\n"));

  fflush(stdout);

  if ( dummy != NULL) 
    {
      if (sl_strcmp( dummy, _("fail")) == 0 ) 
	  _exit (EXIT_FAILURE);
    }

  _exit (EXIT_SUCCESS);
  return 0; /* make compilers happy */
}

  

int sh_getopt_get (int argc, char * argv[])
{
  int           count   = 0;
  unsigned long len     = 0;
  int           foundit = 0;
  int           i;
  char        * theequal;

  /* -- Return if no args. --
   */
  if (argc < 2) return 0;
 
  while (argc > 1  && argv[1][0] == '-') 
    {

      /* Initialize
       */
      foundit = 0;
      len     = sl_strlen (argv[1]);
    
      /* a '-' with no argument: error
       */
      if (len == 1)
	sh_getopt_usage(_("fail"));

      /* a '--' with no argument: stop argument processing
       */
      if (len == 2 && argv[1][1] == '-') 
	return count;

      /* a short option: process it
       */
      if (len == 2) 
	{

	  for (i = 0; table[i].shortopt != '\0'; ++i) 
	    {
	    
	      if ( table[i].shortopt == argv[1][1] ) 
		{
		  foundit = 1;
		  if ( table[i].hasArg == HAS_ARG_YES ) 
		    {
		      if (argc < 3) 
			{ 
			  /* argument required, but no avail 
			   */
			  fprintf (stderr, _("Error: missing argument\n"));
			  sh_getopt_usage(_("fail"));
			} 
		      else 
			{
			  /* call function with argument */
			  --argc; ++argv;
			  (* table[i].func )(argv[1]);
			  break;
			}
		    } 
		  else 
		    {
		      (* table[i].func )(NULL);
		      break;
		    }
		}
	    }

	  /* 'break' should get here 
	   */
	  if (foundit == 1) 
	    {
	      --argc; ++argv;
	      continue;
	    } 
	  else 
	    {
	      /* unrecognized short option */
	      fprintf (stderr, _("Error: unrecognized short option\n"));
	      sh_getopt_usage(_("fail"));
	    }
	}

      /* a long option: process it
       */
      if (len > 2) 
	{

	  for (i = 0; table[i].longopt != NULL; ++i) 
	    {
      
	      if (sl_strncmp(_(table[i].longopt), 
			     &argv[1][2], 
			     sl_strlen(table[i].longopt)) == 0 ) 
		{
		  foundit = 1; 
		  if ( table[i].hasArg == HAS_ARG_YES ) 
		    {
		      if ( (theequal = strchr(argv[1], '=')) == NULL) 
			{ 
			  fprintf (stderr, _("Error: missing argument\n"));
			  /* argument required, but no avail */
			  sh_getopt_usage(_("fail"));
			} 
		      else 
			{
			  if (sl_strlen (theequal) > 1) 
			    {
			      ++theequal;
			      /* call function with argument */
			      (* table[i].func )(theequal);
			      break;
			    } 
			  else 
			    {
			      fprintf (stderr, _("Error: invalid argument\n"));
			      /* argument required, but no avail */
			      sh_getopt_usage(_("fail"));
			    }
			}
		    } 
		  else 
		    {
		      (* table[i].func )(NULL);
		      break;
		    }
		}
	    }

	  /* 'break' should get here */
	  if (foundit == 1) 
	    {
	      ++count;
	      --argc; 
	      ++argv;
	      continue;
	    } 
	  else 
	    {
	      /* unrecognized long option */
	      fprintf (stderr, _("Error: unrecognized long option\n"));
	      sh_getopt_usage(_("fail"));
	    }
	}
    }

  return count;
}
