/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2000 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


#include "config_xor.h"


#include <stdio.h>
#include <string.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

/* replace #if 0 by #if 1 and set an appropriate path in front of '/pdbg.'
 * for debugging
 */
#if 0
#define PDGBFILE "/pdbg."
#endif

#if defined(WITH_EXTERNAL)

#if defined(PDGBFILE)
FILE * pdbg;
FILE * pdbgc;
#define PDBG_OPEN    pdbg = fopen(PDGBFILE"main",  "a")  
#define PDBG_CLOSE   fclose (pdbg)
#define PDBG(arg)    fprintf(pdbg,  "PDBG: step %d\n", arg); fflush(pdbg)
#define PDBG_D(arg)  fprintf(pdbg,  "PDBG: %d\n", arg); fflush(pdbg)
#define PDBG_S(arg)  fprintf(pdbg,  "PDBG: %s\n", arg); fflush(pdbg)

#define PDBGC_OPEN   pdbgc = fopen(PDGBFILE"child", "a")  
#define PDBGC_CLOSE  fclose (pdbgc)
#define PDBGC(arg)   fprintf(pdbgc, "PDBG: step %d\n", arg); fflush(pdbgc)
#define PDBGC_D(arg) fprintf(pdbgc, "PDBG: %d\n", arg); fflush(pdbgc)
#define PDBGC_S(arg) fprintf(pdbgc, "PDBG: %s\n", arg); fflush(pdbgc)
#else
#define PDBG_OPEN    
#define PDBG_CLOSE   
#define PDBG(arg)    
#define PDBG_D(arg)  
#define PDBG_S(arg)  
#define PDBGC_OPEN    
#define PDBGC_CLOSE   
#define PDBGC(arg)    
#define PDBGC_D(arg)  
#define PDBGC_S(arg)  
#endif


#include <stdlib.h>
#include <pwd.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/wait.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/types.h>

#include "samhain.h"
#include "sh_utils.h"
#include "sh_unix.h"
#include "sh_tiger.h"
#include "sh_extern.h"
#include "sh_calls.h"

#ifndef FD_SET
#define NFDBITS         32
#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#endif /* !FD_SET */
#ifndef FD_SETSIZE
#define FD_SETSIZE      32
#endif
#ifndef FD_ZERO
#define FD_ZERO(p)      memset((char *)(p), '\0', sizeof(*(p)))
#endif


#ifndef SH_UID
#define SH_UID long
#endif

#undef  FIL__
#define FIL__  _("sh_extern.c")


typedef struct _sh_com_t
{
  char     type[4];

  char   * command;

  int      for_c;
  char   * for_v[32];
  int      fand_c;
  char   * fand_v[32];
  int      fnot_c;
  char   * fnot_v[32];

  int      argc;
  char   * argv[32];
  int      envc;
  char   * envv[32];
  char     checksum[KEY_LEN + 1];
  time_t   deadtime;
  time_t   last_run;

  SH_UID   trusted_users[32];
  SH_UID   run_user_uid;
  SH_UID   run_user_gid;
  int      privileged;

  int      pipeFD;
  pid_t    pid;
  int      isset;
  FILE   * pipe;
  char     rw;

  struct _sh_com_t * next;

} sh_com_t;

static
void set3 (char * pos, char c1, char c2, char c3)
{
  pos[0] = c1;
  pos[1] = c2;
  pos[2] = c3;
  pos[3] = '\0';
  return;
}



/* initialize the external command structure
 */
static
sh_com_t * command_init(void)
{
  int         i;
  uid_t       ff_euid;
  sh_com_t  * ext_com = NULL;

  ext_com = (sh_com_t *) SH_ALLOC(sizeof(sh_com_t));

  if (!ext_com)
    return NULL;

  set3(ext_com->type, 'l', 'o', 'g');
  ext_com->command      = NULL;
  ext_com->for_c        = 0;
  ext_com->fand_c       = 0;
  ext_com->fnot_c       = 0;
  ext_com->argc         = 0;
  ext_com->envc         = 0;
  ext_com->checksum[0]  = '\0';
  ext_com->pipeFD       = (-1);
  ext_com->pid          = (pid_t) -1;
  ext_com->isset        = 0;
  ext_com->privileged   = 1;
  ext_com->pipe         = NULL;
  ext_com->rw           = 'w';
  ext_com->deadtime     = 0;
  ext_com->last_run     = 0;

  for (i = 0; i < 32; ++i)
    {
      ext_com->for_v[i]         = NULL;
      ext_com->fand_v[i]        = NULL;
      ext_com->fnot_v[i]        = NULL;
      ext_com->argv[i]          = NULL;
      ext_com->envv[i]          = NULL;
      ext_com->trusted_users[i] = -1;
    }

  ext_com->run_user_uid     = (SH_UID) getuid();
  ext_com->run_user_gid     = (SH_UID) getgid();

  sl_get_euid(&ff_euid);
  ext_com->trusted_users[0] = 0;
  ext_com->trusted_users[1] = (SH_UID) (ff_euid);


  ext_com->next             = NULL;

  return ext_com;
}

/* the list of external commands
 */
sh_com_t * ext_coms   = NULL;

/* if -1, allocation of last command has failed,
 * thus don't fill in options
 */
static int ext_failed = -1;

static
int sh_ext_add_envv(char * key, char * val)
{
  int sk = 0, sv = 0, si;

  if (ext_coms == NULL || ext_failed == (-1) || 
      (key == NULL      && val == NULL)      || 
      ext_coms->envc >= 30)
    return (-1);

  if (key != NULL)
    sk = strlen(key) + 1;
  if (val != NULL)
    sv = strlen(val) + 1;

  si = ext_coms->envc;

  ext_coms->envv[si] = SH_ALLOC(sk + sv);

  if (ext_coms->envv[si] == NULL)
    return (-1);

  if (key != NULL)
    {
      sl_strlcpy(ext_coms->envv[si], key, sk+sv);
      sl_strlcat(ext_coms->envv[si], "=", sk+sv);
      if (val != NULL)
	sl_strlcat(ext_coms->envv[si], val, sk+sv);
    }
  else
    sl_strlcpy(ext_coms->envv[si], val, sv);

  ++ext_coms->envc;

  return (0);
}



static 
int sh_ext_init(char * command)
{
  sh_com_t * retval;
  int        size;

  if (command == NULL)
    return (-1);
  size = strlen(command);
  if (command[0] != '/' || size < 2)
    return (-1);

  if (NULL == (retval = command_init()))
    return (-1);
  retval->command = SH_ALLOC(size + 1);
  if (retval->command == NULL)
    return (-1);
  sl_strlcpy(retval->command, command, size+1);

  if (sh.timezone != NULL)
    {
      sh_ext_add_envv( "TZ", sh.timezone);
    }

  retval->next = ext_coms;
  ext_coms     = retval;
  return 0;
}


static
int sh_ext_uid (char * user, uid_t * uid, gid_t * gid)
{
  struct passwd * tempres;

  if (user == NULL)
    return (-1);

  tempres = getpwnam(user);
  if (NULL != tempres) 
    {
      *uid = tempres->pw_uid;  
      *gid = tempres->pw_gid;
      return 0;
    } 
  else 
    return (-1);
}


static
int sh_ext_add (char * argstring, int * ntok, char * stok[])
{
  int    i = 0;
  int    s;
  char * p;

  if (NULL == argstring)
    return (-1);

  do
    {
      if (i == 0)
	p = strtok (argstring, ", ");
      else
	p = strtok (NULL, ", ");
      if (p == NULL)
	break;

      s = strlen(p) + 1;
      if (stok[i] != NULL)
	SH_FREE(stok[i]);
      stok[i] = SH_ALLOC(s);
      sl_strlcpy(stok[i], p, s);

      ++i;
      if (i == 30)
	break;
    }
  while (p != NULL);

  *ntok = i;

  return (0);
}

/*********************************************************
 *
 * Public functions
 *
 *
 *********************************************************/
 
/* 
 * -- start a new external command, and add it to the list
 */ 
int sh_ext_setcommand(char * cmd)
{
  int i;
  if ( (i = sh_ext_init(cmd)) < 0)
    ext_failed = -1;
  else
    ext_failed = 0;
  return i;
}


/* 
 * -- clean up the command list
 */
int sh_ext_cleanup(void)
{
  int i;
  sh_com_t * retval;

  while (ext_coms != NULL)
    {
      retval   = ext_coms;
      ext_coms = retval->next;

      if (NULL != retval->command)    SH_FREE(retval->command);

      for (i = 0; i < 32; ++i)
	{
	  if (NULL != retval->argv[i])   SH_FREE(retval->argv[i]);
	  if (NULL != retval->envv[i])   SH_FREE(retval->envv[i]);
	  if (NULL != retval->for_v[i])  SH_FREE(retval->for_v[i]);
	  if (NULL != retval->fand_v[i]) SH_FREE(retval->fand_v[i]);
	  if (NULL != retval->fnot_v[i]) SH_FREE(retval->fnot_v[i]);
	}

      SH_FREE(retval);

    }

  return (0);
}

/*
 * -- add keywords to the OR filter
 */
int sh_ext_add_or (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->for_c), ext_coms->for_v));
}

/*
 * -- add keywords to the AND filter
 */
int sh_ext_add_and (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->fand_c), ext_coms->fand_v));
}

/*
 * -- add keywords to the NOT filter
 */
int sh_ext_add_not (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->fnot_c), ext_coms->fnot_v));
}

/*
 * -- add keywords to the CL argument list
 */
int sh_ext_add_argv (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->argc), ext_coms->argv));
}

/*
 * -- add a path to the environment
 */
int sh_ext_add_default (char * dummy)
{
  /* while this assignment looks ridiculous, it is here to avoid
   * an 'unused parameter' warning
   */
  char * p = (dummy == NULL ? dummy : NULL);

  p = sh_unix_getUIDdir (SH_ERR_ERR, (uid_t) ext_coms->run_user_uid);
  if (p)
    sh_ext_add_envv (_("HOME"), p);
   sh_ext_add_envv (_("SHELL"), _("/bin/sh")); 
   sh_ext_add_envv (_("PATH"),  _("/sbin:/usr/sbin:/bin:/usr/bin")); 
   return (p == NULL ? (-1) :  0);
}

/*
 * -- add an environment variable
 */
int sh_ext_add_environ (char * str)
{
  return (sh_ext_add_envv (NULL, str));
}

/*
 * -- set deadtime
 */
int sh_ext_deadtime (char * str)
{
  long    deadtime = 0;
  char  * tail     = NULL;

  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    return (-1);

  deadtime = strtol(str, &tail, 10);
  if (tail == str || deadtime < 0 || deadtime == LONG_MAX)
    return (-1);
  
  ext_coms->deadtime = (time_t) deadtime;  
  return (0);
}

/*
 * -- define type
 */
int sh_ext_type (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    return (-1);

  if (strlen(str) != 3)
    return (-1);

  set3(ext_coms->type, str[0], str[1], str[2]);

  if      (str[0] == 'l' && str[1] == 'o' && str[2] == 'g')
    ext_coms->rw = 'w';
  else if (str[0] == 's' && str[1] == 'r' && str[2] == 'v')
    ext_coms->rw = 'w';
  else if (str[0] == 'm' && str[1] == 'o' && str[2] == 'n')
    ext_coms->rw = 'r';
  else
    return (-1);

  return (0);
} 
  


/*
 * -- define checksum
 */
int sh_ext_checksum (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    return (-1);

  if (sl_strlen(str) != KEY_LEN)
    return (-1);

  sl_strlcpy (ext_coms->checksum, str, KEY_LEN+1);

  return (0);
}

/*
 * -- choose privileges
 */
int sh_ext_priv (char * c)
{

  uid_t m_uid;
  gid_t m_gid;

  if (0 == sh_ext_uid (c, &m_uid, &m_gid))
    {
      ext_coms->run_user_uid = (SH_UID) m_uid;
      ext_coms->run_user_gid = (SH_UID) m_gid;
      if (m_uid != (uid_t) 0)
	ext_coms->privileged   = 0;
      return (0);
    }

  return (-1);
}


/*
 * -- generic safe popen
 */
int get_the_fd (SL_TICKET ticket);

int sh_ext_popen (sh_com_t * task)
{
  long status = 0;
  int    flags;
  char * tmp;
  char * tmp2;
  int    errnum;
  int    pipedes[2];
  FILE * outf = NULL;
  char * envp[1];
  char * argp[1];

#if defined (__linux__)
  SL_TICKET   fd;
  char        pname[128];
  int         pfd;
#endif
  
  /* Linux, HP-UX and FreeBSD will happily accept envp = argp = NULL
   * Solaris (and probably some other Unices) 
   *         needs a valid *envp[] with envp[0] = NULL;
   *         and similarly for argp
   */
  envp[0] = NULL;
  argp[0] = NULL;

  /* 
   * --  check whether path is trustworthy
   */
  status = sl_trustfile(task->command, task->trusted_users, NULL);

  PDBG_D( (int) status);

  if ( SL_ENONE != status)
    { 
      PDBG_S("SL_ENONE != status");
      tmp  = sh_util_safe_name (task->command);
      if (sl_trust_errfile != '\0')
	{
	  tmp2  = sh_util_safe_name (sl_trust_errfile());
	  sh_error_handle((-1), FIL__, __LINE__, status, MSG_E_TRUST2,
			  sl_error_string(status), tmp, tmp2);
	  SH_FREE(tmp2);  
	}
      else
	{
	  sh_error_handle((-1), FIL__, __LINE__, status, MSG_E_TRUST1,
			  sl_error_string(status), tmp);
	}
      SH_FREE(tmp);
      return (-1);
    }

  PDBG(1);

  /* 
   * --  check whether the checksum is correct
   */
  if (task->checksum[0]  != '\0')
    {
      PDBG_S("checksum test");
      if (0 != sl_strcmp(task->checksum, 
			 sh_tiger_hash (task->command, TIGER_FILE, 0))
	  )
	{
	  PDBG_S("checksum mismatch");
	  tmp  = sh_util_safe_name (task->command);
	  sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_HASH, tmp);
	  SH_FREE(tmp);
	  return (-1);
 	}
    }

  PDBG(2);

  /* 
   * -- Create the pipe 
   */
  if (aud_pipe(FIL__, __LINE__, pipedes) < 0) 
    {
      PDBG_S("pipe() failure");
      errnum = errno;
      sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_E_SUBGEN, 
		      sh_error_message(errnum), _("pipe"));
      return (-1);
    }

  PDBG(3);
  
  /* 
   * -- Fork 
   */
  task->pid = aud_fork(FIL__, __LINE__);
  if (task->pid == (pid_t) - 1) 
    {
      PDBG_S("fork() failure");
      close(pipedes[0]);
      close(pipedes[1]);
      errnum = errno;
      sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_E_SUBGEN, 
		      sh_error_message(errnum), _("fork"));
      return (-1);
    }
  
  PDBG(4);

  if (task->pid == (pid_t) 0) 
    {
      /* 
       * -- fork again
       */
      if (0 == aud_fork(FIL__, __LINE__))
	{
	  PDBGC_OPEN;
	  PDBGC(1);

	  /*
	   * -- grandchild - make write side of the pipe stdin 
	   */
	  if (task->rw == 'w')
	    {
	      if (retry_aud_dup2(FIL__, __LINE__, 
				 pipedes[STDIN_FILENO], STDIN_FILENO) < 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  else
	    {
	      if (retry_aud_dup2(FIL__, __LINE__,
				 pipedes[STDOUT_FILENO], STDOUT_FILENO) < 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  PDBGC(2);
	    
	  
	  /* close the pipe descriptors 
	   */
	  close   (pipedes[STDIN_FILENO]);
	  close   (pipedes[STDOUT_FILENO]);
	  
	  /* don't leak file descriptors
	   */
#if !defined(PDGBFILE)
	  sh_unix_closeall (3);
#endif

	  /* drop root privileges, if possible && requested
	   */
	  sl_set_suid();
	  if (task->privileged == 0 && 0 == getuid())
	    {
	      PDBGC_S("privileged");

	      /* zero priv info
	       */
	      memset(skey, 0, sizeof(sh_key_t));

	      aud_setgid(FIL__, __LINE__,(gid_t) task->run_user_gid);
	      aud_setuid(FIL__, __LINE__,(uid_t) task->run_user_uid);
	      /* make sure we cannot get root again
	       */
	      if (aud_setuid(FIL__, __LINE__,0) >= 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  
	  PDBGC(3);
	  fflush(NULL);
	  
	  if (task->rw == 'w')
	    {
	      PDBGC_S("w");
	      fcntl  (STDOUT_FILENO, F_SETFD, FD_CLOEXEC);
	      fcntl  (STDERR_FILENO, F_SETFD, FD_CLOEXEC);
	      /*
	      freopen(_("/dev/null"), "r+", stderr);
	      freopen(_("/dev/null"), "r+", stdout);
	      */
	    }
	  else
	    {
	      PDBGC_S("r");
	      retry_aud_dup2 (FIL__, __LINE__, STDOUT_FILENO, STDERR_FILENO);
	      fcntl  (STDIN_FILENO, F_SETFD, FD_CLOEXEC);
	      /*
	      freopen(_("/dev/null"), "r+", stdin);
	      */
	    }
	  
	  PDBGC(4);
	  
	  
#if defined(__linux__)
	  /* 
	   * --  emulate an fdexec with checksum testing
	   */
	  if (task->checksum[0]  != '\0')
	    {
	      PDBGC_S("fexecve");
	      fd=sl_open_read(task->command, 
			      task->privileged == 0 ? SL_NOPRIV : SL_YESPRIV);

	      tiger_fd = fd;
	      if (0 != sl_strcmp(task->checksum, 
				 sh_tiger_hash (task->command, TIGER_FD, 0))
		  )
		{
		  PDBGC_S("fexecve: checksum mismatch");
		  aud__exit(FIL__, __LINE__,EXIT_FAILURE);
		}

	      PDBGC(5);
	      pfd = get_the_fd(fd);
	      sprintf(pname, _("/proc/self/fd/%d"),      /*  known to fit */
			   pfd);
	      if (0 == access(pname, R_OK|X_OK))
		{
		  PDBGC(6);
		  fcntl  (pfd, F_SETFD, FD_CLOEXEC);
		  retry_aud_execve (FIL__, __LINE__, 
				    pname, 
				    (task->argc == 0) ? NULL : task->argv, 
				    (task->envc == 0) ? NULL : task->envv
				    );
		  
		  errnum = errno;
		  PDBGC_S(strerror(errnum));
		  PDBGC_S(task->command);
		  PDBGC_S("fexecve: failed");

		  /* failed 
		   */
		  aud__exit(FIL__, __LINE__,EXIT_FAILURE);
		}

	      PDBGC_S("fexecve: not working");
	      /* procfs not working, go ahead 
	       */
	      sl_close(fd);
	    }
#endif

	  PDBGC_S(" -- non fexecve --");
	  /* 
	   * --  execute path if executable
	   */
	  if (0 == access(task->command, R_OK|X_OK))
	    {
	      PDBGC(5);
	      retry_aud_execve (FIL__, __LINE__, 
				task->command, 
				(task->argc == 0) ? argp : task->argv, 
				(task->envc == 0) ? envp : task->envv
				);
	    }
	  errnum = errno;
	  PDBGC_S(strerror(errnum));
	  PDBGC_S(task->command);
	  PDBGC_S("execve: failed");
	  /* failed 
	   */
	  aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	}
      aud__exit (FIL__, __LINE__,0);
    }

  
  /*
   * -- parent
   */
  waitpid (task->pid, NULL, 0);

  PDBG(5);
  /* open an output stream on top of the write side of the pipe
   */
  if (task->rw == 'w')
    {
      PDBG_S("is w");
      close (pipedes[STDIN_FILENO]);
      retry_fcntl (pipedes[STDOUT_FILENO], F_SETFD, FD_CLOEXEC);
      MBLK( outf = fdopen (pipedes[STDOUT_FILENO], "w"); )
    }
  else
    {
      PDBG_S("is r");
      close (pipedes[STDOUT_FILENO]);
      retry_fcntl (pipedes[STDIN_FILENO], F_SETFD, FD_CLOEXEC);
      MBLK( outf = fdopen (pipedes[STDIN_FILENO], "r"); )
    }

  
  if (outf == NULL) 
    {
      PDBG_S("outf == NULL");
      tmp  = sh_util_safe_name (task->command);
      
      if (task->privileged == 0 && 0 == getuid())
	sh_error_handle((-1), FIL__, __LINE__, MSG_NOEXEC,
			(long) task->run_user_uid, tmp);
      else
	sh_error_handle((-1), FIL__, __LINE__, MSG_NOEXEC,
			(long) getuid(), tmp);

      SH_FREE(tmp);

      aud_kill (FIL__, __LINE__, task->pid, SIGKILL);
      close (pipedes[STDOUT_FILENO]);
      waitpid (task->pid, NULL, 0);
      task->pid = 0;

      return (-1);
    }
  
  if (task->rw == 'w')
    task->pipeFD   = pipedes[STDOUT_FILENO];
  else
    task->pipeFD   = pipedes[STDIN_FILENO];

  PDBG_D(task->pipeFD);

  flags = retry_fcntl (task->pipeFD, F_GETFL, 0);
  if (flags != (-1))
    retry_fcntl (task->pipeFD, F_SETFL, flags|O_NONBLOCK);
  task->pipe     = outf;

  PDBG_S("return from popen");
  
  return (0);
}


/*
 * -- close the pipe
 */
int sh_ext_pclose (sh_com_t * task)
{
  int status = 0;

  PDBG_S(" -> pclose");
  fflush(task->pipe);
  MBLK( fclose(task->pipe); )

  task->pipe     = NULL;
  task->pid      = 0;
  task->pipeFD   = (-1);

  PDBG_S(" <--");
  return (status);
}

/*
 * -- check filters
 */
int sh_ext_filter (char * message, sh_com_t * task)
{
  int i;
  int j = 0;
  time_t now_time;

  /* Presence of any of these keywords prevents execution.
   */
  if (task->fnot_c > 0)
    {
      for (i = 0; i < task->fnot_c; ++i)
	if (NULL != sl_strstr(message, task->fnot_v[i]))
	  return (-1);
    }

  /* Presence of all of these keywords is required for execution.
   */
  if (task->fand_c > 0)
    {
      j = 0;

      for (i = 0; i < task->fand_c; ++i)
	if (NULL != sl_strstr(message, task->fand_v[i]))
	  ++j;

      if (j != task->fand_c)
	return (-1);
    }

  /* Presence of at least one of these keywords is required for execution.
   */
  if (task->for_c > 0)
    {
      for (i = 0; i < task->for_c; ++i)
	{
	  if (NULL != sl_strstr(message, task->for_v[i]))
	    {
	      if (task->deadtime != (time_t) 0)   /* deadtime */
		{
		  now_time = time (NULL);
		  
		  if (task->last_run == (time_t) 0)
		    {
		      task->last_run = now_time;
		    }
		  else if ((time_t)(now_time-task->last_run) < task->deadtime)
		    {
		      return (-1);
		    }
		  else
		    {
		      task->last_run = now_time;
		    }
		}
	      return (0);
	    }
	}
      return (-1);
    }

  return (0);
}



/*
 * -- execute external script/program
 */
int sh_ext_execute (char t1, char t2, char t3, char * message, int msg_siz)
{
  sh_com_t * listval = ext_coms;

  int        status = 0;

  char     * tmp;

  struct  sigaction  new_act;
  struct  sigaction  old_act;

  PDBG_OPEN;

  if (listval == NULL)
    return (-1);

  PDBG(-1);

  if (msg_siz == 0)
    msg_siz = sl_strlen(message);


  /* ignore SIGPIPE (instead get EPIPE if connection is closed)
   */
  new_act.sa_handler = SIG_IGN;
  retry_sigaction (SIGPIPE, &new_act, &old_act);

  while (listval != NULL)
    {
      PDBG(-2);
      if (t1 == listval->type[0] &&
	  t2 == listval->type[1] &&
	  t3 == listval->type[2] &&
	  0 == sh_ext_filter (message, listval))
	{
	  PDBG(-3);

	  if (0 == sh_ext_popen (listval))
	    {
	      PDBG(-4);
	      if (NULL != listval->pipe && listval->rw == 'w')
		{
		  PDBG(-5);
		  if (message != NULL)
		    {
		      PDBG(-6);
		      status = write (listval->pipeFD, 
				      message, msg_siz);
		      if (status >= 0)
			status = write (listval->pipeFD, "\n", 1);
		    }
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "[", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "E", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "O", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "F", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "]", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = write (listval->pipeFD, "\n", 1);
		  PDBG_D(status);
		  if (status < 0)
		    {
		      PDBG_S("some error");
		      PDBG_D(status);
		      tmp  = sh_util_safe_name (listval->command);

		      if (tmp)
			{
			  if (listval->privileged == 0 && 
			      (0 == getuid() || sl_is_suid()) )
			    sh_error_handle((-1), FIL__, __LINE__, MSG_NOEXEC,
					    (long) listval->run_user_uid, tmp);
			  else
			    sh_error_handle((-1), FIL__, __LINE__, MSG_NOEXEC,
					    (long) getuid(), tmp);
			  
			  SH_FREE(tmp);
			}

		    }
		  PDBG(-7);
		  fflush(listval->pipe);
		}
	      PDBG(-8);
	      sh_ext_pclose(listval);
	    }
	  else
	    {
	      PDBG_S("0 != sh_ext_popen()");
	    }
	}
      listval = listval->next;
    }
  PDBG_S("no more commands");

  /* restore old signal handler
   */
  retry_sigaction (SIGPIPE, &old_act, NULL);
  PDBG_S("return");
  PDBG_CLOSE;
  return 0;
}
  
  
/* #if defined(WITH_EXTERNAL) */
#endif


