/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2000 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <string.h>

#include "samhain.h"
#include "sh_error.h"
#include "sh_utils.h"
#include "sh_tiger.h"

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif


#undef  FIL__
#define FIL__  _("sh_err_log.c")

typedef struct _sh_log_buf {
  char   signature[KEY_LEN+1];
  char   timestamp[KEY_LEN+1];
  char * msg;
} sh_sh_log_buf;

extern struct  _errFlags  errFlags;

#define CHK_KEY 0
#define CHK_FIL 1
#define CHK_NON 2

int get_key_from_file(char * path, char * keyid, char * key)
{
  SL_TICKET  fd;
  char * buf;
  char * bufc;

  if (path[strlen(path)-1] == '\n')
    path[strlen(path)-1] = '\0';

  /* open the file, then check it 
   */
  if ( SL_ISERROR(fd = sl_open_read (path, SL_NOPRIV)))
    {
      fprintf(stderr, _("Could not open file <%s>\n"), path);
      _exit (EXIT_FAILURE);
    }

  buf     = SH_ALLOC( (unsigned long)(SH_BUFSIZE+1));
  bufc    = SH_ALLOC( (unsigned long)(SH_MAXBUF+1));

  while (1)
    {
      buf[0]  = '\0';
      bufc[0] = '\0';

      /* find start of next key
       */
      while (0 != sl_strncmp(buf, _("-----BEGIN LOGKEY-----"),
			     sizeof("-----BEGIN LOGKEY-----")-1)) 
	{
	  (void) sh_unix_getline (fd, buf, SH_BUFSIZE);
	  if (buf[0] == '\0')
	    {
	      /* End of file reached, return. 
	       */
	      fflush(stdout);
	      sl_close(fd);
	      return -1; 
	    }
	}

      /* read key
       */
      (void) sh_unix_getline (fd, buf, SH_BUFSIZE);

      if (0 == sl_strncmp(keyid, &buf[KEY_LEN], strlen(keyid)))
	{
	  sl_strlcpy(key, buf, KEY_LEN+1);
	  sl_close(fd);
	  return 0;
	}
    }
	  
  /* not found
   */
  sl_close(fd);
  return -1;
}

int sh_error_logverify (char * s)
{
  SL_TICKET fd;
  int len;
  int status;
  int count =  0;
  int start = -1;
  char * buf;
  char * bufc;
  char signature[64];
  char key[KEY_LEN+1];
  char path[KEY_LEN+1];
  char timestamp[64];
  char c_cont;
  int  chk_mode = CHK_KEY;

  sh_error_logoff();

  if (s == NULL || sl_strlen(s) >= PATH_MAX)
    {
      fprintf(stderr, _("FAILED: msg=<Invalid input.>, path=<%s>\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Open the file, then check it. 
   */
  if ( SL_ISERROR(fd = sl_open_read (s, SL_NOPRIV)) )
    {
      fprintf(stderr, _("FAILED: msg=<File not accessible.>, path=<%s>\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Find space value.
   */
  c_cont  = ' ';
#ifdef SH_STEALTH
  c_cont ^= XOR_CODE;
#endif

  buf  = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );
  bufc = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );

  while (1) 
    {
      /* get the log message
       */
      if (sh_unix_getline (fd, buf, (2*SH_BUFSIZE)) < 0) 
	break;

      len = (int) sl_strlen(buf);

#ifdef SH_STEALTH
      if (0 == sl_strncmp (buf, N_("[SOF]"), 5)) 
#else
      if (0 == sl_strncmp (buf, _("[SOF]"),  5)) 
#endif
	{
	  /* Found start of audit trail, read first line. 
	   */
	  start = 1;
	  do {
	    (void) sh_unix_getline (fd, buf, (2*SH_BUFSIZE));
	  } while (buf[0] == '\0' || buf[0] == '\n');
	  len = (int) sl_strlen(buf);
	  ++count;
	}
      else if (buf[0] == '\n')
	{
	  /* A newline.
	   */
	  ++count;
	  continue;
	}
      else if (start == 0)
	{
	  /* We are inside an audit trail. 
	   */
	  ++count;
	}
      else
	{
	  /* No start-of-file found yet. 
	   */
	  continue;
	}

    /* Check for a continuation line.
     */
    while (1 == 1)
      {
	do {
	  (void) sh_unix_getline (fd, bufc, (2*SH_BUFSIZE));
	} while (bufc[0] == '\0' || bufc[0] == '\n');
	++count;
	if (bufc[0] == c_cont) 
	  {
	    /* A continuation line. Add the newline. 
	     */
	    sl_strlcat(buf, "\n", 2*SH_BUFSIZE+1);
	    ++len;
	    sl_strlcat(buf, bufc, 2*SH_BUFSIZE+1);
	    len += (int) sl_strlen(bufc);
	  }
	else
	  {
	    /* No continuation line. Use it as signature. 
	     * A48014C05604EF7C9472330E85453E704024943E556163C2
	     */
	    sl_strlcpy(signature, bufc, KEY_LEN+1);
	    if (sl_strlen(bufc) > KEY_LEN)
	      sl_strlcpy(timestamp, &bufc[KEY_LEN], 64);
	    break;
	  }
      }

    /* Get starting key from command line. 
     */    
    if (start == 1) 
      {

	/* Get the timestamp. Hardwired for default header.
	 */
	/*
	sl_strlcpy(timestamp, &buf[10], 20);
	timestamp[19] = '\0';
	*/
#ifdef SH_STEALTH
	sh_do_decode (timestamp, sl_strlen(timestamp));
#endif
	key[0] = '\0';

      findKey:

	if (chk_mode != CHK_FIL)
	  {
	    /* Ask for the key.
	     */
	    chk_mode = CHK_KEY;
	    fprintf(stdout, _("\nNew audit trail (%s), enter key|keyfile: "), 
		    timestamp);
	    key[0] = '\0';

	    while (sl_strlen(key) < KEY_LEN ) 
	      { 
		if (key[0] != '\n' && key[0] != '\0')
		  fprintf(stdout, _("New audit trail, enter key: "));
		else if (key[0] == '\n')
		  {
		    sl_strlcpy(key, sh_tiger_hash(NULL, TIGER_DATA, 0), 
			       KEY_LEN+1);
		    chk_mode = CHK_NON;
		    break;
		  }
		fflush(stdout); 
		key[0] = '\0';
		fgets(key, KEY_LEN+1, stdin);
		if (key[0] == '/')
		  {
		    chk_mode = CHK_FIL;
		    sl_strlcpy(path, key, KEY_LEN+1); 
		    break;
		  }
	      }
	  }
	/* we now have either a key (chk_mode == CHK_NON|CHK_KEY)
	 * or a file (chk_mode == CHK_FIL)
	 */
	if (chk_mode == CHK_FIL)
	  {
	    fprintf(stdout, _("\nAudit trail (%s), searching file %s\n"), 
			 timestamp, path);
	    if (-1 == get_key_from_file(path, timestamp, key))
	      {
		chk_mode = CHK_KEY;
		fprintf(stdout, _("Key not found in file\n"));
		goto findKey;
	      }
	  }
	      

	sh_util_encode(key, buf, 1);
	start = 0;
      } 
    else
      { 
	/* Iterate the key.
	 */
	sl_strlcpy (key, 
		    sh_tiger_hash (key, TIGER_DATA, KEY_LEN), 
		    KEY_LEN+1);
      }
    
    sl_strlcat ( buf, key, 2*SH_BUFSIZE + 1);
    
#ifdef SH_STEALTH
    sh_do_decode (signature, sl_strlen(signature));
#endif
    
    status = sl_strncmp (signature, 
			 sh_tiger_hash (buf, TIGER_DATA, sl_strlen(buf)),
			 KEY_LEN);
    
    buf[len] = '\0';    /* do not print out the key */
#ifdef SH_STEALTH
    sh_do_decode (buf, sl_strlen(buf));
#endif
    
    if (status != 0) 
      {
	if (chk_mode == CHK_NON)
	  fprintf (stdout, _("NOVRFY: line=<%5d>, msg=<%s>\n"), count-1, buf);
	else
	  fprintf (stdout, _("FAILED: line=<%5d>, msg=<%s>\n"), count-1, buf);
      }
    else 
      fprintf (stdout, _("passed: line=<%5d>, msg=<%s>\n"), count-1, buf);
    
    }

  /* Cleanup and exit.
   */
  sl_close (fd);
  SH_FREE  (buf);
  SH_FREE  (bufc);
  fflush   (stdout);
  _exit    (EXIT_SUCCESS);

  /* Make compilers happy. 
   */
  return 0; 
}



/*
 *   --- Log error message to log file. ---
 */
int  sh_log_file (char *errmsg)
{
  int                  store1;
  int                  store2;
  SL_TICKET            fd = -1;
  long int             status;
  char               * tmp = NULL;
  struct _sh_log_buf   log_msg;

  if (errFlags.HaveLog == BAD)  /* paranoia */ 
    return (-1);

  /* open/create the file, then check it 
   */
  tmp  = sh_util_safe_name (sh.srvlog.name);

  if (  0 !=  (status = tf_trust_check (sh.srvlog.name, SL_YESPRIV)))
    sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_TRUST,
		      (long) sh.effective.uid, tmp);

  if (status == 0)
    {
      fd = sl_open_write (sh.srvlog.name, SL_YESPRIV);
      if (SL_ISERROR(fd))
	{
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
			    (long) sh.effective.uid, tmp);
	  status = -1;
	}
    }


  if (status == 0)
    {
      status = sh_unix_testlock(sh.srvlog.name);
      if (status < 0)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_LOCKED,
			    (long) sh.effective.uid, tmp, sh.srvlog.alt);
	  status = -1;
	}
    }

  if (status == 0)
    {
      status = SL_ISERROR(sl_forward(fd) ); 
      if (status < 0)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
			    (long) sh.effective.uid, tmp);
	  status = -1;
	}
    }
  
  if (status < 0)
    {
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_SRV_FAIL,
			_("logfile"), tmp);
      SH_FREE(tmp);
      return (-1);
    }


  /* --- Allocate storage and mlock it. ---
   */

  status      = sl_strlen (errmsg);
  log_msg.msg = (char *) SH_ALLOC ( 2*KEY_LEN + status + 3); 

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
  if (skey->mlock_failed == GOOD) 
    {
      sl_set_suid ();
      if ( (-1) == mlock( log_msg.msg, 2*KEY_LEN + status + 3 ) ) 
	{
	  sl_unset_suid ();
	  skey->mlock_failed = BAD;
	  sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);
	}
      else
	sl_unset_suid ();
    }
#else
  if (skey->mlock_failed == GOOD) 
    {
      skey->mlock_failed = BAD;
      sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);
    }
#endif

  /* --- Write the start marker. --- 
   */

  if (sh.flag.log_start == S_TRUE) 
    {
#ifdef SH_STEALTH
      sl_write (fd, "\n", 1);
      sl_write_line (fd, N_("[SOF]"), 5);
      sl_sync(fd);
#else
      sl_write_line (fd, _("\n[SOF]"),  6);
      sl_sync(fd);
#endif
    }

  /* reserve KEY_LEN chars at end for key 
   */
  sl_strlcpy (log_msg.msg, errmsg, status+1 );

#ifdef SH_STEALTH
  sh_do_encode (log_msg.msg, status);
#endif

  /* write the signature 
   */
  if (sh.flag.log_start == S_TRUE) 
    {
      store1               = errFlags.loglevel;
      store2               = errFlags.sysloglevel;
      errFlags.loglevel    = SH_ERR_NOT;
      errFlags.sysloglevel = SH_ERR_NOT;

      if (sh.real.user[0] == '\0') 
	(void) sh_unix_getUser();

      /* Initialize the key.
       */
      sh_util_keyinit(skey->sigkey_old, KEY_LEN+1);

      /* Hash the key to make sure it has the correct format.
       */
      sl_strlcpy(skey->sigkey_new, 
		 sh_tiger_hash (skey->sigkey_old, TIGER_DATA, KEY_LEN), 
		 KEY_LEN+1);

      /* Copy it to 'crypt' for encryption.
       */
      sl_strlcpy(skey->crypt, skey->sigkey_new, KEY_LEN+1);

      /* Use message and compiled-in key to encrypt.
       */
      sh_util_encode(skey->crypt, log_msg.msg, 0);

      /* Send out the key.
       */
      sl_strlcpy(log_msg.timestamp, sh_unix_time(0), KEY_LEN+1); 
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_KEY,
		       sh.prg_name, skey->crypt, 
		       skey->crypt, log_msg.timestamp);

      /* Cleanup.
       */
      memset (skey->crypt, '\0', KEY_LEN);
      errFlags.loglevel    = store1;
      errFlags.sysloglevel = store2;
      sh.flag.log_start   = S_FALSE;  
    } 
  else 
    {
      log_msg.timestamp[0] = '\0';
      sl_strlcpy (skey->sigkey_new, 
		  sh_tiger_hash (skey->sigkey_old, TIGER_DATA, KEY_LEN),
		  KEY_LEN+1);
    }

  /* --- Sign the message with the signature key. ---
   */
  sl_strlcat (log_msg.msg, skey->sigkey_new, status + KEY_LEN + 2);
  
  sl_strlcpy (log_msg.signature,
	      sh_tiger_hash (log_msg.msg, TIGER_DATA, status + KEY_LEN), 
	      KEY_LEN+1);
  sl_strlcpy (skey->sigkey_old, skey->sigkey_new, KEY_LEN+1); 

#ifdef SH_STEALTH
  sh_do_encode (log_msg.signature, KEY_LEN);
  sh_do_encode (log_msg.timestamp, sl_strlen(log_msg.timestamp));
#endif
  
  log_msg.msg[status] = '\0';
  sl_strlcat (log_msg.msg,              "\n", status + KEY_LEN + 2);
  sl_strlcat (log_msg.msg, log_msg.signature, status + KEY_LEN + 2);
  if (log_msg.timestamp[0] != '\0')
    sl_strlcat (log_msg.msg, log_msg.timestamp, status + 2*KEY_LEN + 2);
  sl_strlcat (log_msg.msg,              "\n", status + 2*KEY_LEN + 3);
  
  /* --- Write out the record. ---
   */
  sl_write (fd, log_msg.msg, strlen(log_msg.msg));
  sl_sync  (fd);
  sl_close (fd);

  /* --- Clean up and free record. ---
   */
  memset (log_msg.msg,       '\0', status + 2*KEY_LEN + 3);
  memset (log_msg.signature, '\0', KEY_LEN);
  MUNLOCK(log_msg.msg,  status + 2*KEY_LEN + 3);
  SH_FREE(log_msg.msg);

  if (tmp != NULL)
    SH_FREE(tmp);
  return (0);
}

