/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdlib.h>
#include <errno.h>

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/socket.h>

#include "samhain.h"
#include "sh_error.h"

typedef struct cht_struct 
{
  char           * str;
  unsigned long    val;
} cht_type;

static cht_type aud_tab[] =
{
  { N_("execve"),    AUD_EXEC   },
  { N_("utime"),     AUD_UTIME  },
  { N_("unlink"),    AUD_UNLINK },
  { N_("dup"),       AUD_DUP    },
  { N_("chdir"),     AUD_CHDIR  },
  { N_("open"),      AUD_OPEN   },
  { N_("kill"),      AUD_KILL   },
  { N_("exit"),      AUD_EXIT   },
  { N_("fork"),      AUD_FORK   },
  { N_("setuid"),    AUD_SETUID },
  { N_("setgid"),    AUD_SETGID },
  { N_("pipe"),      AUD_PIPE   },
  { NULL,            0 }
};

/* Set aud functions
 */
int sh_aud_set_functions(char * str_s)
{
  int i = 0;
  
  if (str_s == NULL)
    return -1;

  while (aud_tab[i].str != NULL)
    {
      if (NULL != sl_strstr (str_s, aud_tab[i].str))
	{
	  sh.flag.audit     = 1;
	  sh.flag.aud_mask |= aud_tab[i].val;
	}
      ++i;
    }

  return 0;
}

  


/* Need to catch EINTR for these functions.
 */
long int retry_sigaction(int signum,  const  struct  sigaction  *act,
			 struct sigaction *oldact)
{
  long int val_retry = -1;
  errno              = 0;

  do {
    val_retry = sigaction(signum, act, oldact);
  } while (val_retry < 0 && errno == EINTR);
  return val_retry;
}

long int retry_connect(int fd, struct sockaddr *serv_addr, int addrlen)
{
  long int val_retry = -1;
  errno              = 0;

  do {
    val_retry = connect(fd, serv_addr, addrlen);
  } while (val_retry < 0 && errno == EINTR);
  return val_retry;
}

long int retry_accept(int fd, struct sockaddr *serv_addr, int * addrlen)
{
  long int val_retry = -1;
#ifdef HOST_IS_AIX
  /* on AIX, addrlen is a size_t
   */
  size_t my_addrlen  = (size_t) *addrlen;
#else
  int    my_addrlen  = *addrlen;
#endif

  errno              = 0;

  do {
    val_retry = accept(fd, serv_addr, &my_addrlen);
  } while (val_retry < 0 && errno == EINTR);
  *addrlen = (int) my_addrlen;
  return val_retry;
}

long int retry_lstat(const char *file_name, struct stat *buf)
{
  long int val_retry = -1;
 
  do {
    val_retry = lstat (file_name, buf);
  } while (val_retry < 0 && errno == EINTR);
  return val_retry;
}

long int retry_stat(const char *file_name, struct stat *buf)
{
  long int val_retry = -1;
 
  do {
    val_retry = stat (file_name, buf);
  } while (val_retry < 0 && errno == EINTR);
  return val_retry;
}

long int retry_fstat(int filed, struct stat *buf)
{
  long int val_retry = -1;
 
  do {
    val_retry = fstat (filed, buf);
  } while (val_retry < 0 && errno == EINTR);
  return val_retry;
}

long int retry_fcntl(int fd, int cmd, long arg)
{
  long int val_retry = -1;
  errno              = 0;

  if (cmd == F_GETFD || cmd == F_GETFL)
    {
      do {
	val_retry = fcntl(fd, cmd);
      } while (val_retry < 0 && errno == EINTR);
    }
  else
    {
      do {
	val_retry = fcntl(fd, cmd, arg);
      } while (val_retry < 0 && errno == EINTR);
    }
  return val_retry;
}

/***************************************************
 *
 *   Audit these functions.
 *
 ***************************************************/

long int retry_aud_execve  (char * file, int line, 
			    const  char *dateiname, char * argv[],
			    char * envp[])
{
  uid_t a = geteuid();
  gid_t b = getegid();
  int   i;

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_EXEC) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_EXEC,
		     dateiname, (long) a, (long) b );
  do {
    i = execve(dateiname, argv, envp);
  } while (i < 0 && errno == EINTR);
  return i;
}


long int retry_aud_utime (char * file, int line, 
			   char * path, struct utimbuf *buf)
{
  long int val_return;
  errno      = 0;

  do {
    val_return = utime (path, buf);
  } while (val_return < 0 && errno == EINTR);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_UNLINK) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_UTIME,
		     path, 
		     (unsigned long) buf->actime, 
		     (unsigned long) buf->modtime);
  return val_return;
}

long int retry_aud_unlink (char * file, int line, 
			   char * path)
{
  long int val_return;
  errno      = 0;

  do {
    val_return = unlink (path);
  } while (val_return < 0 && errno == EINTR);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_UNLINK) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_UNLINK,
		     path);
  return val_return;
}

long int retry_aud_dup2 (char * file, int line, 
			 int fd, int fd2)
{
  long int val_return;
  errno      = 0;

  do {
    val_return = dup2 (fd, fd2);
  } while (val_return < 0 && errno == EINTR);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_DUP) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_DUP,
		      (long) fd, val_return);
  return val_return;
}

long int retry_aud_dup (char * file, int line, 
			int fd)
{
  long int val_return;
  errno      = 0;

  do {
    val_return = dup (fd);
  } while (val_return < 0 && errno == EINTR);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_DUP) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_DUP,
		     (long) fd, val_return);
  return val_return;
}


  
long int retry_aud_chdir (char * file, int line, 
			  const char *path)
{
  long int val_return;
  int      error      = 0;
  errno      = 0;

  do {
    val_return = chdir (path);
  } while (val_return < 0 && errno == EINTR);

  error = errno;
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_CHDIR) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_CHDIR,
		     path);
  errno = error;
  return val_return;
}
  

long int aud_open (char * file, int line, int privs,
		   const char *pathname, int flags, mode_t mode)
{
  long int val_return;

  if (privs == SL_YESPRIV)
    sl_set_suid();
  val_return = open (pathname, flags, mode);
  if (privs == SL_YESPRIV)
    sl_unset_suid();
  
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_OPEN) != 0)
    {
      sh_error_handle ((-1), file, line, 0, MSG_AUD_OPEN,
		       pathname, (long) flags, (long) mode, val_return);
    }
  return val_return;
}
  
long int aud_dup (char * file, int line, int fd)
{
  long int val_return = dup (fd);
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_DUP) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_DUP,
		      (long) fd, val_return);
  return val_return;
}
  
long int aud_kill (char * file, int line, pid_t pid, int sig)
{
  int  myerror;
  long int val_return = kill (pid, sig);
  myerror = errno;
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_KILL) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_KILL,
		      (long) pid, (long) sig);
  errno = myerror;
  return val_return;
}
  
void aud_exit (char * file, int line, int fd)
{
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_EXIT) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_EXIT,
		      (long) fd);
  sh.flag.exit = fd;
  exit(fd);
}

void aud__exit (char * file, int line, int fd)
{
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_EXIT) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_EXIT,
		      (long) fd);
  sh.flag.exit = fd;
  _exit(fd);
}

pid_t aud_fork (char * file, int line)
{
  pid_t i = fork();
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_FORK) != 0 && (i > 0))
    sh_error_handle ((-1), file, line, 0, MSG_AUD_FORK,
		      (long) i);
  return i;
}

int aud_setuid (char * file, int line, uid_t uid)
{
  int i = 0;

  if (uid != (uid_t) 0) 
    i = setuid(uid);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_SETUID) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_SETUID,
		      (long) i);
  if (uid == (uid_t) 0) 
    i = setuid(uid);

  return i;
}

int aud_setgid (char * file, int line, gid_t gid)
{
  int i = 0;

  if (gid != (gid_t) 0) 
    i = setgid(gid);

  if (sh.flag.audit && (sh.flag.aud_mask & AUD_SETGID) != 0)
    sh_error_handle ((-1), file, line, 0, MSG_AUD_SETGID,
		      (long) gid);
  if (gid == (gid_t) 0) 
    i = setgid(gid);

  return i;
}

int aud_pipe (char * file, int line, int modus[2])
{
  int i = pipe (modus);
  if (sh.flag.audit && (sh.flag.aud_mask & AUD_PIPE) != 0)
    {
      if (i < 0)
	sh_error_handle ((-1), file, line, 0, MSG_AUD_PIPE,
			 0, 0);
      else
	sh_error_handle ((-1), file, line, 0, MSG_AUD_PIPE,
			 modus[0], modus[1]);
    }
  return i;
}



