/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif


#include "samhain.h"
#include "sh_files.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_unix.h"
#include "sh_getopt.h"
#include "sh_readconf.h"
#include "sh_hash.h"

#include "sh_mail.h"

#include "sh_tiger.h"
#include "sh_gpg.h"
#include "sh_mem.h"
#include "sh_forward.h"
#include "sh_tools.h"
#include "sh_hash.h"
#if defined(WITH_EXTERNAL)
#include "sh_extern.h"
#endif
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
#include "sh_modules.h"
#endif

#undef  FIL__
#define FIL__  _("samhain.c")


/**************************************************
 *
 * Needed to compile the key into the code.
 *
 **************************************************/

extern UINT32  ErrFlag[2];
#include "sh_MK.h"

/**************************************************
 *
 * Variables for signal handling.
 *
 **************************************************/

volatile  int      sig_raised;
volatile  int      sig_debug_on;           /* SIGUSR1 */
volatile  int      sig_debug_off;          /* SIGUSR2 */
volatile  int      sig_fresh_trail;        /* SIGIOT  */
volatile  int      sig_config_read_again;  /* SIGHUP  */
volatile  int      sig_terminate;          /* SIGQUIT */
long int           eintr__result;
char               sh_sig_msg[SH_MINIBUF];


#ifdef SH_STEALTH
/**************************************************
 *
 * The following set of functions is required for
 * the 'stealth' mode.
 *
 **************************************************/

#ifndef SH_MAX_GLOBS
#define SH_MAX_GLOBS 16
#endif

#ifndef GLOB_LEN
#define GLOB_LEN 255
#endif

char * globber(char * str)
{
  int i;
  int j;

  static   int  items = 0;
  static   int  count = 0;
  static   char glob[SH_MAX_GLOBS * (GLOB_LEN+1)];

  if (str == NULL)
    return NULL;
  else
    j = strlen(str);

  ++items;

  ASSERT((j <= GLOB_LEN), _("j <= GLOB_LEN"))

  if (j > GLOB_LEN) 
    j = GLOB_LEN;

  /* Overwrap the buffer.
   */
  if ( (count + j) >= (SH_MAX_GLOBS * (GLOB_LEN+1)))
    {
      count = 0;
      items = 0;
    }

  for (i = 0; i < j; ++i)
    {
      if (str[i] != '\n' && str[i] != '\t' && str[i] != '\r' && str[i] != '"')
	glob[count + i] = str[i] ^ XOR_CODE;
      else
	glob[count + i] = str[i];
    }
  glob[count + j] = '\0';

  i     = count;
  count = count + j + 1;
  return &glob[i];
}

void sh_do_encode (char * str, int len)
{
  register          int i;

  /* this is a symmetric operation
   */
  for (i = 0; i < len; ++i)
    str[i] = str[i] ^ XOR_CODE;
  return;
}

#endif

/**************************************************
 *
 * Global variables.
 *
 **************************************************/

sh_struct   sh;
sh_key_t  * skey = NULL;

extern unsigned char TcpFlag[8][PW_LEN+1];

/**************************************************
 *
 * Initializing.
 *
 **************************************************/

static
void sh_init (void)
{
  unsigned char * dez = NULL;
  int             i;
#if defined(SH_WITH_MAIL)
  char          * p;
  char            q[SH_PATHBUF];
#endif

#ifdef MKA_09
  ErrFlag[0] |= (1 << 8);
#endif
#ifdef MKA_10
  ErrFlag[0] |= (1 << 9);
#endif
#ifdef MKA_11
  ErrFlag[0] |= (1 << 10);
#endif
#ifdef MKA_12
  ErrFlag[0] |= (1 << 11);
#endif
#ifdef MKA_13
  ErrFlag[0] |= (1 << 12);
#endif
#ifdef MKA_14
  ErrFlag[0] |= (1 << 13);
#endif
#ifdef MKA_15
  ErrFlag[0] |= (1 << 14);
#endif
#ifdef MKA_16
  ErrFlag[0] |= (1 << 15);
#endif

  /* Signal handling.
   */
  sig_raised            = 0;
  sig_config_read_again = 0;           /* SIGHUP  */
  sig_debug_on          = 0;           /* SIGUSR1 */
  sig_debug_off         = 0;           /* SIGUSR2 */
  sig_fresh_trail       = 0;           /* SIGIOT  */
  sig_terminate         = 0;           /* SIGQUIT */
  strcpy ( sh_sig_msg, _("None"));

#ifdef MKB_01
  ErrFlag[1] |= (1 << 0);
#endif
#ifdef MKB_02
  ErrFlag[1] |= (1 << 1);
#endif
#ifdef MKB_03
  ErrFlag[1] |= (1 << 2);
#endif
#ifdef MKB_04
  ErrFlag[1] |= (1 << 3);
#endif
#ifdef MKB_05
  ErrFlag[1] |= (1 << 4);
#endif
#ifdef MKB_06
  ErrFlag[1] |= (1 << 5);
#endif
#ifdef MKB_07
  ErrFlag[1] |= (1 << 6);
#endif
#ifdef MKB_08
  ErrFlag[1] |= (1 << 7);
#endif

#if defined(SH_WITH_SERVER) && !defined(SH_WITH_CLIENT)
  strncpy(sh.prg_name, _("Yule"), 8);
  sh.prg_name[4] = '\0';
#else
  strncpy(sh.prg_name, _("Samhain"), 8);
  sh.prg_name[7] = '\0';
#endif

  /* The flags.
   */
  sh.flag.checkSum        = SH_CHECK_NONE;
  sh.flag.update          = S_FALSE;
  sh.flag.opts            = S_FALSE;
  sh.flag.isdaemon        = S_FALSE;
  sh.flag.isserver        = S_FALSE;
  sh.flag.islocked        = S_FALSE;
  sh.flag.smsg            = S_FALSE;
  sh.flag.log_start       = S_TRUE;
  sh.flag.reportonce      = S_TRUE;
  sh.flag.fulldetail      = S_FALSE;
  sh.flag.audit           = S_FALSE;
  sh.flag.aud_mask        = 0xFFFFFFFFUL;
  sh.flag.client_severity = S_FALSE;
  sh.flag.client_class    = S_FALSE;
  sh.flag.hidefile        = S_FALSE;

#ifdef MKB_09
  ErrFlag[1] |= (1 << 8);
#endif
#ifdef MKB_10
  ErrFlag[1] |= (1 << 9);
#endif
#ifdef MKB_11
  ErrFlag[1] |= (1 << 10);
#endif
#ifdef MKB_12
  ErrFlag[1] |= (1 << 11);
#endif
#ifdef MKB_13
  ErrFlag[1] |= (1 << 12);
#endif
#ifdef MKB_14
  ErrFlag[1] |= (1 << 13);
#endif
#ifdef MKB_15
  ErrFlag[1] |= (1 << 14);
#endif
#ifdef MKB_16
  ErrFlag[1] |= (1 << 15);
#endif

  /* The stats.
   */
  sh.stat.openfiles    = 0;
  sh.stat.bytes_hashed = 0;
  sh.stat.mail_success = 0;
  sh.stat.mail_failed  = 0;
  sh.stat.log_success  = 0;
  sh.stat.log_failed   = 0;
  sh.stat.time_start   = time(NULL);
  sh.stat.time_check   = (time_t) 0;

#ifdef MKC_01
  ErrFlag[0] |= (1 << 16);
#endif
#ifdef MKC_02
  ErrFlag[0] |= (1 << 17);
#endif
#ifdef MKC_03
  ErrFlag[0] |= (1 << 18);
#endif
#ifdef MKC_04
  ErrFlag[0] |= (1 << 19);
#endif
#ifdef MKC_05
  ErrFlag[0] |= (1 << 20);
#endif
#ifdef MKC_06
  ErrFlag[0] |= (1 << 21);
#endif
#ifdef MKC_07
  ErrFlag[0] |= (1 << 22);
#endif
#ifdef MKC_08
  ErrFlag[0] |= (1 << 23);
#endif


  /* The local host.
   */
  sl_strlcpy (sh.host.name,  _("localhost"),  SH_MINIBUF);
  sh.host.system[0]     = '\0';
  sh.host.release[0]    = '\0';
  sh.host.machine[0]    = '\0';

#ifdef MKC_09
  ErrFlag[0] |= (1 << 24);
#endif
#ifdef MKC_10
  ErrFlag[0] |= (1 << 25);
#endif
#ifdef MKC_11
  ErrFlag[0] |= (1 << 26);
#endif
#ifdef MKC_12
  ErrFlag[0] |= (1 << 27);
#endif
#ifdef MKC_13
  ErrFlag[0] |= (1 << 28);
#endif
#ifdef MKC_14
  ErrFlag[0] |= (1 << 29);
#endif
#ifdef MKC_15
  ErrFlag[0] |= (1 << 30);
#endif
#ifdef MKC_16
  ErrFlag[0] |= (1UL << 31);
#endif

  /* The paths.
   */
  sl_strlcpy (sh.conf.path,  DEFAULT_CONFIGFILE,    SH_PATHBUF);
  sh.conf.hash[0] = '\0';
  sl_strlcpy (sh.data.path,  DEFAULT_DATA_FILE,     SH_PATHBUF);
  sh.data.hash[0] = '\0';
  sh.exec.path[0] = '\0';
  sh.exec.hash[0] = '\0';

#ifdef MKD_01
  ErrFlag[1] |= (1 << 16);
#endif
#ifdef MKD_02
  ErrFlag[1] |= (1 << 17);
#endif
#ifdef MKD_03
  ErrFlag[1] |= (1 << 18);
#endif
#ifdef MKD_04
  ErrFlag[1] |= (1 << 19);
#endif
#ifdef MKD_05
  ErrFlag[1] |= (1 << 20);
#endif
#ifdef MKD_06
  ErrFlag[1] |= (1 << 21);
#endif
#ifdef MKD_07
  ErrFlag[1] |= (1 << 22);
#endif
#ifdef MKD_08
  ErrFlag[1] |= (1 << 23);
#endif

  /* The addresses.
   */
#if defined(SH_WITH_MAIL)
  if (0 == strcmp (DEFAULT_MAILADDRESS, _("NULL")))
    {
      sl_strncpy(q, DEFAULT_MAILADDRESS, SH_PATHBUF);
      p = strtok (q, ", \t");
      if (p)
	{
	  sh_mail_setaddress_int (p);
	  while (NULL != (p = strtok (NULL, ", \t")))
	    sh_mail_setaddress_int (p);
	}
    }
#endif

  if (0 == strcmp (ALT_TIMESERVER, _("NULL")))
    sh.srvtime.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvtime.alt, ALT_TIMESERVER,        SH_PATHBUF);
  if (0 == strcmp (DEFAULT_TIMESERVER, _("NULL")))
    sh.srvtime.name[0] = '\0';
  else
    sl_strlcpy (sh.srvtime.name, DEFAULT_TIMESERVER,   SH_PATHBUF);


  if (0 == strcmp (ALT_LOGSERVER, _("NULL")))
    sh.srvexport.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvexport.alt,  ALT_LOGSERVER,  SH_PATHBUF);
  if (0 == strcmp (DEFAULT_LOGSERVER, _("NULL")))
    sh.srvexport.name[0] = '\0';
  else
    sl_strlcpy (sh.srvexport.name,  DEFAULT_LOGSERVER, SH_PATHBUF);


  if (0 == strcmp (DEFAULT_ERRLOCK, _("NULL")))
    sh.srvlog.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.alt,  DEFAULT_ERRLOCK,       SH_PATHBUF);
  if (0 == strcmp (DEFAULT_ERRFILE, _("NULL")))
    sh.srvlog.name[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.name,  DEFAULT_ERRFILE,      SH_PATHBUF);


  if (0 == strcmp (ALT_CONSOLE, _("NULL")))
    sh.srvcons.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvcons.alt,  ALT_CONSOLE,          SH_PATHBUF);
#ifndef DEFAULT_CONSOLE
  sl_strlcpy (sh.srvcons.name, _("/dev/console"),    SH_PATHBUF);
#else
  if (0 == strcmp (DEFAULT_CONSOLE, _("NULL")))
    sl_strlcpy (sh.srvcons.name, _("/dev/console"),    SH_PATHBUF);
  else
    sl_strlcpy (sh.srvcons.name,  DEFAULT_CONSOLE,     SH_PATHBUF);
#endif

#ifdef MKD_09
  ErrFlag[1] |= (1 << 24);
#endif
#ifdef MKD_10
  ErrFlag[1] |= (1 << 25);
#endif
#ifdef MKD_11
  ErrFlag[1] |= (1 << 26);
#endif
#ifdef MKD_12
  ErrFlag[1] |= (1 << 27);
#endif
#ifdef MKD_13
  ErrFlag[1] |= (1 << 28);
#endif
#ifdef MKD_14
  ErrFlag[1] |= (1 << 29);
#endif
#ifdef MKD_15
  ErrFlag[1] |= (1 << 30);
#endif
#ifdef MKD_16
  ErrFlag[1] |= (1UL << 31);
#endif


  /* The timers.
   */
  sh.fileCheck.alarm_last     = 0;
  sh.fileCheck.alarm_interval = 600; /* ten minutes */

  sh.mailTime.alarm_last     = 0;
  sh.mailTime.alarm_interval = 86400;

  sh.mailNum.alarm_last      = 0;
  sh.mailNum.alarm_interval  = 10;

  sh.looptime     = 60;


  /* The struct to hold privileged information.
   */
  skey = (sh_key_t *) my_malloc (sizeof(sh_key_t));
  if (skey == NULL) 
    {
      perror(_("sh_init"));
      _exit (EXIT_FAILURE);
    }

  skey->mlock_failed = GOOD;
  skey->rngI         = BAD;

  skey->poolc        = 0;

  skey->ErrFlag[0]   = ErrFlag[0];
  ErrFlag[0]         = 0;
  skey->ErrFlag[1]   = ErrFlag[1];
  ErrFlag[1]         = 0;

  dez = &(TcpFlag[POS_TF-1][0]);
  for (i = 0; i < PW_LEN; ++i)
    { 
      skey->pw[i] = (*dez); 
      (*dez)      = '\0';
      ++dez; 
    }

  sh_unix_mlock();
  return;
}


#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif

/*******************************************************
 * 
 * Exit Handler
 *
 *******************************************************/
void exit_handler(void)
{
  /* --- Clean up modules, if any. ---
   */
#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)
  int modnum;

  for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
    {
      if (modList[modnum].initval == GOOD)
	modList[modnum].mod_cleanup();
    }
#endif

  /* --- Push out all pending messages. ---
   */
#if defined(SH_WITH_MAIL)
  if (sh.mailNum.alarm_last > 0) 
    {
      (void) sh_mail_msg (NULL);
    }
#endif

  /* --- Write the server stat. ---
   */
#if defined(SH_WITH_SERVER)
  sh_forward_html_write();
#endif

  /* --- Clean up memory to check for problems. ---
   */
#ifdef MEM_DEBUG
#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)
  sh_files_deldirstack ();
  sh_files_delfilestack ();
  sh_hash_hashdelete();
#endif
#if defined(SH_WITH_SERVER)
  sh_forward_free_all ();
#endif
  delete_cache();
  sh_mem_stat();
#endif

  /* --- Checksum of executable. ---
   */
  sh_unix_self_check();


  /* --- Exit Message. ---
   */
  sh_error_handle ((-1), FIL__, __LINE__, sh.flag.exit, MSG_EXIT_NORMAL, 
		   sh.prg_name, sh_sig_msg);


  /* --- Restrict error logging to stderr. ---
   */
#ifdef WITH_MESSAGE_QUEUE
  close_ipc ();
#endif
  sh_error_only_stderr (S_TRUE);


  /* --- Remove lock, delete critical information. ---
   */
  sh_unix_unlock ();
  if (skey != NULL)
    memset (skey, '\0', sizeof(sh_key_t));
  
  /* --- Exit. ---
   */
  return;
}


/*******************************************************
 * 
 * Main program
 *
 *******************************************************/
int main(int argc, char * argv[])
{

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  int           modnum;
  time_t        runtim;
  float         st_1, st_2;
  int           status;
  long          cct = 0; /* main loop iterations */
#endif

  unsigned long told;
  unsigned long tcurrent;
  size_t        tzlen;

#if defined (SH_STEALTH_NOCL)
  char  * command_line;
  int     my_argc = 0;
  char  * my_argv[32];
#endif


  /* --- Install the exit handler. ---
   */
  (void) atexit(exit_handler);

  /* --- Zero the mailer key, and fill it. ---
   */
  memset (ErrFlag, 0, 2*sizeof(UINT32));

#ifdef MKA_01
  ErrFlag[0] |= (1 << 0);
#endif
#ifdef MKA_02
  ErrFlag[0] |= (1 << 1);
#endif
#ifdef MKA_03
  ErrFlag[0] |= (1 << 2);
#endif
#ifdef MKA_04
  ErrFlag[0] |= (1 << 3);
#endif
#ifdef MKA_05
  ErrFlag[0] |= (1 << 4);
#endif
#ifdef MKA_06
  ErrFlag[0] |= (1 << 5);
#endif
#ifdef MKA_07
  ErrFlag[0] |= (1 << 6);
#endif
#ifdef MKA_08
  ErrFlag[0] |= (1 << 7);
#endif

  (void) sh_derr();

  /* Save the timezone.
   */
  if (getenv("TZ") != NULL)
    {
      tzlen       = strlen(getenv("TZ"));
      sh.timezone = my_malloc (tzlen + 1);
      if (sh.timezone != NULL)
	sl_strlcpy (sh.timezone, getenv("TZ"), tzlen + 1);
    }
  else
    sh.timezone = NULL;
  


  /* --------  INIT  --------    
   */
  SL_REQUIRE(sl_policy_get_user(DEFAULT_IDENT) == SL_ENONE, 
	     _("sl_policy_get_user(DEFAULT_IDENT) == SL_ENONE")); 

  /* Restrict error logging to stderr.
   */
  sh_error_only_stderr (S_TRUE);

  /* Check that first three descriptors are open.
   */
  if ( retry_fcntl(0, F_GETFL, 0) == (-1) || 
       retry_fcntl(1, F_GETFL, 0) == (-1) || 
       retry_fcntl(2, F_GETFL, 0) == (-1))
    {
      sh_unix_closeall (0);
      aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/null"), O_RDWR, 0);
      aud_dup (FIL__, __LINE__, 0); 
      aud_dup (FIL__, __LINE__, 0);
    }


  /* --- Set default values. ---
   */
  sh_init ();


#if (defined (SH_WITH_SERVER) && !defined (SH_WITH_CLIENT))
  sh.flag.isserver = S_TRUE;
#endif

  /* --- Get local hostname. ---
   */
  sh_unix_localhost();

  /* --- Read the command line. ---
   */
  sh.flag.opts = S_TRUE;

#if !defined(SH_STEALTH_NOCL)
  (void) sh_getopt_get (argc, argv);
#else
  if (argc > 1 && argv[1] != NULL && 
      strlen(argv[1]) > 0 && strlen(NOCL_CODE) > 0)
    {
      if ( 0 == strcmp(argv[1], NOCL_CODE) &&
	   NULL != (command_line = (char *) SH_ALLOC(256 * sizeof(char))))
	{
	  my_argv[0] = argv[0]; ++my_argc;  
	  command_line[0] = '\0';
	  fgets (command_line, 255, stdin);
	  do {
	    my_argv[my_argc] = 
	      strtok( (my_argc == 1) ? command_line : NULL, " \n"); 
	    if (my_argv[my_argc] != NULL) {
	      ++my_argc;
	    } else {
	      break;
	    }
	  } while (my_argc < 32);
	  (void) sh_getopt_get (my_argc, my_argv);
	  SH_FREE (command_line);
	}
      else
	{
	  /* discard command line */
	  /* _exit(EXIT_FAILURE)  */  ; 
	}
    }
#endif
  sh.flag.opts = S_FALSE;
  

  /* --- Get user info. ---
   */
  TPT((0, FIL__, __LINE__, _("msg=<Get user hostname.>\n")))
  if (0 != sh_unix_getUser ())
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }


  /* *****************************
   *
   *  Read the configuration file.
   *
   * *****************************/

  TPT((0, FIL__, __LINE__, _("msg=<Read the configuration file.>\n")))
  (void) sh_readconf_read ();

  /* do not append to database if run SUID
   */
  if (sh.flag.checkSum == SH_CHECK_INIT && sl_is_suid()) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, EACCES, MSG_ACCESS,
		       (long) sh.real.uid, sh.data.path);
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }

  /* avoid daemon mode for initialization 
   */
  if (sh.flag.checkSum == SH_CHECK_INIT)
    sh.flag.isdaemon = S_FALSE;

  /* initialize signal handling etc
   */
  if (sh.flag.isdaemon == S_TRUE)
    { 
      sh_error_only_stderr (BAD);
#if defined(WITH_TRACE) || defined(WITH_TPT) 
      dbg_use_console();
#endif
    }

  TPT((0, FIL__, __LINE__, _("msg=<Initialize signal handling etc.>\n")))
  if (sh_unix_init(sh.flag.isdaemon) == -1) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }


  /* checksum of database
   */
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  TPT((0, FIL__, __LINE__, _("msg=<Get checksum of the database.>\n")))
  if (sh.flag.checkSum == SH_CHECK_CHECK) 
    {
      if (0 == strcmp(file_path('D', 'R'), _("REQ_FROM_SERVER")))
	{
	  /* fetch the file from server to get checksum
	   */
	  sh_hash_init ();
	  sh_hash_hashdelete ();
	}
      else
	{
	  sl_strlcpy(sh.data.hash,
		     sh_tiger_hash (file_path('D', 'R'), TIGER_FILE, 0), 
		     KEY_LEN+1);
	}
    }
#endif


  /* --- Enable full error logging --- 
   */
  sh_error_only_stderr (S_FALSE);

  /****************************************************
   *
   *   SERVER 
   *
   ****************************************************/

#if defined(SH_WITH_SERVER) && !defined(SH_WITH_CLIENT)

#if (defined(WITH_GPG) || defined(WITH_PGP))
  /* do nothing -- we exit earlier if error 
  if (0 != sh_gpg_check_sign (1)) 
    aud_exit(FIL__, __LINE__, EXIT_FAILURE);
  */
#else
  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_1H,
		   sh.prg_name, (long) sh.real.uid, 
		   (sh.flag.hidefile == S_TRUE) ? 
		   _("(hidden)") : file_path('C', 'R'), 
		   sh.conf.hash);
#endif

#else

  /****************************************************
   *
   *   CLIENT/STANDALONE
   *
   ****************************************************/

  if (sh.flag.checkSum == SH_CHECK_CHECK) 
    {
#if (defined(WITH_GPG) || defined(WITH_PGP))
      /* do nothing -- we exit earlier if error 
	 if (0 != sh_gpg_check_sign (2)) 
	 aud_exit(FIL__, __LINE__, EXIT_FAILURE);
      */
      ;
#else
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_2H,
		       sh.prg_name, (long) sh.real.uid,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('C', 'R'), sh.conf.hash,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('D', 'R'), sh.data.hash);
#endif
    }
  else
    {
#if (defined(WITH_GPG) || defined(WITH_PGP))
      /* do nothing -- we exit earlier if error 
      if (0 != sh_gpg_check_sign (1)) 
	aud_exit(FIL__, __LINE__, EXIT_FAILURE);
      */
      ;
#else
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_1H,
		       sh.prg_name, (long) sh.real.uid,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('C', 'R'), sh.conf.hash);
#endif
    }
#endif

 
  if (skey->mlock_failed == BAD)
    sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);

  /* timer
   */
  tcurrent                   = (unsigned long) time (NULL);
  told                       = tcurrent;
  sh.mailTime.alarm_last     = tcurrent;


  /****************************************************
   *
   *   SERVER 
   *
   ****************************************************/

#if defined(SH_WITH_SERVER)
  TPT((0, FIL__, __LINE__, _("msg=<Start server.>\n")))

#if defined (SH_WITH_CLIENT)
  if (sh.flag.isserver == S_TRUE)
    { 
      sh_receive();
      TPT((0, FIL__, __LINE__, _("msg=<End server.>\n")))
      aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
    }
#else
  sh_receive();
  TPT((0, FIL__, __LINE__, _("msg=<End server.>\n")))
  aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
#endif

#endif

  /****************************************************
   *
   *   CLIENT/STANDALONE
   *
   ****************************************************/
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 

  /* --- Initialize modules. ---
   */
  TPT((0, FIL__, __LINE__, _("msg=<Initialize modules.>\n")))
  for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
    {
      if (0 != (status = modList[modnum].mod_init()))
	{
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_MOD_FAIL,
			   _(modList[modnum].name),
			   status);
	  modList[modnum].initval = S_FALSE;
	}
      else
	{
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_MOD_OK,
			   _(modList[modnum].name));
	  modList[modnum].initval = S_TRUE;
	}
    }
    
  /*  --------  TEST SETUP  ---------
   */
  sh_files_setrec();
  sh_files_test_setup();

  /*  --------  MAIN LOOP  ---------
   */
  while (1) 
    {
      ++cct;

      TPT((0, FIL__, __LINE__, _("msg=<Start main loop.>, iter=<%ld>\n"), cct))

      tcurrent                   = (unsigned long) time (NULL);

      if (sig_raised != 0)
	{

	  TPT((0, FIL__, __LINE__, _("msg=<Process a signal.>\n")))

	  if (sig_config_read_again == 1) /* SIGHUP */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Re-read configuration.>\n")))
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_RECONF);

#if defined(WITH_EXTERNAL)
	      /* delete list of external tasks
	       */
	      (void) sh_ext_cleanup();
#endif
	      /* delete the file list, make all database
	       * entries visible (allignore = FALSE)
	       */
	      sh_files_deldirstack ();
	      sh_files_delfilestack ();
	      hash_full_tree ();

#if defined(SH_WITH_CLIENT)
	      reset_count_dev_server();
#endif
#if defined(SH_WITH_MAIL)
	      reset_count_dev_mail();
#endif
	      reset_count_dev_console();
	      reset_count_dev_time();

	      /* Should this be included ??? 
		 (i.e. should we reload the database ?)
	      */
#ifdef RELOAD_DATABASE
	      sh_hash_hashdelete();
#endif

	      (void) sh_readconf_read ();
	      sig_config_read_again = 0;
	      sh_files_setrec();
	      sh_files_test_setup();
	    }
	  
	  if (sig_fresh_trail == 1) /* SIGIOT */
	    {
	      /* Logfile access 
	       */
	      TPT((0, FIL__, __LINE__, _("msg=<Logfile stop/restart.>\n")))
	      sh_error_only_stderr (S_TRUE);
	      sh_unix_unlock();
	      sleep(3);
	      sh.flag.log_start = S_TRUE;
	      sh_error_only_stderr (S_FALSE);
	      sig_fresh_trail       = 0;
	    }
	  
	  if (sig_terminate == 1)  /* SIGQUIT */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Terminate.>\n")))
	      strncpy (sh_sig_msg, _("SIGQUIT"), 20);
	      aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
	    }
	  
	  if (sig_debug_on == 1)  /* SIGUSR1 */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Debug on.>\n")))
	      sh_error_dbg_on();
	      sig_debug_on = 0;
	    }
	  
	  if (sig_debug_off == 1)  /* SIGUSR2 */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Debug off.>\n")))
	      sh_error_dbg_off();
	      sig_debug_off = 0;
	    }
	  sig_raised = 0;
	  TPT((0, FIL__, __LINE__, _("msg=<End signal processing.>\n")))
	}
      
      
      /* see whether its time to check files
       */
      if (sh.flag.checkSum != SH_CHECK_NONE &&
	  tcurrent - sh.fileCheck.alarm_last >= sh.fileCheck.alarm_interval) 
	{
	  /* 
	   * check directories and files
	   * ORDER IS IMPORTANT -- DIRZ FIRST
	   */
	  sh.stat.bytes_hashed  = 0;
	  sh.stat.time_start    = time (NULL);
	  TPT((0, FIL__, __LINE__, _("msg=<Check directories.>\n")))
	  sh.stat.dirs_checked  = sh_dirs_chk  (); 
	  TPT((0, FIL__, __LINE__, _("msg=<Check files.>\n")))
	  sh.stat.files_checked = sh_files_chk ();

	  if (sig_terminate == 1)
	    continue;

	  /*
	   * check for files not visited
	   */
	  TPT((0, FIL__, __LINE__, _("msg=<Check for missing files.>\n")))
	  sh_hash_unvisited (ShDFLevel[SH_ERR_T_FILE]);

	  if (sig_terminate == 1)
	    continue;

	  /* reset
	   */
	  TPT((0, FIL__, __LINE__, _("msg=<Reset status.>\n")))
	  sh_dirs_reset  ();
	  if (sig_terminate == 1)
	    continue;

	  sh_files_reset ();
	  if (sig_terminate == 1)
	    continue;

	  runtim = time(NULL) - sh.stat.time_start;

	
	  if (sh.stat.dirs_checked == 0 && sh.stat.files_checked == 0)
	    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_CHECK_0);

	  else
	    {
	      st_1 = sh.stat.bytes_hashed;
	      st_2 = runtim;


	      if (st_1 > 0.0 && st_2 > 0.0) 
		st_1 = st_1/st_2;
	      else if (st_1 > 0.0)
		st_1 = st_1 * 1.0;
	      else
		st_1 = 0.0;
	      st_1 = 0.001 * st_1;
	       
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_CHECK_1,
			       (long) runtim, 
			       st_1);
	    }
	  sh.fileCheck.alarm_last = (unsigned long) time (NULL);

	  if (sig_terminate == 1)
	    continue;

	  /*
	   * flush mail queue
	   */
#if defined(SH_WITH_MAIL)
	  TPT((0, FIL__, __LINE__, _("msg=<Flush mail queue.>\n")))
	  (void) sh_mail_msg (NULL);
#endif
	}
      
      if (sig_terminate == 1)
	continue;
      
      /* execute modules
       */
      TPT((0, FIL__, __LINE__, _("msg=<Execute modules.>\n")))
      for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
	{
	  if (modList[modnum].initval == GOOD &&
	      0 != modList[modnum].mod_timer(tcurrent))
	    if (0 != (status = modList[modnum].mod_check()))
	      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_MOD_EXEC,
			       _(modList[modnum].name), (long) status);
	}
      
      /* no loop if not daemon
       */
      if (sh.flag.isdaemon != S_TRUE)
	break; 
      if (sig_terminate == 1)
	continue;
      
      /* see whether its time to send mail
       */
#if defined(SH_WITH_MAIL)
      if (tcurrent - sh.mailTime.alarm_last >= sh.mailTime.alarm_interval) 
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Flush mail queue.>\n")))
	  (void) sh_mail_msg (NULL);
	  sh.mailTime.alarm_last = (unsigned long) time (NULL);
	}
#endif
      if (sig_terminate == 1)
	continue;
            
      /* log the timestamp
       */
      if ((tcurrent - told) > (unsigned int) sh.looptime )
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Log the timestamp.>\n")))
	  told = tcurrent;
#ifdef MEM_DEBUG
	  sh_mem_check();
#else
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_STAMP);
#endif
	}
    
      /* seed / re-seed the PRNG if required
       */
      (void) taus_seed();
      
      if (sig_terminate == 1)
	continue;
      
      /* go to sleep
       */
      (void) sleep (1);

      (void) sh_derr();
    }
  
  /*   ------  END  -----------
   */

  /*
   * cleanup
   */
  TPT((0, FIL__, __LINE__, _("msg=<Cleanup.>\n")));
  sh_hash_hashdelete(); 

#if defined(SH_WITH_MAIL)
  if (sh.mailNum.alarm_last > 0) 
    (void)sh_mail_msg (NULL);
#endif

  /* #if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) */
#endif

  aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
  return (0);
}
