/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "config.h"
#include "shadow.hpp"
#include "strsep.hpp"

CShadowEntry::CShadowEntry(FILE *fpFile) : CEntry(fpFile, (void *)&strEntry, sizeof(struct spwd), &GetEntry)
{
	// Now if GetError return NO_ERROR in strEntry we have a valid shadow parsed entry
}

CShadowEntry::CShadowEntry(char *pszName)
{
	iErrorCode = NO_ERROR;	
	
	strEntry.sp_namp = strdup(pszName);
	strEntry.sp_pwdp = strdup("*");
	strEntry.sp_lstchg = time(NULL) / SECONDS_PER_DAY;
	strEntry.sp_min = 0;
	strEntry.sp_max = 99999;
	strEntry.sp_warn = -1;
	strEntry.sp_inact = -1;
	strEntry.sp_expire = -1;
	strEntry.sp_flag = (unsigned long)-1;
	if (NULL == strEntry.sp_namp || NULL == strEntry.sp_pwdp)
		iErrorCode = MEMORY_ERROR;
}

CShadowEntry::~CShadowEntry()
{
	if (NULL != strEntry.sp_namp)
		free(strEntry.sp_namp);
	if (NULL != strEntry.sp_pwdp)
		free(strEntry.sp_pwdp);
}

char *CShadowEntry::GetName()
{
	return strEntry.sp_namp;
}

char *CShadowEntry::GetPassword()
{
	return strEntry.sp_pwdp;
}

bool CShadowEntry::SetPassword(char *pszPassword,
							   int iMin,
							   int iMax,
							   int iWarn,
							   int iInact)
{
	time_t tNow = time(NULL) / SECONDS_PER_DAY;
	
	if (NULL != pszPassword) {				// Possibly we want to change aging
		if (((strEntry.sp_min != -1) && (tNow < strEntry.sp_lstchg + strEntry.sp_min)) ||
			((strEntry.sp_max != -1) && (tNow > strEntry.sp_lstchg + strEntry.sp_max)))
			return false;
		char *pszNewPassword = strdup(pszPassword);
		if (NULL == pszNewPassword)
			return false;
		free(strEntry.sp_pwdp);
		strEntry.sp_pwdp = pszNewPassword;
	}
	
	strEntry.sp_lstchg = tNow;
	if (iMin != -2)
		strEntry.sp_min = iMin;
	if (iMax != -2)
		strEntry.sp_max = iMax;
	if (iWarn != -2)
		strEntry.sp_warn = iWarn;
	if (iInact != -2)
		strEntry.sp_inact = iInact;
	
	return true;
}

bool CShadowEntry::GetEntry(char *pszEntry, void *pstrTarget)
{
	assert(NULL != pszEntry);
	assert(NULL != pstrTarget);
	
	struct spwd *pstrEntry = (struct spwd *)pstrTarget;
	
	register int i, j;
	
	for (i = j = 0; pszEntry[i] != '\0'; i++)
		if (':' == pszEntry[i])
			j += 1;
	
	if ((4 != j) && (8 != j))
		return false;
	
	bool fgError = false;
// user login name
	pstrEntry->sp_namp = strdup(strsep(&pszEntry, DELIMITERS));
// encrypted password
	pstrEntry->sp_pwdp = strdup(strsep(&pszEntry, DELIMITERS));
// last password change, if this is empty string then value of -1
	pstrEntry->sp_lstchg = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// days until change allowed, if this is empty string then value of -1
	pstrEntry->sp_min = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// days before change required, if this is empty string then value of -1
	pstrEntry->sp_max = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// days of warning before password expires, if this is empty string then value of -1
	pstrEntry->sp_warn = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// days after password expires until inactive, if this is empty string then value of -1
	pstrEntry->sp_inact = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// days until account expires, if this is empty string then value of -1
	pstrEntry->sp_expire = ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
// reserved, this field is not mandatory, if this is empty string then value of -1
	pstrEntry->sp_flag = (unsigned long)ToLong(strsep(&pszEntry, DELIMITERS), -1, &fgError);
	
	if (fgError ||
		 (NULL == pstrEntry->sp_namp) ||
		 (NULL == pstrEntry->sp_pwdp))
		return false;
	
	return true;
}

bool CShadowEntry::PutEntry(FILE *fpFile)
{
	char szFormat[64] = "%s:%s";
// There is no way the resulting format string to exceed 34 characters, but...
	
	if ((NULL == strEntry.sp_namp) || 
		 (NULL == strEntry.sp_pwdp))
		return false;
	
	strcat(szFormat, strEntry.sp_lstchg == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_min == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_max == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_warn == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_inact == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_expire == -1 ? ":%s" : ":%ld");
	strcat(szFormat, strEntry.sp_flag == (unsigned long)-1 ? ":%s" : ":%ld");
	strcat(szFormat, "\n");
	fprintf(fpFile,
			  szFormat,
			  strEntry.sp_namp,
			  strEntry.sp_pwdp,
			  strEntry.sp_lstchg == -1 ? (long)"" : strEntry.sp_lstchg,
			  strEntry.sp_min == -1 ? (long)"" : strEntry.sp_min,
			  strEntry.sp_max == -1 ? (long)"" : strEntry.sp_max,
			  strEntry.sp_warn == -1 ? (long)"" : strEntry.sp_warn,
			  strEntry.sp_inact == -1 ? (long)"" : strEntry.sp_inact,
			  strEntry.sp_expire == -1 ? (long)"" : strEntry.sp_expire,
			  strEntry.sp_flag == (unsigned long)-1 ? (long)"" : strEntry.sp_flag);
	
	return true;
}
