/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <stdio.h>
#include <utime.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>

#include "config.h"
#include "errors.h"
#include "database.hpp"

#ifdef HAVE_LCKPWDF
	extern "C" int lckpwdf(void);
	extern "C" int ulckpwdf(void);
#endif // HAVE_LCKPWDF

int iLockCount = 0;

template <class TEntry> CDatabase <TEntry>::CDatabase(const char *pszFile, bool fgLckpwdf, bool fgReadMode)
{
	fgReadOnly = fgReadMode;
	fgModified = false;
	fgLocked = false;
	fgUseLckpwdf = fgLckpwdf;
	
	snprintf(szDatabaseName, sizeof(szDatabaseName), pszFile);
	snprintf(szBackupName, sizeof(szBackupName), "%s-", pszFile);
	snprintf(szTempName, sizeof(szBackupName), "%s+", pszFile);	

	ppEntry = NULL;
	fpDatabase = NULL;
	
	iEntries = 0;
	iErrorCode = NO_ERROR;

	if (false == fgReadOnly)
		if (false == Lock()) {
			iErrorCode = LOCK_ERROR;
			return;
		}
	
	fpDatabase = fopen(pszFile, "r");
	if (NULL == fpDatabase) {
		iErrorCode = OPEN_ERROR;
		return;
	}
	
	TEntry *pEntry;
	
	do 
	{
		pEntry = new TEntry(fpDatabase);
		if (NULL == pEntry) {
			iErrorCode = MEMORY_ERROR;
			break;
		}
		if (PARSE_ERROR == pEntry->GetError()) {
			delete pEntry;
			continue;
		}
		if (MEMORY_ERROR == pEntry->GetError()) {
			iErrorCode = MEMORY_ERROR;					// No way - there is no memory
			break;											// Have to go...
		}
		if (NO_ERROR == pEntry->GetError()) {
			iEntries += 1;
			TEntry **ppNewEntry = (TEntry **)realloc(ppEntry, sizeof(TEntry *) * iEntries);
			if (NULL == ppNewEntry) {
				iErrorCode = MEMORY_ERROR;
				break;
			}
			ppEntry = ppNewEntry;
			ppEntry[iEntries - 1] = pEntry;
		}
	} while (EOF_ERROR != pEntry->GetError());
	delete pEntry;
}

template <class TEntry> CDatabase <TEntry>::~CDatabase()
{
	if (true == fgModified && false == fgReadOnly)
		UpdateDatabase();

	for (int i = 0; i < iEntries; i++)
		delete ppEntry[i];
	delete ppEntry;
	if (NULL != fpDatabase)
		fclose(fpDatabase);
	
	if (false == fgReadOnly)
		Unlock();
}

template <class TEntry> bool CDatabase <TEntry>::FindUser(char *pszName)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			return true;
	return false;
}

template <class TEntry> char * CDatabase <TEntry>::GetPassword(char *pszName)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			return ppEntry[i]->GetPassword();
	return NULL;
}

template <class TEntry> char * CDatabase <TEntry>::GetGecos(char *pszName)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			return ppEntry[i]->GetGecos();
	return NULL;
}

template <class TEntry> char * CDatabase <TEntry>::GetHome(char *pszName)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			return ppEntry[i]->GetHome();
	return NULL;
}

template <class TEntry> bool CDatabase <TEntry>::GetGroup(char *pszName, gid_t *piGroup)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName())) {
			*piGroup = ppEntry[i]->GetGroup();
			return true;
		}
	return false;
}

template <class TEntry> char * CDatabase <TEntry>::GetName(gid_t iGroup)
{
	for (int i = 0; i < iEntries; i++)
		if (iGroup == ppEntry[i]->GetGroup())
			return ppEntry[i]->GetName();
	return (char *)NULL;
}

template <class TEntry> bool CDatabase <TEntry>::ChangePassword(char *pszName,
																char *pszPassword,
																int iMin,
																int iMax,
																int iWarn,
																int iInact)
{
	if (true == fgReadOnly)
		return false;
	
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			if (true == ppEntry[i]->SetPassword(pszPassword, iMin, iMax, iWarn, iInact)) {
				fgModified = true;
				return true;
			} else
				break;
	return false;
}

template <class TEntry> bool CDatabase <TEntry>::ChangeGecos(char *pszName, char *pszGecos)
{
	if (true == fgReadOnly)
		return false;
	
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			if (true == ppEntry[i]->SetGecos(pszGecos)) {
				fgModified = true;
				return true;
			} else
				break;
	return false;
}

template <class TEntry> bool CDatabase <TEntry>::DeleteEntry(char *pszName)
{
	if (true == fgReadOnly)
		return false;
	
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName())) {
			delete ppEntry[i];
			for (int j = i + 1; j < iEntries; j++)
				ppEntry[j - 1] = ppEntry[j];
			iEntries -= 1;
			TEntry **ppNewEntry = (TEntry **)realloc(ppEntry, sizeof(TEntry *) * iEntries);
			if (NULL == ppNewEntry) {
				iErrorCode = MEMORY_ERROR;
				return false;
			}
			ppEntry = ppNewEntry;
			fgModified = true;
			return true;
		}
	iErrorCode = NOUSER_ERROR;	
	return false;
}


template <class TEntry> bool CDatabase <TEntry>::CreateBackup()
{
	struct stat strFileStat;
	if (0 != fstat(fileno(fpDatabase), &strFileStat)) {
		iErrorCode = STAT_ERROR;
		return false;
	}
	char *pBuffer = new char[BLOCK_SIZE];
	if (NULL == pBuffer) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	
	int iOldMask = umask(077);				// Create the backup file with perms
													// only to read and write by owner
	FILE *fpBackup = fopen(szBackupName, "w");
	umask(iOldMask);
	if (NULL == fpBackup) {
		iErrorCode = OPEN_ERROR;
		delete pBuffer;
		return false;
	}
	rewind(fpDatabase);	
	register unsigned int iRead;
	while ((iRead = fread(pBuffer, sizeof(char), BLOCK_SIZE, fpDatabase)))
		if (fwrite(pBuffer, sizeof(char), iRead, fpBackup) != iRead) {
			iErrorCode = WRITE_ERROR;
			fclose(fpBackup);
			delete pBuffer;
			return false;
		}
	delete pBuffer;
	if (fflush(fpBackup)) {
		iErrorCode = FLUSH_ERROR;
		fclose(fpBackup);
		return false;
	}
	fclose(fpBackup);
	
	struct utimbuf strFileTime = {strFileStat.st_atime, strFileStat.st_mtime};
	utime(szBackupName, &strFileTime);
	
	return true;
}

template <class TEntry> bool CDatabase <TEntry>::CreateTemp()
{
	struct stat strFileStat;
	if (0 != fstat(fileno(fpDatabase), &strFileStat)) {
		iErrorCode = STAT_ERROR;
		return false;
	}
	
	int iOldMask = umask(0777);
	FILE *fpTemp = fopen(szTempName, "w");
	umask(iOldMask);
	if (NULL == fpTemp) {
		iErrorCode = OPEN_ERROR;
		return false;
	}
#ifdef HAVE_FCHOWN
	if (fchown(fileno(fpTemp), strFileStat.st_uid, strFileStat.st_gid)) {
#else
	if (chown(szTempName, strFileStat.st_uid, strFileStat.st_gid)) {
#endif // HAVE_FCHOWN
		iErrorCode = CHOWN_ERROR;
		goto err_exit;		
	}

#ifdef HAVE_CHMOD
	if (fchmod(fileno(fpTemp), strFileStat.st_mode & 0644)) {
#else
	if (chmod(szTempName, strFileStat.st_mode & 0644)) {
#endif // HAVE_CHMOD
		iErrorCode = CHMOD_ERROR;
		goto err_exit;		
	}
		
	for (int i = 0; i < iEntries; i++)
		ppEntry[i]->PutEntry(fpTemp);
		  
	if (fflush(fpTemp)) {
		iErrorCode = FLUSH_ERROR;
		goto err_exit;		
	}
#ifdef HAVE_FSYNC
	if (fsync(fileno(fpTemp))) {
		iErrorCode = SYNC_ERROR;
		goto err_exit;
	}
#else
	sync();
#endif // HAVE_FSYNC		
	fclose(fpTemp);
	
	return true;
		
err_exit:		
	fclose(fpTemp);
	unlink(szTempName);		
	return false;
}

template <class TEntry> bool CDatabase <TEntry>::UpdateDatabase()
{
	if (CreateBackup())
		if (CreateTemp())
			if (rename(szTempName, szDatabaseName))
				iErrorCode = RENAME_ERROR;
			else
				return true;
	return false;
}

template <class TEntry> bool CDatabase <TEntry>::Lock()
{
#ifdef HAVE_LCKPWDF
	if (true == fgUseLckpwdf) {
		if (0 == iLockCount)
			if (-1 == lckpwdf())
				return false;
		if (false == LockNow()) {
			ulckpwdf();
			return false;
		}
		iLockCount += 1;
		return true;
	}
#endif // HAVE_LCKPWDF
	for (int i = 1; i < LOCK_TRIES; i++) {
		if (true == LockNow())
			return true;
		sleep(LOCK_SLEEP);
	}
	return LockNow();
}
		
template <class TEntry> bool CDatabase <TEntry>::LockNow()
{
	int iLength;
	
	if (true == fgLocked)
		return true;
	
	char szLock[MAX_PATH];
	char szFile[MAX_PATH];
	
	snprintf(szLock, sizeof(szLock), "%s.lock", szDatabaseName);
	snprintf(szFile, sizeof(szFile), "%s.%ld", szDatabaseName, (long)getpid());
	
	int iFile = open(szFile, O_CREAT | O_EXCL | O_WRONLY, 0600);
	if (-1 == iFile)
		return false;
	
	char szBuf[32];
	
	snprintf(szBuf, sizeof(szBuf), "%d", getpid());
	iLength = strlen(szBuf) + 1;
	if (iLength != (int)write(iFile, szBuf, iLength)) {
		close(iFile);
		unlink(szFile);
		return false;
	}
	close(iFile);
	
	if (0 == link(szFile, szLock))
		goto successful_exit;
	
	if (-1 == (iFile = open(szLock, O_RDWR))) {
		unlink(szFile);
		return false;
	}
	iLength = read(iFile, szBuf, sizeof(szBuf) - 1);
	close(iFile);
	if (0 >= iLength) {
		unlink(szFile);
		return false;
	}
	szBuf[iLength] = '\0';
	
	int iProcess;
	if ((0 == (iProcess = atol(szBuf))) ||
	    (0 == kill(iProcess, 0)) ||
	    (0 != unlink(szLock)) ||
	    (0 != link(szFile, szLock)))
		goto err_exit;

successful_exit:	
	struct stat strStat;
	if ((0 != stat(szFile, &strStat)) ||
		 (2 != strStat.st_nlink)) {
		unlink(szFile);			
		return false;
	}
	unlink(szFile);
	fgLocked = true;		
	return true;								// At last exit...
	
err_exit:
	unlink(szFile);
	return false;								// And once again exit :)...
}

template <class TEntry> void CDatabase <TEntry>::Unlock()
{
	if (true == fgLocked) {
		fgLocked = false;
		
		char szLockFile[MAX_PATH];
		
		snprintf(szLockFile, sizeof(szLockFile), "%s.lock", szDatabaseName);
		unlink(szLockFile);
		
#ifdef HAVE_LCKPWDF
		if (fgUseLckpwdf && (iLockCount > 0)) {
			iLockCount -= 1;
			if (0 == iLockCount)
				ulckpwdf();
		}
#endif // HAVE_LCKPWDF
	}
}
	
template <class TEntry> void CDatabase <TEntry>::CancelUpdate()
{
	fgModified = false;
}

CShadowDatabase::CShadowDatabase(bool fgReadOnly) : CDatabase <CShadowEntry>(SHADOW_FILE, true, fgReadOnly)
{
}
	
bool CShadowDatabase::AddEntry(char *pszName)
{
	if (true == fgReadOnly)
		return false;
	
	CShadowEntry *pEntry = new CShadowEntry(pszName);
	if (NULL == pEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	if (MEMORY_ERROR == pEntry->GetError()) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	iEntries += 1;
	CShadowEntry **ppNewEntry = (CShadowEntry **)realloc(ppEntry, sizeof(CShadowEntry *) * iEntries);
	if (NULL == ppNewEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	ppEntry = ppNewEntry;
	ppEntry[iEntries - 1] = pEntry;
	
	fgModified = true;
	
	return true;
}

CPasswdDatabase::CPasswdDatabase(bool fgReadOnly) : CDatabase <CPasswdEntry>(PASSWD_FILE, true, fgReadOnly)
{
}

bool CPasswdDatabase::AddEntry(char *pszName,
										 unsigned int iUserID,
										 unsigned int iGroupID,
										 char *pszGecos,
										 char *pszDirectory,
										 char *pszShell)
{
	if (true == fgReadOnly)
		return false;
	
	CPasswdEntry *pEntry = new CPasswdEntry(pszName, iUserID, iGroupID, pszGecos, pszDirectory, pszShell);
	if (NULL == pEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	if (MEMORY_ERROR == pEntry->GetError()) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	iEntries += 1;
	CPasswdEntry **ppNewEntry = (CPasswdEntry **)realloc(ppEntry, sizeof(CPasswdEntry *) * iEntries);
	if (NULL == ppNewEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	ppEntry = ppNewEntry;
	ppEntry[iEntries - 1] = pEntry;
	
	fgModified = true;
	
	return true;
}
	
int CPasswdDatabase::GetMaxUserID()
{
	int iResult = 0;
	
	for (int i = 0; i < iEntries; i++)
		if ((int)ppEntry[i]->GetUser() > iResult && ppEntry[i]->GetUser() < 0x7fff)
			iResult = ppEntry[i]->GetUser();
	  
	return iResult;
}
	
CGroupDatabase::CGroupDatabase(bool fgReadOnly) : CDatabase <CGroupEntry>(GROUP_FILE, true, fgReadOnly)
{
}

bool CGroupDatabase::AddEntry(char *pszName, unsigned int iGroupID, char *pszMembers)
{
	if (true == fgReadOnly)
		return false;
	
	CGroupEntry *pEntry = new CGroupEntry(pszName, iGroupID, pszMembers);
	if (NULL == pEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	if (MEMORY_ERROR == pEntry->GetError()) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	iEntries += 1;
	CGroupEntry **ppNewEntry = (CGroupEntry **)realloc(ppEntry, sizeof(CGroupEntry *) * iEntries);
	if (NULL == ppNewEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	ppEntry = ppNewEntry;
	ppEntry[iEntries - 1] = pEntry;
	
	fgModified = true;
	
	return true;
}
	
int CGroupDatabase::GetMaxGroupID()
{
	int iResult = 0;
	
	for (int i = 0; i < iEntries; i++)
		if ((int)ppEntry[i]->GetGroup() > iResult && ppEntry[i]->GetGroup() < 0x7fff)
			iResult = ppEntry[i]->GetGroup();
	  
	return iResult;
}

CGshadowDatabase::CGshadowDatabase(bool fgReadOnly) : CDatabase <CGshadowEntry>(GSHADOW_FILE, true, fgReadOnly)
{
}

bool CGshadowDatabase::AddEntry(char *pszName, char *pszAdmins, char *pszMembers)
{
	if (true == fgReadOnly)
		return false;
	
	CGshadowEntry *pEntry = new CGshadowEntry(pszName, pszAdmins, pszMembers);
	if (NULL == pEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	if (MEMORY_ERROR == pEntry->GetError()) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	iEntries += 1;
	CGshadowEntry **ppNewEntry = (CGshadowEntry **)realloc(ppEntry, sizeof(CGshadowEntry *) * iEntries);
	if (NULL == ppNewEntry) {
		iErrorCode = MEMORY_ERROR;
		return false;
	}
	ppEntry = ppNewEntry;
	ppEntry[iEntries - 1] = pEntry;
	
	fgModified = true;
	
	return true;
}
	
bool CGshadowDatabase::IsAdministrator(char *pszName, char *pszLogin)
{
	for (int i = 0; i < iEntries; i++)
		if (0 == strcmp(pszName, ppEntry[i]->GetName()))
			if (ppEntry[i]->IsAdmin(pszLogin))
				return true;
	return false;
}
