/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __RSA_HPP__
#define __RSA_HPP__

#include <stdio.h>

#include "globals.hpp"
#include "obn.hpp"
#include "primes.hpp"
#include "public.hpp"
#include "der.hpp"

#define RSA_PUBLIC_KEY				0x10001

#define RSA_PRIVATE_KEY_VERSION			0

#define BEGIN_RSA_PRIVATE_KEY			"-----BEGIN RSA PRIVATE KEY-----"
#define END_RSA_PRIVATE_KEY				"-----END RSA PRIVATE KEY-----"
#define BEGIN_RSA_PUBLIC_KEY			"-----BEGIN RSA PUBLIC KEY-----"
#define END_RSA_PUBLIC_KEY				"-----END RSA PUBLIC KEY-----"

class CRSAKey : public CAsymmetricKey
{
	public:
		CRSAKey();
		CRSAKey(Word, Word = RSA_PUBLIC_KEY);
		CRSAKey(const CRSAKey &);
		CRSAKey(const CBigNumber &, const CBigNumber &);
		CRSAKey(const CBigNumber &, const CBigNumber &, const CProbablePrime &, const CProbablePrime &);
	
	public:
		Word GetModulusSize() { return wModulusSize; };
		CBigNumber & GetModulus() { return N; };
		CBigNumber & GetPublic() { return PUB; };
		CBigNumber & GetSecret() { return PRI; };
		CProbablePrime & GetP() { return P; };
		CProbablePrime & GetQ() { return Q; };
		CBigNumber & GetP1() { return P1; };
		CBigNumber & GetQ1() { return Q1; };
		CBigNumber & GetDP() { return DP; };
		CBigNumber & GetDQ() { return DQ; };
		CBigNumber & GetU() { return U; };
	
	public:
		bool EncryptOnlyFlag() { return fgEncryptOnly; };
		bool HoldKeyFlag() { return fgHoldKey; };
		virtual void WritePrivateKey(int, bool = true);
		virtual void WritePublicKey(int, bool = true);
		virtual void ReadPrivateKey(int, bool = true);
		virtual void ReadPublicKey(int, bool = true);
		void Dump();
		Word Check();
	
	private:
		void GenerateKeys();
	
	private:
		CProbablePrime P, Q;
		CBigNumber N, P1, Q1, PHI, PUB, U, DP, DQ, PRI;
		Word wModulusSize;
		bool fgEncryptOnly;
		bool fgHoldKey;
};

class CRSABlock : public CAsymmetricBlock
{
	public:
		CRSABlock(const CRSAKey &);
		CRSABlock(const CRSAKey &, void *, Word);
		CRSABlock(const CRSAKey &, const CBigNumber &);
	
	public:
		virtual void Encrypt();
		virtual void Decrypt();
		virtual void Write(int);							// ciphertext
		virtual void Read(int);								// ciphertext
		virtual void SetData(Byte *, Word);					// plaintext
		virtual Byte *GetData();							// plaintext
		virtual Word GetDataSize();							// plaintext

	private:
		void ModularRoot(CBigNumber &, CBigNumber &, CBigNumber &, CBigNumber &, CBigNumber &, CBigNumber &);

	public:
		void Dump();

	private:
		CRSAKey cKey;
		CBigNumber cData;
};

#endif // __RSA_HPP__
