/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "primes.hpp"
#include "smallprimes.hpp"

void (*pfnCallback1)() = NULL;
void (*pfnCallback2)() = NULL;
void (*pfnCallback3)() = NULL;

bool CProbablePrime::IsSmallPrime()
{
	for (unsigned i = 0; i < (sizeof(iSmallPrimes) / sizeof(int)); i++)
		if ((Word)iSmallPrimes[i] == pwBigNumber[0])
			return true;
	return false;
}

bool CProbablePrime::SmallPrimeDivisibilityTest()
{
	for (unsigned i = 0; i < sizeof(iSmallPrimes) / sizeof(int); i++)
		if (*this % iSmallPrimes[i] == 0)
			return false;

	if (NULL != pfnCallback1)
		pfnCallback1();

	return true;
}

// Perform Rabin - Miller's probabilistic test whether the given number is prime
bool CProbablePrime::RabinMillerIsPrime()
{
	CBigNumber w = *this - 1;
	
	Word v = 0;
	while (w % 2 == 0) {
		v += 1;
		w >>= 1;
	}
	
	for (Word j = 1; j <= 8; j++) {
		CBigNumber a(iSmallPrimes[(Word)((float)rand() * (sizeof(iSmallPrimes) / sizeof(int)) / RAND_MAX)]);
		a.Mod(*this);
		a += 1;
		CBigNumber b = ModExp(a, w, *this);
		if ((b != 1) && (b != *this - 1)) {
			Word i = 1;
			while (1) {
				if (i == v)
					return false;
				b.Sqr();
				b.Mod(*this);
				if (b == *this - 1)
					break;
				if (b == 1)
					return false;
				i += 1;
			}
		}
	}
	
	if (NULL != pfnCallback2)
		pfnCallback2();

	return true;
}
 
// Test based on Fermat's theorem for prime number
bool CProbablePrime::FermatIsPrime()
{
	CBigNumber a(2);
	CBigNumber b = *this;
	CBigNumber c = *this;
	
	c -= 1;
	
	CBigNumber d = ModExp(a, c, b);
	
	if (d == 1) {
		if (NULL != pfnCallback3)
			pfnCallback3();
		return true;
	}

	return false;
}

bool CProbablePrime::IsPrime()
{
	if (pwBigNumber[0] < (Word)iSmallPrimes[sizeof(iSmallPrimes) / sizeof(int) - 1])
		return IsSmallPrime();
	return SmallPrimeDivisibilityTest() &&
		   RabinMillerIsPrime() &&
		   FermatIsPrime();
}
