/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
 
#include "strings.hpp"

#ifdef WIN32
#define snprintf _snprintf
#endif

char **AddString(char **ppszList, char *pszString, int iLength, int *piStrings)
{
	char **ppszResult;
	char *pszCopy = pszString;
	
	if (NULL != pszString) {
		pszCopy = new char[iLength + 1];
		if (NULL != pszCopy) {
			memcpy(pszCopy, pszString, iLength);
			pszCopy[iLength] = '\0';
		}
	}
	if (NULL != pszCopy || NULL == pszString) {
		ppszResult = (char **)realloc(ppszList, sizeof(char *) * (*piStrings + 1));
		if (NULL != ppszResult) {
			ppszList = ppszResult;
			ppszResult[*piStrings] = pszCopy;
			*piStrings += 1;
		}
	}
	return ppszList;
}

char **Tokenize(char *pszString, char *pszDelimiters)
{
	char **ppszResult = NULL;
	int iStrings = 0;

	char *pszStartOfToken;
	char *pszEndOfToken;
	for (pszStartOfToken = pszEndOfToken = pszString; *pszString != '\0'; pszString++)
		for (int j = 0; pszDelimiters[j] != '\0'; j++)
			if (*pszString == pszDelimiters[j]) {
				pszEndOfToken = pszString;
				ppszResult = AddString(ppszResult, pszStartOfToken, pszEndOfToken - pszStartOfToken, &iStrings);
				pszStartOfToken = pszString + 1;
			}
	pszEndOfToken = pszString;
	if (('\0' != *pszStartOfToken) && (0 != (pszEndOfToken - pszStartOfToken)))
		ppszResult = AddString(ppszResult, pszStartOfToken, pszEndOfToken - pszStartOfToken, &iStrings);
	ppszResult = AddString(ppszResult, NULL, 0, &iStrings);
	return ppszResult;
}

char *Concatenate(char **ppszStrings, char cDelimiter)
{
	char szDelimiter[] = { cDelimiter, '\0' };
	int j = 0;
	for (int i = 0; ppszStrings[i] != NULL; i++)
		j += (strlen(ppszStrings[i]) + 1);
	char *pszResult = new char[j];
	if (NULL != pszResult) {
		pszResult[0] = '\0';
		for (int i = 0; ppszStrings[i] != NULL; i++) {
			if (0 != i)
				strcat(pszResult, szDelimiter);
			strcat(pszResult, ppszStrings[i]);
		}
	}
	return pszResult;
}

char *ReplaceString(char *pszStorage, char *pszHaystack, char *pszNeedle, unsigned nMax)
{
	char *pszTemp = strstr(pszStorage, pszHaystack);
	if ((NULL != pszTemp) && (strlen(pszStorage) - strlen(pszHaystack) + strlen(pszNeedle) < nMax)) {
		memmove(pszTemp + strlen(pszNeedle), pszTemp + strlen(pszHaystack), strlen(pszTemp + strlen(pszHaystack)) + 1);
		memcpy(pszTemp, pszNeedle, strlen(pszNeedle));
	}

	return pszStorage;
}

char *TrimLeft(char *pszString, char *pszChars)
{
	int k = strspn(pszString, pszChars);
	int l = strlen(pszString);
	int i;
	if (0 != k) {
		for (i = k; i < l; i++)
			pszString[i - k] = pszString[i];
		pszString[i - k] = '\0';
	}
	return pszString;
}

char *TrimRight(char *pszString, char *pszChars)
{
	int i = strlen(pszString) - 1;
	for (; i >= 0; i--)
		if (NULL == strchr(pszChars, pszString[i]))
			break;
	pszString[i + 1] = '\0';
	return pszString;
}

char *TrimBoth(char *pszString, char *pszChars)
{
	return TrimLeft(TrimRight(pszString, pszChars), pszChars);
}

CStringArray::~CStringArray()
{
	for (int i = 0; i < iStrings; i++)
		delete ppszStrings[i];
	if (NULL != ppszStrings)
		delete ppszStrings;
}

bool CStringArray::FindString(char *pszString)
{
	for (int i = 0; i < iStrings; i++)
		if (0 == strcmp(ppszStrings[i], pszString))
			return true;
	return false;
}

bool CStringArray::AddString(char *pszString, int iLength)
{
	if (-1 == iLength)
		iLength = strlen(pszString);
	char *pszNewString = new char[iLength + 1];
	if (NULL == pszNewString)
		return false;
	memcpy(pszNewString, pszString, iLength);
	pszNewString[iLength] = '\0';
	char **ppszNewStrings = (char **)realloc(ppszStrings, sizeof(char *) * (iStrings + 1));
	if (NULL == ppszNewStrings) {
		delete pszNewString;
		return false;
	}
	
	ppszStrings = ppszNewStrings;
	
	ppszStrings[iStrings] = pszNewString;
	
	iStrings += 1;
	
	return true;
}

char *CStringArray::GetString(int iNum)
{
	if (iNum >= 0 && iNum < iStrings)
		return ppszStrings[iNum];
	
	return NULL;
}

char *CStringArray::GetFirstString()
{
	if (0 == iStrings)
		return NULL;
	
	iCurrent = 0;
	
	return ppszStrings[0];
}

char *CStringArray::GetNextString()
{
	if (iCurrent + 1 >= iStrings)
		return NULL;
	
	iCurrent += 1;
	
	return ppszStrings[iCurrent];
}

char *CStringArray::Concatenate(char cDelimiter)
{
	return ::Concatenate(ppszStrings, cDelimiter);
}

void CStringArray::DeleteString()
{
	if (iCurrent >= iStrings)
		return;

	delete ppszStrings[iCurrent];
	for (int i = iCurrent + 1; i < iStrings; i++)
		ppszStrings[i - 1] = ppszStrings[i];
}

CTokenizedString::CTokenizedString(char *pszString, char *pszDelimiters) : CStringArray()
{
	char *pszStartOfToken;
	char *pszEndOfToken;
	for (pszStartOfToken = pszEndOfToken = pszString; *pszString != '\0'; pszString++)
		for (int j = 0; pszDelimiters[j] != '\0'; j++)
			if (*pszString == pszDelimiters[j]) {
				pszEndOfToken = pszString;
				AddString(pszStartOfToken, pszEndOfToken - pszStartOfToken);
				pszStartOfToken = pszString + 1;
			}
	pszEndOfToken = pszString;
	if (('\0' != *pszStartOfToken) && (0 != (pszEndOfToken - pszStartOfToken)))
		AddString(pszStartOfToken, pszEndOfToken - pszStartOfToken);
}
