/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "inetaddr.hpp"

char *GetDDN(char *pszHost) 
{
	static char szResult[128];
	bool fgError = false;
	unsigned long l = GetHostByName(pszHost, &fgError);
	if (true == fgError)
		return "";
	snprintf(szResult, 
				sizeof(szResult), 
				"%d.%d.%d.%d", 
				((unsigned char *)&l)[3],
				((unsigned char *)&l)[2],
				((unsigned char *)&l)[1],
				((unsigned char *)&l)[0]);
	return szResult;
}

char *GetDDN(struct sockaddr_in *saptr) 
{
	static char szResult[256];
	char *bp, *ap;
	int l;
	struct sockaddr sa;
	
// Check for null/zero family
	if (saptr == NULL)
		return "NULLADDR";
	if (saptr->sin_family == 0)
		return "0";

	switch (saptr->sin_family) {
//		case AF_UNIX:
//			if (saptr->sunix.sun_path[0] != '\0')
//				snprintf(str, 256, "[UNIX: %.64s]", saptr->sunix.sun_path);
//			else
//				snprintf(str, 256, "[UNIX: localhost]");
//			return str;
		case AF_INET:
			return inet_ntoa(saptr->sin_addr);
	}

// Unknown family -- just dump bytes
	memcpy(&sa, saptr, sizeof(sa));
	snprintf(szResult, 256, "Family %d: ", sa.sa_family);
	bp = &szResult[strlen(szResult)];
	ap = sa.sa_data;
	for (l = sizeof(sa.sa_data); --l >= 0; ) {
		sprintf(bp, "%02x:", *ap++ & 0377);
		bp += 3;
	}
	*--bp = '\0';
	return szResult;
}

unsigned long GetHostByName(char *pszName, bool *pfgError)
{
	struct hostent *pstrHost;

	if (NULL == (pstrHost = gethostbyname(pszName))) {
		*pfgError = true;
		return 0;
	}
	*pfgError = false;
	return ntohl(*((unsigned long *)(pstrHost->h_addr)));
}

bool ParseDDN(char *pszHost,
				  unsigned long *pulTargetHost,
				  unsigned long *pulTargetMask,
				  bool *pfgInvert)
{
	bool fgError = false;
	
	char *pszMask;
	
	char *pszTmp = new char[strlen(pszHost) + 1];
	if (NULL == pszTmp)
		return false;
	
	pszHost = strcpy(pszTmp, pszHost);
	if (NULL != (pszMask = strchr(pszHost, '/'))) {
		*pszMask = '\0';
		pszMask += 1;
	}
	*pfgInvert = false;
	if ('!' == *pszHost) {
		*pfgInvert = true;
		pszHost += 1;
	}
	*pulTargetHost = GetHostByName(pszHost, &fgError);
	if (true == fgError) {
		delete pszTmp;
		return false;
	}
	*pulTargetMask = 0xFFFFFFFF;
	if (NULL != pszMask) {
		char *p;
		unsigned long l = strtol(pszMask, &p, 10);
		if (('\0' == *p) && (l >= 0) && (l <= 32)) {
			*pulTargetMask = 0;
			for (unsigned long q = 0; q < l; q++) {
				*pulTargetMask >>= 1;
				*pulTargetMask |= 0x80000000;
			}
		} else
			*pulTargetMask = GetHostByName(pszMask, &fgError);
	}
	
	delete pszTmp;
	return !fgError;
}
