/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <ctype.h>

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#else
#include "getpass.hpp"
#endif

#include "passwddio.hpp"
#include "globals.hpp"
#include "strings.hpp"
#include "getarg.h"
#include "conf.hpp"
#include "rsa.hpp"

// Function prototypes
int option_value(char *, bool *, int, int, char *);

bool fgEncryptedPasswords;

char szConfigFile[256] = DEFAULT_CLIENTCONFFILE;
char szShell[256] = "";
char szDirectory[256] = "";
char szFullName[256] = "";
char szRoomNumber[256] = "";
char szWorkPhone[256] = "";
char szHomePhone[256] = "";
char szOther[256] = "";

void usage()
{
	printf("Usage: useradd <target> [<user>] [-c <file>]\n"
	       "       [-s <shell>] [-d <directory>] [-u <user id>]\n"
	       "       [-g <group id>] [-f <full name>] [-r <room number>]\n"
	       "       [-w <work phone>] [-h <home phone>] [-o <other>]\n"
	       "Where:\n"
	       "  <target>         The user whose login you want to add.\n"
	       "  <login>          Your login if you are master user for the given target.\n"
	       "Options:\n"
	       "  -c <file>        Use alternate config file instead the default\n"
	       "                   " DEFAULT_CLIENTCONFFILE " system wide config file.\n"
	       "  -s <shell>       User's shell.\n"
	       "  -d <directory>   User's home directory.\n"
	       "  -u <user id>     User ID.\n"
	       "  -g <group id>    Group ID.\n"
	       "  -f <full name>   User fullname.\n"
	       "  -r <room number> Office number.\n"
	       "  -w <work phone>  Office extension.\n"
	       "  -h <home phone>  Home phone.\n"
	       "  -o <other>       Additional info.\n"
	       "Written by: Alexander Feldman <alex@varna.net>\n\n");
}

int main(int argc, char **argv)
{
	CStringArray cFailed;

	static char szServers[512];
	
	static char szPassword[128];
	
	struct sockaddr_in client;
	struct hostent *pHost;
	
	char *pszLogin = (char *)NULL;
	char *pszTarget = (char *)NULL;
	char *pszServer;
	char *pszPassword;
	int iUserID = -1;
	int iGroupID = -1;
	
	int sockfd;
	int passed = -1;
	
	int iServerPort;
	
	int iResult = 0;
	
	char *pszConfig = getenv("PASSWDCCONF");
	bool fgError = false;

	int w;
	while (-1 != (w = getopt(argc, argv, "c:s:d:u:g:f:r:w:h:o:")))
		switch (w) {
			case 'c':
				strncpy(szConfigFile, optarg, sizeof(szConfigFile));
			break;
			case 's':
				strncpy(szShell, optarg, sizeof(szShell));
			break;
			case 'd':
				strncpy(szDirectory, optarg, sizeof(szDirectory));
			break;
			case 'u':
				iUserID = option_value(optarg, &fgError, 1, 99999, "-u");
			break;
			case 'g':
				iGroupID = option_value(optarg, &fgError, 1, 99999, "-g");
			break;
			case 'f':
				strncpy(szFullName, optarg, sizeof(szFullName));
			break;
			case 'r':
				strncpy(szRoomNumber, optarg, sizeof(szRoomNumber));
			break;
			case 'w':
				strncpy(szWorkPhone, optarg, sizeof(szWorkPhone));
			break;
			case 'h':
				strncpy(szHomePhone, optarg, sizeof(szHomePhone));
			break;
			case 'o':
				strncpy(szOther, optarg, sizeof(szOther));
			break;
			default:
				usage();
				return 0;
			break;
		}
	argc -= optind;
	argv += optind;

	if ((argc < 1) || (argc > 2) || (true == fgError)) {
		usage();
		return 0;
	}
	if (argc >= 1)
		pszTarget = argv[0];
	if (argc == 2)
		pszLogin = argv[1];

#ifdef WIN32
	GetWindowsDirectory(szConfigFile, sizeof(szConfigFile));
	strncat(szConfigFile, SEPARATOR DEFAULT_CLIENTCONFFILE, sizeof(szConfigFile));
#endif
	if (NULL == pszConfig)
		pszConfig = szConfigFile;
	
	char *pszMessage = NULL;	
	CConfigurationFile *pConfigFile = new CConfigurationFile(pszConfig, &pszMessage);
	if (NULL == pConfigFile) {
		fprintf(stderr, "Error allocating memory...\n");
		return 1;
	}
	if (NULL != pszMessage) {
		fprintf(stderr, "%s\n", pszMessage);
		delete pConfigFile;
		return 1;
	}

#ifdef WIN32
	WSADATA WSAData;
	int iStatus;
	if (0 != (iStatus = WSAStartup(MAKEWORD(2, 2), &WSAData))) {
		printf("Socket init error: %s\n%s.\n", WSAData.szDescription, WSAData.szSystemStatus);
		return 1;
	}
#endif

	iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
	memzero(szServers);
	strncpy(szServers, pConfigFile->GetString("Global", "targets", "localhost"), sizeof(szServers) - 1);

#ifndef WIN32
	if (NULL == pszLogin)
//		pszLogin = getlogin();				// This returns NULL in some glibc
		pszLogin = getenv("LOGNAME");		// Thanks to Mark Siegler <tech@netalia.com>
	
	if (NULL == pszLogin) {
		fprintf(stderr, "Error getting login from the environment...\n");
		return 1;
	}
	if (NULL == pszTarget)
		pszTarget = pszLogin;
#else
	if (NULL == pszLogin)
		pszLogin = pszTarget;
#endif
	
	if (('\0' != *szOther) && (0 == strcmp(pszTarget, pszLogin))) {
		fprintf(stderr, "Only the superuser has the right to change the additional info...\n");
		delete pConfigFile;
		return 1;
	}
	
	char *pszGecos = new char[strlen(szFullName) +
							  strlen(szRoomNumber) +
							  strlen(szWorkPhone) +
							  strlen(szHomePhone) +
							  strlen(szOther) + 5];
	if (NULL == pszGecos) {
		delete pConfigFile;		
		fprintf(stderr, "Error allocating memory...\n");
		return 1;
	}
	strcpy(pszGecos, szFullName);
	strcat(pszGecos, ",");
	strcat(pszGecos, szRoomNumber);
	strcat(pszGecos, ",");
	strcat(pszGecos, szWorkPhone);
	strcat(pszGecos, ",");
	strcat(pszGecos, szHomePhone);
	strcat(pszGecos, ",");
	strcat(pszGecos, szOther);
	int j = 0;
	for (int i = 0; pszGecos[i] != '\0'; i++) {
		if (',' == pszGecos[i])
			j += 1;
		if (j > 4 ||
			iscntrl(pszGecos[i]) ||
			';' == pszGecos[i] || 
			'=' == pszGecos[i]) {
			delete pConfigFile;
			delete pszGecos;
			fprintf(stderr, "Invalid characters in gecos field...\n");
			return 1;
		}
	}
	
	pszPassword = getpass("Password:");
	strcpy(szPassword, pszPassword);
	strzero(pszPassword);

	pszServer = strtok(szServers, ";,");
	do {
		bool fgPassed = false;
		
		pszServer += strspn(pszServer, " \t");			
		printf("Adding %s on %s\n", pszTarget, pszServer);

		if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
			printf("Socket error: %s\n", strerror(errno));
			cFailed.AddString(pszServer);
			continue;
		}

		pHost = gethostbyname(pszServer);
		if (NULL == pHost) {
			printf("Resolving error: %s\n", pszServer);
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}
		memset(&client, 0, sizeof(client));
		client.sin_family = AF_INET;
		client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
		client.sin_port = htons(iServerPort);
		if (connect(sockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
			printf("Connect error: %s\n", strerror(errno));
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}

		unsigned char c = '\0';
		read_byte(sockfd, &c);
		if (WELCOME != c) {
			printf("Invalid response from server.\n");
			close(sockfd);
			cFailed.AddString(pszServer);			
			continue;
		}
		if (1 != read_integer(sockfd, (int *)&fgEncryptedPasswords)) {
			fprintf(stderr, "Recieve data error.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}

		if (true == fgEncryptedPasswords) {
			char szPublicKey[256];
			snprintf(szPublicKey,
					 sizeof(szPublicKey),
					 "%s/%s_%d.pub",
					 pConfigFile->GetString("Security", "key_directory", "/var/spool/passwdc"),
					 pszServer,
					 iServerPort);
			int i = recv_remote_key(sockfd, szPublicKey);
			if (1 == i)
				fprintf(stderr, "Error allocating memory.\n");
			if (2 == i)
				fprintf(stderr, "Read public key error.\n");
			if (3 == i)
				fprintf(stderr, "Read public key error.\n");
			if (4 == i)
				fprintf(stderr, "Public key mismatch! Please remove manually the file '%s' if this is not an attack.\n", szPublicKey);
			if (5 == i)
				fprintf(stderr, "Write public key error.\n");
			if (0 != i) {
				write_byte(sockfd, FAILURE);
				close(sockfd);
				cFailed.AddString(pszServer);
				continue;
			}
			write_byte(sockfd, OK);
		}

		if (((int)strlen(pszLogin) + 1 != write_string(sockfd, pszLogin)) ||
			(false == write_password(sockfd, szPassword))) {
			fprintf(stderr, "Write data error.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}

		read_byte(sockfd, &c);			
		switch (c) {
			case GO_AHEAD:
				write_byte(sockfd, ACTION_ADDUSER);
				if (((int)strlen(pszTarget) + 1 != write_string(sockfd, pszTarget)) ||
					((int)strlen(pszGecos) + 1 != write_string(sockfd, pszGecos)) ||
					((int)strlen(szDirectory) + 1 != write_string(sockfd, szDirectory)) ||
					((int)strlen(szShell) + 1 != write_string(sockfd, szShell)) ||
					(false == write_integer(sockfd, iUserID)) ||
					(false == write_integer(sockfd, iGroupID))) {
					printf("Write error.\n");
					close(sockfd);
					cFailed.AddString(pszServer);					
					continue;
				}
				read_byte(sockfd, &c);
				switch (c) {
					case OK:
						printf("User successfully added.\n");
						fgPassed = true;
					break;
					case INVALID_USER:
						printf("Invalid target specified.\n");
					break;
					case FAILURE:
						printf("User is not added. Server returned error.\n");
					break;
					case NO_RIGHTS:
						printf("Access denied.\n");
					break;
					case BAD_SHELL:
						printf("Invalid shell.\n");
					break;
					case ERROR_MAKEDIR:
						printf("Error making directory.\n");
					break;
					default:
						printf("Invalid response from server.\n");
				}
			break;
			case LOGIN_FAILED:
				printf("Incorrect login or password for %s on %s.\n", pszLogin, pszServer);				
			break;
			default:
				printf("Invalid response from server.\n");
		}
		if (true == fgPassed) {
			if (-1 != passed) {
				write_string(passed, "");
				close(passed);
			}
			passed = sockfd;
		} else {
			cFailed.AddString(pszServer);
			close(sockfd);
		}
	} while (NULL != (pszServer = strtok(NULL, ";,")));

	if (-1 != passed) {
		char *pszFailed;
		if ((pszFailed = cFailed.GetFirstString()))
			do {
				write_string(passed, pszFailed);
			} while ((pszFailed = cFailed.GetNextString()));
		write_string(passed, "");
		close(passed);
	}
		
	delete pszGecos;

	strzero(szPassword);

#ifdef WIN32
	WSACleanup();
#endif

	delete pConfigFile;

	destroy_remote_key();

	return iResult;
}

int option_value(char *pszValue, bool *pfgError, int iMin, int iMax, char *pszInfo)
{
	int iResult = 0;
	
	if ((NULL == pszValue) || ('\0' == *pszValue)) {
		fprintf(stderr, "Invalid value for option: %s...\n", pszInfo);
		*pfgError = true;
		return iResult;
	}
	char *pszInvalid;
	iResult = strtol(pszValue, &pszInvalid, 10);
	if ('\0' != *pszInvalid) {
		fprintf(stderr, "Invalid value for option: %s...\n", pszInfo);
		*pfgError = true;
	}
	if (iResult < iMin || iResult > iMax) {
		fprintf(stderr, "Out of range value for option: %s...\n", pszInfo);
		*pfgError = true;
	}
	
	return iResult;
}
