/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <glob.h>

#include "config.h"
#include "globals.hpp"
#include "allow.hpp"
#include "database.hpp"
#include "obscure.hpp"
#include "server.hpp"
#include "pwdb.hpp"
#include "sockio.hpp"
#include "salt.hpp"
#include "conf.hpp"
#include "group.hpp"
#include "keeper.hpp"

#ifdef HAVE_LIBCRYPT
	extern "C" char *crypt(const char *, const char *);
#endif
#ifndef HAVE_DEFINED_GETUSERSHELL
	extern "C" char *getusershell(void);
#endif
#ifndef HAVE_DEFINED_SETUSERSHELL
	extern "C" void setusershell(void);
#endif
#ifndef HAVE_DEFINED_ENDUSERSHELL
	extern "C" void endusershell(void);
#endif

extern CConfigurationFile *pConfigFile;

void server(int sockfd, char *pszClient)
{
	char szLogin[128];
	char szPassword[128];

	dprintf("Client connection.\n");

	write_byte(sockfd, WELCOME);

	if (fgEncryptedPasswords) {
		if (false == write_integer(sockfd, true)) {
			syslog(LOG_ERR, "send error: %s", strerror(errno));
			return;
		}
		if (1 == send_local_key(sockfd)) {
			syslog(LOG_ERR, "write public key failes");
			return;
		}
		unsigned char cValidKey;
		if (true != read_byte(sockfd, &cValidKey)) {
			syslog(LOG_ERR, "error receiving the key status from the client '%s'", pszClient);
			return;
		}
		if (OK != cValidKey) {
			syslog(LOG_ERR, "invalid public key sent to the clent '%s'", pszClient);
			return;
		}
	} else {
		if (false == write_integer(sockfd, false)) {
			syslog(LOG_ERR, "send error: %s", strerror(errno));
			return;
		}
	}

	if (-1 == read_string(sockfd, szLogin, sizeof(szLogin)))
		write_byte(sockfd, TIMEOUT);

	uid_t i = getuid();
	if (0 != setuid(0)) {
		syslog(LOG_ERR, "passwdd needs root privileges to change passwords setuid error: %s", strerror(errno));
		return;
	}

	int q;
	if (fgEncryptedPasswords)
		q = read_password(sockfd, szPassword, sizeof(szPassword));
	else
		q = read_string(sockfd, szPassword, sizeof(szPassword));

	if (-1 == q) {
		syslog(LOG_ERR, "error reading the password of user: %s", szLogin);
		return;
	}

	dprintf(" Login: %s.\n", szLogin);
	dprintf(" Password: %s.\n", szPassword);

	if (false == check_login(szLogin, szPassword)) {
		setuid(i);
		write_byte(sockfd, LOGIN_FAILED);
		return;
	}
	write_byte(sockfd, GO_AHEAD);
	
	unsigned char cAction = '\0';
	if (true != read_byte(sockfd, &cAction)) {
		setuid(i);
		write_byte(sockfd, TIMEOUT);
		return;
	}

	CStringArray cFailedServers;
	
	switch (cAction) {
// Info services
		case ACTION_GETGROUP:
			action_get_group(sockfd, szLogin);
		break;
// This service is used for remote authentication
		case ACTION_NOTHING:
			syslog(LOG_INFO, "user %s did nothing", szLogin);
			write_byte(sockfd, GOOD_BYE);
		break;
// Group services
		case ACTION_CHANGEGROUPPASSWORD:
			action_change_group_password(sockfd, szLogin, szPassword);
		break;
		case ACTION_ADDGROUP:
			action_add_group(sockfd, szLogin, szPassword);
		break;
		case ACTION_DELETEGROUP:
			action_delete_group(sockfd, szLogin, szPassword);
		break;
// Personal services
		case ACTION_CHANGEPASSWORD:
			action_change_password(sockfd, szLogin, szPassword);
		break;
		case ACTION_ADDUSER:
			action_add_user(sockfd, szLogin, szPassword);
		break;
		case ACTION_DELETEUSER:
			action_delete_user(sockfd, szLogin, szPassword);
		break;
		case ACTION_CHFN:
			action_chfn(sockfd, szLogin, szPassword);
		break;
		default:
			write_byte(sockfd, INVALID_ACTION);
		break;
	}
	memset(szPassword, 0, sizeof(szPassword));
	setuid(i);
}

bool get_failed_servers(int sockfd, CStringArray *pServers)
{
	bool fgResult = false;
	char szServer[256];
	do {
		if (-1 == read_string(sockfd, szServer, sizeof(szServer))) {
			syslog(LOG_ERR, "error receiving failed transactions list");
			return false;
		}
		if ('\0' != szServer[0]) {
			pServers->AddString(szServer);
			fgResult = true;
		}
	} while ('\0' != szServer[0]);
	return fgResult;
}

bool check_gecos_field(char *pszString) 
{
	for (int q = 0; '\0' != pszString[q]; q++)
		if (pszString[q] != ' ' &&
			(!isalnum(pszString[q]) ||
			 ',' == pszString[q] ||
			 ';' == pszString[q] ||
			 '=' == pszString[q]))
			return false;
	return true;
}

bool change_password(char *pszLogin,
					 char *pszNewPassword,
					 int iMin,
					 int iMax,
					 int iWarn,
					 int iInact)
{
	char *pszCryptedPassword = crypt(pszNewPassword, crypt_make_salt());
	
	memset(pszNewPassword, 0, strlen(pszNewPassword));		// Clear password

#ifdef SHADOWPWD
	CShadowDatabase cShadowDatabase(false);
	if (NO_ERROR != cShadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading shadow file");
		return false;
	}
	if (false == cShadowDatabase.ChangePassword(pszLogin,
												pszCryptedPassword,
												iMin,
												iMax,
												iWarn,
												iInact)) {
		syslog(LOG_ERR, "error modifying shadow file");
		return false;
	}
#else
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	if (false == cPasswdDatabase.ChangePassword(pszLogin,
												pszCryptedPassword,
												iMin,
												iMax,
												iWarn,
												iInact)) {
		syslog(LOG_ERR, "error modifying passwd file");
		return false;
	}
#endif
	
	return true;
}

bool change_gecos(char *pszLogin,
				  char *pszFullName,
				  char *pszRoomNumber,
				  char *pszWorkPhone,
				  char *pszHomePhone,
				  char *pszOther)
{
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	char *pszOldGecos = cPasswdDatabase.GetGecos(pszLogin);
	if (NULL == pszOldGecos) {
		syslog(LOG_ERR, "error getting old gecos field");
		return false;
	}
	CTokenizedString cOldGecos(pszOldGecos, ",");
	if (*pszFullName == '\0' && cOldGecos.GetString(0))
		pszFullName = cOldGecos.GetString(0);
	if (*pszRoomNumber == '\0' && cOldGecos.GetString(1))
		pszRoomNumber = cOldGecos.GetString(1);
	if (*pszWorkPhone == '\0' && cOldGecos.GetString(2))
		pszWorkPhone = cOldGecos.GetString(2);
	if (*pszHomePhone == '\0' && cOldGecos.GetString(3))
		pszHomePhone = cOldGecos.GetString(3);
	if (*pszOther == '\0' && cOldGecos.GetString(4))
		pszOther = cOldGecos.GetString(4);
	char *pszNewGecos = new char[1024];
	if (NULL == pszNewGecos) {
		syslog(LOG_ERR, "memory allocation error");
		return false;
	}
	strcpy(pszNewGecos, pszFullName);
	strcat(pszNewGecos, ",");
	strcat(pszNewGecos, pszRoomNumber);
	strcat(pszNewGecos, ",");
	strcat(pszNewGecos, pszWorkPhone);
	strcat(pszNewGecos, ",");
	strcat(pszNewGecos, pszHomePhone);
	if ('\0' != *pszOther) {
		strcat(pszNewGecos, ",");
		strcat(pszNewGecos, pszOther);
	}
	if (false == cPasswdDatabase.ChangeGecos(pszLogin, pszNewGecos)) {
		syslog(LOG_ERR, "error modifying passwd file");
		delete pszNewGecos;
		return false;
	}
	delete pszNewGecos;
	
	return true;
}

bool change_password_group(char *pszLogin, char *pszNewPassword)
{
	char *pszCryptedPassword = crypt(pszNewPassword, crypt_make_salt());
	
	memset(pszNewPassword, 0, strlen(pszNewPassword));		// Clear password

#ifdef SHADOWGRP
	CGshadowDatabase cGshadowDatabase(false);
	if (NO_ERROR != cGshadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading gshadow file");
		return false;
	}
	if (false == cGshadowDatabase.ChangePassword(pszLogin, 
												 pszCryptedPassword, 
												 0, 
												 0, 
												 0, 
												 0)) {
		syslog(LOG_ERR, "error modifying gshadow file");
		return false;
	}
#else
	CGroupDatabase cGroupDatabase(false);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return false;
	}
	if (false == cGroupDatabase.ChangePassword(pszLogin,
											   pszCryptedPassword,
											   0,
											   0,
											   0,
											   0)) {
		syslog(LOG_ERR, "error modifying group file");
		return false;
	}
#endif
	
	return true;
}

bool delete_user(char *pszTarget)
{
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	
	char *pszHome = cPasswdDatabase.GetHome(pszTarget);
	if (NULL == pszHome) {
		syslog(LOG_ERR, "error getting home directory for user %s", pszTarget);
		return false;
	}
	pszHome = strdup(pszHome);
	if (NULL == pszHome) {
		syslog(LOG_ERR, "memory allcoation error");
		return false;
	}
	
	if (false == cPasswdDatabase.DeleteEntry(pszTarget)) {
		syslog(LOG_ERR, "error modifying passwd file");
		delete pszHome;
		return false;
	}
	
#ifdef SHADOWPWD
	CShadowDatabase cShadowDatabase(false);
	if (NO_ERROR != cShadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading shadow file");
		delete pszHome;
		return false;
	}
	if (false == cShadowDatabase.DeleteEntry(pszTarget)) {
		syslog(LOG_ERR, "error modifying shadow file");
		delete pszHome;
		return false;
	}
#endif

	char *pszTrash = pConfigFile->GetString("Delete", "trash", "/tmp");
	if (pConfigFile->GetBoolean("Delete", "move", false)) {
// Move home directory to trash
		char *pszNew = new char[1024];
		if (NULL == pszNew) {
			syslog(LOG_ERR, "memory allcoation error");
			delete pszHome;
			return false;
		}
		strcpy(pszNew, pszTrash);
		if (NULL != strrchr(pszHome, '/'))
			strcat(pszNew, strrchr(pszHome, '/'));
		else
			strcat(pszNew, pszHome);
		if (0 != rename(pszHome, pszNew)) {
			syslog(LOG_WARNING, "error moving home directory '%s' to trash directory '%s', %s", pszHome, pszNew, strerror(errno));
			delete pszNew;
			delete pszHome;
			return false;
		}
		syslog(LOG_INFO, "moving home directory '%s' to trash directory '%s'", pszHome, pszNew);
		delete pszNew;		
	}
	if (pConfigFile->GetBoolean("Delete", "remove", false)) {
// Remove home directory.
// Do not remove failure if the directory is not removed. Just log the error.
		if (false == remove_tree(pszHome))
			syslog(LOG_WARNING, "error removing home directory '%s'", pszHome);
		else
			syslog(LOG_WARNING, "removing home directory '%s'", pszHome);
	} 
	delete pszHome;
	
	return true;
}

bool delete_group(char *pszTarget)
{
	CGroupDatabase cGroupDatabase(false);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return false;
	}
	if (false == cGroupDatabase.DeleteEntry(pszTarget)) {
		syslog(LOG_ERR, "error modifying group file");
		return false;
	}

#ifdef SHADOWGRP
	CGshadowDatabase cGshadowDatabase(false);
	if (NO_ERROR != cGshadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading gshadow file");
		return false;
	}
	if (false == cGshadowDatabase.DeleteEntry(pszTarget)) {
		syslog(LOG_ERR, "error modifying gshadow file");
		return false;
	}
#endif
	
	return true;
}

bool add_user(char *pszTarget, char *pszGecos, char *pszDirectory, char *pszShell, int iUserID, int iGroupID)
{
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}

	if (false == cPasswdDatabase.AddEntry(pszTarget, iUserID, iGroupID, pszGecos, pszDirectory, pszShell)) {
		syslog(LOG_ERR, "error modifying passwd file");
		return false;
	}

#ifdef SHADOWPWD
	CShadowDatabase cShadowDatabase(false);
	if (NO_ERROR != cShadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading shadow file");
		return false;
	}
	if (false == cShadowDatabase.AddEntry(pszTarget)) {
		syslog(LOG_ERR, "error modifying shadow file");
		return false;
	}
#endif

	return true;
}

bool add_group(char *pszTarget,
			   char *pszMembers,
			   char *
#ifdef SHADOWGRP					
					 pszAdministrators
#endif // SHADOWGRP
									  ,
			   unsigned int iNewGroupID
)
{
	CGroupDatabase cGroupDatabase(false);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return false;
	}

	unsigned int iGroupID = (((unsigned int)-1 == iNewGroupID) ? cGroupDatabase.GetMaxGroupID() + 1 : iNewGroupID);
	
	if (false == cGroupDatabase.AddEntry(pszTarget, iGroupID, pszMembers)) {
		syslog(LOG_ERR, "error modifying group file");
		return false;
	}

#ifdef SHADOWGRP
	CGshadowDatabase cGshadowDatabase(false);
	if (NO_ERROR != cGshadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading gshadow file");
		return false;
	}
	if (false == cGshadowDatabase.AddEntry(pszTarget, pszAdministrators, pszMembers)) {
		syslog(LOG_ERR, "error modifying gshadow file");
		return false;
	}
#endif
	
	return true;
}

char *get_group(char *pszTarget)
{
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return "";
	}
	CGroupDatabase cGroupDatabase(false);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return "";
	}
	gid_t iGroup;
	if (false == cPasswdDatabase.GetGroup(pszTarget, &iGroup))
		return "";
	char *pszResult = cGroupDatabase.GetName(iGroup);
	if ((char *)NULL == pszResult)
		return "";
	
	return pszResult;
}

bool remove_tree(char *pszPath)
{
	bool fgResult = true;
	
	glob_t globbuf;
   
	char szTmp[512];
	strncpy(szTmp, pszPath, sizeof(szTmp));
	if (szTmp[strlen(szTmp) - 1] != '/')
		strncat(szTmp, "/", sizeof(szTmp));
	strncat(szTmp, "*", sizeof(szTmp));
	
	if (0 == glob(szTmp, /* GLOB_PERIOD | */ GLOB_MARK | GLOB_NOSORT, NULL, &globbuf)) {
		int i = 0;
		while((globbuf.gl_pathv[i])) {
			if ((globbuf.gl_pathv[i][strlen(globbuf.gl_pathv[i]) - 1] == '/') &&
				(strstr(globbuf.gl_pathv[i], "/../") == (char *)NULL) &&
				(strstr(globbuf.gl_pathv[i], "/./") == (char *)NULL)) {
				strncpy(szTmp, globbuf.gl_pathv[i], sizeof(szTmp));
				strncat(szTmp, "*", sizeof(szTmp));
				if (0 != glob(szTmp, /* GLOB_PERIOD | */ GLOB_MARK | GLOB_APPEND | GLOB_NOSORT, NULL, &globbuf))
					fgResult = false;
			}
			i += 1;
		}

		i = 0;
		while((globbuf.gl_pathv[i]))
			if (globbuf.gl_pathv[i][strlen(globbuf.gl_pathv[i]) - 1] == '/')
				rmdir(globbuf.gl_pathv[i++]);
			else
				remove(globbuf.gl_pathv[i++]);
		
		rmdir(pszPath);
	
		globfree(&globbuf);
	} else {
		if (0 == rmdir(pszPath))
			return true;
		dprintf(" glob() returned errror: %s.\n", strerror(errno));
		return false;
	}
                                                                      
	return fgResult;
}

bool action_delete_group(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[32];
	
	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	
	if (false == check_target_group(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_change_group(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	if (false == delete_group(szTarget)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s deleted group %s", pszLogin, szTarget);	
	write_byte(sockfd, OK);

	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_DELETE_GROUP,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_DELETEGROUP,
									   szTarget,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

bool action_change_group_password(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[128];
	char szNewPassword[128];
	
	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	int q;
	if (fgEncryptedPasswords)
		q = read_password(sockfd, szNewPassword, sizeof(szNewPassword));
	else
		q = read_string(sockfd, szNewPassword, sizeof(szNewPassword));
	if (-1 == q) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	
	if (false == check_target_group(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_change_group(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszObscure = obscure(pszPassword, szNewPassword, false);
	if (NULL != pszObscure) {
		write_byte(sockfd, WEAK_PASSWORD);
		write_string(sockfd, pszObscure);
		return false;
	}
	if (false == change_password_group(szTarget, szNewPassword)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s changed the password of group %s", pszLogin, szTarget);
	write_byte(sockfd, OK);
	
	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_GROUP_PASSWORD,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_CHANGEGROUPPASSWORD,
									   szTarget,
									   szNewPassword,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

bool action_add_user(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[32];
	char szGecos[128];
	char szDirectory[128];
	char szShell[128];
	int iGroupID;
	int iUserID;
	
	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		(-1 == read_string(sockfd, szGecos, sizeof(szGecos))) ||
		(-1 == read_string(sockfd, szDirectory, sizeof(szDirectory))) ||
		(-1 == read_string(sockfd, szShell, sizeof(szShell))) ||
		!read_integer(sockfd, &iUserID) ||
		!read_integer(sockfd, &iGroupID)) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	if ('\0' == szDirectory[0])
		strncpy(szDirectory, pConfigFile->GetString("Add", "directory", "/tmp"), sizeof(szDirectory));
	if ('\0' == szShell[0])
		strncpy(szShell, pConfigFile->GetString("Add", "shell", "/dev/null"), sizeof(szShell));
	
	if (strlen(szTarget) < 1 || strlen(szTarget) > 8) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	for (int q = 0; '\0' != szTarget[q]; q++)
		if (!isalnum(szTarget[q])) {
			write_byte(sockfd, FAILURE);
			return false;
		}
	
	int j, k = 0;
	for (j = 0; '\0' != szGecos[j]; j++) {
		if (',' == szGecos[j])
			k += 1;
		if (k > 4 ||
			';' == szGecos[j] ||
			'=' == szGecos[j] ||
			iscntrl(szGecos[j])) {
			write_byte(sockfd, FAILURE);
			return false;
		}
	}
	if ((j > 0) &&
		(0 == strcmp(pszLogin, szTarget)) &&
		(',' != szGecos[strlen(szGecos) - 1])) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	
	if (true == check_target(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_add(pszLogin)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	if (false == check_shell(szShell)) {
		syslog(LOG_NOTICE, "bad shell %s for user %s", szShell, szTarget);
		write_byte(sockfd, BAD_SHELL);
		return false;
	}

	{
		CPasswdDatabase cPasswdDatabase(true);
		if (NO_ERROR != cPasswdDatabase.GetError()) {
			syslog(LOG_ERR, "error reading passwd file");
			write_byte(sockfd, FAILURE);		
			return false;
		}
		CGroupDatabase cGroupDatabase(true);
		if (NO_ERROR != cGroupDatabase.GetError()) {
			syslog(LOG_ERR, "error reading group file");
			write_byte(sockfd, FAILURE);
			return false;
		}

		if (-1 == iUserID)
			iUserID = cPasswdDatabase.GetMaxUserID() + 1;

		if (-1 == iGroupID) {
			char *pszGroup = pConfigFile->GetString("Add", "group", "nogroup");
			if (0 == strcmp(pszGroup, "$UID"))
				iGroupID = iUserID;
			if (false == cGroupDatabase.GetGroup(pszGroup, (gid_t *)&iGroupID)) {
				if (false == pConfigFile->GetBoolean("Add", "add_group", false)) {
					if ((char *)NULL == cGroupDatabase.GetName(iGroupID)) {
						syslog(LOG_WARNING, "invalid group for user %s", szTarget);
						write_byte(sockfd, FAILURE);
						return false;
					}
				} else {
					if (false == add_group(szTarget, "", "", iGroupID)) {
						write_byte(sockfd, FAILURE);
						return false;
					}
				}
			}
		}
	}
// Here we know the numerical UID and GID of the new user.

	char szNew[128];	
	if (strstr(szDirectory, "$USER") != (char *)NULL) {
		strncpy(szNew, szDirectory, sizeof(szNew));
		strncpy(strstr(szNew, "$USER"), szTarget, sizeof(szNew));
		strncpy(szDirectory, szNew, sizeof(szNew));
	}
	
	if (true == pConfigFile->GetBoolean("Add", "add_directory", true)) {
		if (false == make_home(szTarget, szDirectory, iUserID, iGroupID)) {
			write_byte(sockfd, ERROR_MAKEDIR);
			return false;
		}
	}
	if (false == add_user(szTarget, szGecos, szDirectory, szShell, iUserID, iGroupID)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	if (false == make_pwdb()) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s added new user with login %s", pszLogin, szTarget);
	write_byte(sockfd, OK);

	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_ADD_USER,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_ADDUSER,
									   szTarget,
									   szGecos,
									   szDirectory,
									   szShell,
									   iUserID,
									   iGroupID,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	
	return true;
}

bool action_change_password(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[128];
	char szNewPassword[128];
	int iMin, iMax, iWarn, iInact;
	
	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		(-1 == (fgEncryptedPasswords 
				?
				read_password(sockfd, szNewPassword, sizeof(szNewPassword))
				:
				read_string(sockfd, szNewPassword, sizeof(szNewPassword)))) ||
		(false == read_integer(sockfd, &iMin)) ||
		(false == read_integer(sockfd, &iMax)) ||
		(false == read_integer(sockfd, &iWarn)) ||
		(false == read_integer(sockfd, &iInact))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
				
	if (false == check_target(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_change(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	bool fgSelf = (0 == strcmp(pszLogin, szTarget));
	if (true == fgSelf && (iMax != -2 || iMin != -2 || iWarn != -2 || iInact != -2)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszObscure = obscure(pszPassword, szNewPassword, fgSelf);
	if (NULL != pszObscure) {
		write_byte(sockfd, WEAK_PASSWORD);
		write_string(sockfd, pszObscure);
		return false;
	}
	if (false == change_password(szTarget, szNewPassword, iMin, iMax, iWarn, iInact)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	if (false == make_pwdb()) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s changed the password of user %s", pszLogin, szTarget);
	write_byte(sockfd, OK);
	
	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_USER_PASSWORD,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_CHANGEPASSWORD,
									   szTarget,
									   szNewPassword,
									   iMin,
									   iMax,
									   iWarn,
									   iInact,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

bool action_add_group(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[32];
	char szMembers[128];
	char szAdministrators[128];

	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		(-1 == read_string(sockfd, szMembers, sizeof(szMembers))) ||
		(-1 == read_string(sockfd, szAdministrators, sizeof(szAdministrators)))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}

	if (strlen(szTarget) < 1 || strlen(szTarget) > 8) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	for (int q = 0; '\0' != szTarget[q]; q++)
		if (!isalnum(szTarget[q])) {
			write_byte(sockfd, FAILURE);
			return false;
		}

	if (true == check_target_group(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_add(pszLogin)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	if (false == add_group(szTarget, szMembers, szAdministrators)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s added new group %s", pszLogin, szTarget);
	write_byte(sockfd, OK);
	
	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_ADD_GROUP,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_ADDGROUP,
									   szTarget,
									   szMembers,
									   szAdministrators,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

bool action_delete_user(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[32];

	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}

	if (false == check_target(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_change(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	if (false == delete_user(szTarget)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	if (false == make_pwdb()) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s deleted user %s", pszLogin, szTarget);
	write_byte(sockfd, OK);
	
	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_DEL_USER,
									   WELCOME,
									   NULL,
									   ACTION_DELETEUSER,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   szTarget,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

bool action_chfn(int sockfd, char *pszLogin, char *pszPassword)
{
	char szTarget[32];
	char szFullName[64];
	char szRoomNumber[64];
	char szWorkPhone[64];
	char szHomePhone[64];
	char szOther[64];				

	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		(-1 == read_string(sockfd, szFullName, sizeof(szFullName))) ||
		(-1 == read_string(sockfd, szRoomNumber, sizeof(szRoomNumber))) ||
		(-1 == read_string(sockfd, szWorkPhone, sizeof(szWorkPhone))) ||
		(-1 == read_string(sockfd, szHomePhone, sizeof(szHomePhone))) ||
		(-1 == read_string(sockfd, szOther, sizeof(szOther)))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}

	if (!check_gecos_field(szFullName) ||
		!check_gecos_field(szRoomNumber) ||
		!check_gecos_field(szWorkPhone) ||
		!check_gecos_field(szHomePhone) ||
		!check_gecos_field(szOther)) {
		syslog(LOG_WARNING, "invalid gecos for user %s supplied by user %s", pszLogin, szTarget);
		write_byte(sockfd, FAILURE);
		return false;
	}

	if ((0 == strcmp(pszLogin, szTarget)) && ('\0' != szOther[0])) {
		syslog(LOG_NOTICE, "user %s wants to changes his own other fiel", pszLogin);					
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}

	if (false == check_target(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return false;
	}
	if (false == allow_change(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	if (false == change_gecos(szTarget,
							  szFullName,
							  szRoomNumber,
							  szWorkPhone,
							  szHomePhone,
							  szOther)) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	if (false == make_pwdb()) {
		write_byte(sockfd, FAILURE);
		return false;
	}
	syslog(LOG_INFO, "user %s changed gecos for user %s", pszLogin, szTarget);
	write_byte(sockfd, OK);

	CStringArray cFailedServers;
	if (get_failed_servers(sockfd, &cFailedServers)) {
		CKeeper *pKeeper = new CKeeper(&cFailedServers,
									   SEQUENCE_CHFN,
									   WELCOME,
									   NULL,
									   pszLogin,
									   pszPassword,
									   GO_AHEAD,
									   ACTION_CHFN,
									   szTarget,
									   szFullName,
									   szRoomNumber,
									   szWorkPhone,
									   szHomePhone,
									   szOther,
									   OK);
		Postpone(pKeeper);
		delete pKeeper;
	}
	return true;
}

void action_get_group(int sockfd, char *pszLogin)
{
	char szTarget[32];

	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return;
	}

	if (false == check_target(szTarget)) {
		write_byte(sockfd, INVALID_USER);
		return;
	}
	if (false == allow_change(pszLogin, szTarget)) {
		write_byte(sockfd, NO_RIGHTS);
		return;
	}
	char *pszGroup = get_group(szTarget);
	write_string(sockfd, pszGroup);
	if ('\0' == *pszGroup) {
		write_byte(sockfd, FAILURE);
		return;
	}
	syslog(LOG_INFO, "user %s read group name of user %s", pszLogin, szTarget);
	write_byte(sockfd, OK);
}

bool check_shell(char *pszShell)
{
	bool fgResult = false;
	setusershell();
	char *pszUserShell;
	while (NULL != (pszUserShell = getusershell()))
		if (0 == strcmp(pszShell, pszUserShell))
			fgResult = true;
	endusershell();
	return fgResult;
}

bool make_home(char *pszTarget, char *pszDirectory, int iUserID, int iGroupID)
{
	if (0 != mkdir(pszDirectory, 0755)) {
		syslog(LOG_WARNING, "error creating user directory %s, %s", pszDirectory, strerror(errno));	
		return false;
	}
	if (0 != chown(pszDirectory, iUserID, iGroupID)) {
		syslog(LOG_WARNING, "error calling chown for user directory %s, %s", pszDirectory, strerror(errno));	
		return false;
	}
	
	return true;
}
