/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#define	PW_KEYBYNAME		'1'			// stored by name
#define	PW_KEYBYNUM			'2'			// stored by entry in the "file"
#define	PW_KEYBYUID			'3'			// stored by uid

#define PATH_PWD		"/etc"
#define MP_DB			"pwd.db"
#define SMP_DB			"spwd.db"
#define PWD				"passwd"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef MAKE_PWDB								// There is no BerkelyDB in the
#include <db.h>								// standard include directories
#endif MAKE_PWDB								// on Solaris 7.0 boxes. Thanks
													// to Michael Dietz <dietzma@pscts.com>
#include <unistd.h>
#include <syslog.h>

#include "database.hpp"
#include "passwd.hpp"
#include "pwdb.hpp"

// Function prototypes
void *compact(char *, bool *, long *, ...);

bool make_pwdb()
{
#ifdef MAKE_PWDB
// We are opening the master.passwd not in read only mode to use the file
// locking mechanism. This will prevent race condition when an user changes
// manually her password.
	CPasswdDatabase cPasswdDatabase(false);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error opening /etc/master.passwd");
		return false;
	}
	
	char szInsecure[256];
	char szInsecureDest[256];
	char szSecure[256];
	char szSecureDest[256];

	snprintf(szInsecure, sizeof(szInsecure), "%s/%s.tmp", PATH_PWD, MP_DB);
	snprintf(szSecure, sizeof(szSecure), "%s/%s.tmp", PATH_PWD, SMP_DB);
	snprintf(szInsecureDest, sizeof(szInsecureDest), "%s/%s", PATH_PWD, MP_DB);
	snprintf(szSecureDest, sizeof(szSecureDest), "%s/%s", PATH_PWD, SMP_DB);
	
#ifdef PASSWD43
	char szOldPasswd[256];
	char szOldPasswdDest[256];
	snprintf(szOldPasswd, sizeof(szOldPasswd), "%s/%s.tmp", PATH_PWD, PWD);
	snprintf(szOldPasswdDest, sizeof(szOldPasswdDest), "%s/%s", PATH_PWD, PWD);
#endif // PASSWD43
	
	HASHINFO strOpenInfo = {
		4096,									// bucket size
		32,										// ffactor
		256,									// nelem
		2048 * 1024,							// cachesize
		NULL,									// hash
		0										// lorder
	};

#ifdef PASSWD43
	int iOldPasswd = open(szOldPasswd,
						  O_RDWR | O_CREAT | O_EXCL,
						  S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	if (-1 == iOldPasswd) {
		syslog(LOG_ERR, "error opening %s", szOldPasswd);
		return false;
	}
	FILE *fpOldPasswd = fdopen(iOldPasswd, "w+");
	if (NULL == fpOldPasswd) {
		close(iOldPasswd);
		syslog(LOG_ERR, "error opening %s", szOldPasswd);
		return false;
	}
#endif

	DB *dbfInsecure = dbopen(szInsecure,
							 O_RDWR | O_CREAT | O_EXCL,
							 S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH,
							 DB_HASH,
							 &strOpenInfo);
	if (NULL == dbfInsecure) {
		fclose(fpOldPasswd);
		syslog(LOG_ERR, "error opening %s", szInsecure);
		return false;
	}
	DB *dbfSecure = dbopen(szSecure,
						   O_RDWR | O_CREAT | O_EXCL,
						   S_IRUSR | S_IWUSR,
						   DB_HASH,
						   &strOpenInfo);
	if (NULL == dbfSecure) {
		fclose(fpOldPasswd);
		dbfInsecure->close(dbfInsecure);		
		syslog(LOG_ERR, "error opening %s", szSecure);
		return false;
	}
	
	for (int i = 1; i <= cPasswdDatabase.GetEntries(); i++) {
		struct passwd *pstrEntry = cPasswdDatabase.GetEntry(i - 1)->GetEntry();

#ifdef PASSWD43
		fprintf(fpOldPasswd, "%s:%s:%d:%d:%s:%s:%s\n",
				pstrEntry->pw_name,
				"*",
				pstrEntry->pw_uid,
				pstrEntry->pw_gid,
				pstrEntry->pw_gecos,
				pstrEntry->pw_dir,
				pstrEntry->pw_shell);
#endif // PASSWD43		

		bool fgError;
		long lInsecure;
		void *pvInsecure = compact("ssiiissssii",
								   &fgError,
								   &lInsecure,
								   pstrEntry->pw_name,
								   "*",
								   pstrEntry->pw_uid,
								   pstrEntry->pw_gid,
								   pstrEntry->pw_change,
								   pstrEntry->pw_class,
								   pstrEntry->pw_gecos,
								   pstrEntry->pw_dir,
								   pstrEntry->pw_shell,
								   pstrEntry->pw_expire,
								   pstrEntry->pw_fields);
		if (true == fgError) {
			syslog(LOG_ERR, "error allocating memory");
			if (NULL != pvInsecure)
				free(pvInsecure);
			continue;
		}
		long lSecure;
		void *pvSecure = compact("ssiiissssii",
								 &fgError,
								 &lSecure,
								 pstrEntry->pw_name,
								 pstrEntry->pw_passwd,
								 pstrEntry->pw_uid,
								 pstrEntry->pw_gid,
								 pstrEntry->pw_change,
								 pstrEntry->pw_class,
								 pstrEntry->pw_gecos,
								 pstrEntry->pw_dir,
								 pstrEntry->pw_shell,
								 pstrEntry->pw_expire,
								 pstrEntry->pw_fields);
		if (true == fgError) {
			syslog(LOG_ERR, "error allocating memory");
			if (NULL != pvSecure)
				free(pvSecure);
			if (NULL != pvInsecure)
				free(pvInsecure);
			continue;
		}
		
		DBT strDataInsecure = { pvInsecure, lInsecure };
		DBT strDataSecure = { pvSecure, lSecure };
		
		unsigned char bBuf[256];
		
		DBT strKey = { bBuf, 0 };

		bBuf[0] = PW_KEYBYNAME;
		int l = strlen(pstrEntry->pw_name);
		memcpy(bBuf + 1, pstrEntry->pw_name, l);
		strKey.size = l + 1;
// Store insecure by name.
		if ((dbfInsecure->put)(dbfInsecure, &strKey, &strDataInsecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szInsecure);
// Store secure by name.
		if ((dbfSecure->put)(dbfSecure, &strKey, &strDataSecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szSecure);

		bBuf[0] = PW_KEYBYNUM;
		memcpy(bBuf + 1, &i, sizeof(int));
		strKey.size = sizeof(int) + 1;
// Store insecure by number.
		if ((dbfInsecure->put)(dbfInsecure, &strKey, &strDataInsecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szInsecure);
// Store secure by name.
		if ((dbfSecure->put)(dbfSecure, &strKey, &strDataSecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szSecure);

		bBuf[0] = PW_KEYBYUID;
		memcpy(bBuf + 1, &pstrEntry->pw_uid, sizeof(unsigned));
		strKey.size = sizeof(unsigned) + 1;
// Store insecure by uid.
		if ((dbfInsecure->put)(dbfInsecure, &strKey, &strDataInsecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szInsecure);
// Store secure by uid.
		if ((dbfSecure->put)(dbfSecure, &strKey, &strDataSecure, R_NOOVERWRITE) == -1)
			syslog(LOG_ERR, "error putting record in %s", szSecure);

		if (NULL != pvInsecure)
			free(pvInsecure);
		if (NULL != pvSecure)
			free(pvSecure);
	}
	
	dbfInsecure->close(dbfInsecure);
	dbfSecure->close(dbfSecure);
	
#ifdef PASSWD43
	fclose(fpOldPasswd);
	rename(szOldPasswd, szOldPasswdDest);
#endif // PASSWD43
	
	rename(szInsecure, szInsecureDest);
	rename(szSecure, szSecureDest);
#endif // MAKE_PWDB
	return true;
}

void *compact(char *pszMask, bool *pfgError, long *plLength, ...)
{
	int l;
	void *pvResult = NULL;
	void *pvNewResult = NULL;
	*plLength = 0;
	*pfgError = false;
	
	unsigned *pvArg = (unsigned *)&plLength + 1;
	for ( ; *pszMask; pszMask++) {
		switch (*pszMask) {
			case 's':
				l = (int)(strlen((char *)*pvArg) + 1);
				pvNewResult = realloc(pvResult, *plLength + l);
				if (NULL == pvNewResult) {
					*pfgError = true;
					return pvResult;
				}
				pvResult = pvNewResult;
				memcpy((char *)pvResult + *plLength, (char *)*pvArg, l);
				*plLength += l;
			break;
			case 'i':
				pvNewResult = realloc(pvResult, *plLength + sizeof(int));
				if (NULL == pvNewResult) {
					*pfgError = true;
					return pvResult;
				}
				pvResult = pvNewResult;
				memcpy((char *)pvResult + *plLength, pvArg, sizeof(int));
				*plLength += sizeof(int);
			break;
		}
		pvArg++;
	}
	
	return pvResult;
}
