/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "passwdd.hpp"

// Global variables
char *pszKeyFile;

char szConfigFile[256] = DEFAULT_SERVERCONFFILE;

bool fgEncryptedPasswords;

int iServerPort;
int iMaxProcesses;

int listenfd, newlistenfd;
int sockfd;

#ifdef SYSVSEM
key_t idCounter = 1099;
#endif

CConfigurationFile *pConfigFile = NULL;
CPidFile *pPidFile = NULL;

void usage()
{
	printf("Usage: passwdd [-c <file>]\n"
	       "       passwdd -v\n"
	       "Options:\n"
	       "  -c <file>   Use alternate config file instead the default\n"
	       "              '" DEFAULT_SERVERCONFFILE " system wide config file.\n"
	       "  -v          Print version and exit.\n"
	       "Written by: Alexander Feldman <alex@varna.net>\n\n");
}

int main(int argc, char **argv)
{
	int pidstatus;
	int child_process;
	char *pszResult;
	struct sockaddr_in serv;
	struct sockaddr_in child;
	
	int w;
	while (-1 != (w = getopt(argc, argv, "c:v")))
		switch (w) {
			case 'c':
				strncpy(szConfigFile, optarg, sizeof(szConfigFile));
			break;
			case 'v':
				printf("%s\n", GetVersion());
				return 0;						// Successful exit
			break;
			default:
				usage();
				return 0;						// Successful exit
			break;
		}
	argc -= optind;							// optind is declared in <unistd.h>
	argv += optind;
	
	if (0 != argc) {
		usage();
		return 0;
	}
	
	pPidFile = new CPidFile("passwdd.pid");
	if (NULL == pPidFile) {
		fprintf(stderr, "passwdd: error allocating memory.\n");
		return 1;
	}

	dprintf("Checking pidfile.\n");
	if (true == pPidFile->CheckPid()) {
#ifndef DEBUG
		pid_t pid;
		if ((pid = fork()) > 0)
			exit(0);								// Parent process
		else
			if (pid < 0) {						// An error occured in the fork
				fprintf(stderr, "passwdd: initial fork error:%s.\n", strerror(errno));
				delete pPidFile;
				exit(1);
			}
		int i = getdtablesize();
		for (int j = 0; j < i; j++)
		  close(j);
		setsid();								// Creates a session and closes the controlling tty
#endif
	} else {
		delete pPidFile;
		fputs("passwdd: already running.\n", stderr);
		return 1;
	}
	dprintf("Writing pidfile.\n");
	if (true == pPidFile->CheckPid()) {
		if (false == pPidFile->WritePid()) {
			delete pPidFile;
			dprintf("Can't write pid file.\n");
			return 1;
		}
	} else {
		delete pPidFile;
		dprintf("Pidfile (and pid) already exists.\n");
		return 1;
	}
	
	signal(SIGTERM, SignalProc);
	signal(SIGHUP, SignalProc);
	signal(SIGCHLD, SignalProc);
	signal(SIGINT, SignalProc);
	signal(SIGQUIT, SignalProc);
	dprintf("Starting.\n");
	
	openlog("passwdd", LOG_PID, LOG_USER);

	pConfigFile = new CConfigurationFile(szConfigFile, &pszResult);
	if (NULL == pConfigFile) {
		syslog(LOG_ERR, "error allocating memory");
		delete pPidFile;
		return 1;
	}
	if (NULL != pszResult) {
		syslog(LOG_ERR, "conf error: %s", pszResult);
		delete pPidFile;
		delete pConfigFile;
		return 1;
	}

	fgEncryptedPasswords = pConfigFile->GetBoolean("Security", "encrypt_passwords", true);
	pszKeyFile = pConfigFile->GetString("Security", "key", PATH_SYSCONFDIR "passwdd.key");
	if (true == fgEncryptedPasswords) {
		int i = read_local_key(pszKeyFile);
		if (1 == i)
			syslog(LOG_ERR, "error allocating memory");
		if (2 == i)
			syslog(LOG_ERR, "error reading private key");
		if (0 != i) {
			delete pPidFile;
			delete pConfigFile;
			return 1;
		}
	}

	if ((listenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		if (true == fgEncryptedPasswords)
			destroy_local_key();
		delete pPidFile;
		delete pConfigFile;
		syslog(LOG_ERR, "socket error: %s", strerror(errno));
		return 1;
	}
	
	iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
	iMaxProcesses = pConfigFile->GetInteger("Global", "max_processes", DEFAULT_MAXPROCS);
	
	memset(&serv, 0, sizeof(serv));
	serv.sin_family = AF_INET;
	serv.sin_addr.s_addr = htonl(INADDR_ANY);
	serv.sin_port = htons(iServerPort);

	if (bind(listenfd, (struct sockaddr *)&serv, sizeof(serv)) < 0) {
		if (true == fgEncryptedPasswords)
			destroy_local_key();
		delete pPidFile;
		delete pConfigFile;
		syslog(LOG_ERR, "bind error: %s", strerror(errno));
		return 1;
	}
	if (listen(listenfd, SOMAXCONN) < 0) {
		if (true == fgEncryptedPasswords)
			destroy_local_key();
		delete pPidFile;
		delete pConfigFile;
		syslog(LOG_ERR, "listen error: %s", strerror(errno));
		return 1;
	}

#ifdef SYSVSEM
	idCounter = ftok(szConfigFile, 1);
	int s;
	if (-1 == (s = semget(idCounter, 1, IPC_CREAT | 0660))) {
		if (true == fgEncryptedPasswords)
			destroy_local_key();
		delete pPidFile;
		delete pConfigFile;
		syslog(LOG_WARNING, "semget error for key %ld: %s", idCounter, strerror(errno));
		return 1;
	}
	union semun arg;	
	arg.val = iMaxProcesses;
	if (-1 == semctl(s, 0, SETVAL, arg)) {
		if (true == fgEncryptedPasswords)
			destroy_local_key();
		delete pPidFile;
		delete pConfigFile;
		syslog(LOG_ERR, "semctl error: %s", strerror(errno));
		return 1;
	}
#endif

	syslog(LOG_INFO, "daemon starting");
	child_process = 0;
	while (0 == child_process) {			// Loop begins here
		memset(&child, 0, sizeof(child));
		socklen_t childlen = (socklen_t)sizeof(child);
		if ((sockfd = accept(listenfd, (struct sockaddr *)&child, &childlen)) < 0) {
			if (EINTR != errno)
				syslog(LOG_ERR, "accept error (%s): %s", GetDDN(&child), strerror(errno));
			continue;
		}
		syslog(LOG_INFO, "opened client connection from %s", GetDDN(&child));
		if (false == AllowHost(child.sin_addr)) {
			syslog(LOG_NOTICE, "unauthorized client connection from %s", GetDDN(&child));
			close(sockfd);
			continue;
		}
#ifndef DEBUG		
		pid_t pid;
		if ((pid = fork()) == 0) {			// This is the child
#ifdef SYSVSEM
			struct sembuf sbuf = { 0, -1, SEM_UNDO };
			if (-1 == semop(s, &sbuf, 1)) {
				syslog(LOG_ERR, "semget error: %s", strerror(errno));
				close(listenfd);
				close(sockfd);
				return 0;
			}
#endif			
			close(listenfd);
#endif			
			server(sockfd, GetDDN(&child));
#ifndef DEBUG
			close(sockfd);
			child_process = 1;				// Exit from main loop for the child process
		} else {									// We are parent, go on listening and error checking
			if (pid < 0)
				syslog(LOG_USER | LOG_INFO, "fork error: %s", strerror(errno));
			while(waitpid(-1, &pidstatus, WNOHANG) > 0);
			close(sockfd);
			child_process = 0;
		}
#endif
	}
	return 0;
}

bool AllowHost(struct in_addr lSource)
{
// We will compare the source address, the target address and target netmask
// all in host byte order. There is no matter what ordering we will use
// but is logically more correct to perform it in native ordering.	
	unsigned long int ulSourceHost = ntohl(lSource.s_addr);
	unsigned long int ulTargetHost;
	unsigned long int ulTargetMask;
	char *pszHost;
	bool fgInvert;

	CTokenizedString cAllowedHosts(pConfigFile->GetString("Permissions", "hosts_allow", "localhost"), ",;");

	if ((pszHost = cAllowedHosts.GetFirstString()))
		do {
			if (false == ParseDDN(TrimBoth(pszHost, " \t\x0a\x0d"), &ulTargetHost, &ulTargetMask, &fgInvert)) {
				syslog(LOG_WARNING, "configuration file error, skipping host");
				continue;
			}
			if (fgInvert && ((ulSourceHost & ulTargetMask) == ulTargetHost))
				return false;
			if ((ulSourceHost & ulTargetMask) == ulTargetHost)
				return true;
		} while ((pszHost = cAllowedHosts.GetNextString()));
	
	return false;
}

void SignalProc(int iSignal)
{
	CConfigurationFile *pNewConfigFile;	
	struct sockaddr_in serv;
	char *pszResult;
	int iPidStatus;	
	
	switch (iSignal) {
		case SIGCHLD:
			while(waitpid(-1, &iPidStatus, WNOHANG) > 0);
		break;
		case SIGHUP:
			syslog(LOG_INFO, "reinitializing");		
			pNewConfigFile = new CConfigurationFile(szConfigFile, &pszResult);
			if (NULL == pNewConfigFile) {
				syslog(LOG_ERR, "error allocating memory");
				return;
			}
			if (NULL != pszResult) {
				syslog(LOG_ERR, "conf error: %s", pszResult);
				delete pNewConfigFile;
				return;
			}
			pConfigFile = pNewConfigFile;
		
			if (iServerPort != pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT)) {
				iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
				if ((newlistenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
					syslog(LOG_ERR, "socket error: %s", strerror(errno));
					return;
				}
				memset(&serv, 0, sizeof(serv));
				serv.sin_family = AF_INET;
				serv.sin_addr.s_addr = htonl(INADDR_ANY);
				serv.sin_port = htons(iServerPort);
	
				if (bind(newlistenfd, (struct sockaddr *)&serv, sizeof(serv)) < 0) {
					syslog(LOG_ERR, "bind error: %s", strerror(errno));
					return;
				}
	
				if (listen(newlistenfd, SOMAXCONN) < 0) {
					syslog(LOG_ERR, "listen error: %s", strerror(errno));
					return;
				}
				close(listenfd);
				listenfd = newlistenfd;
			}
		break;
		case SIGTERM:
		case SIGINT:
		case SIGQUIT:
#ifdef SYSVSEM
			union semun arg;
			int s;
			if (-1 != (s = semget(idCounter, 1, 0)))
				semctl(s, 0, IPC_RMID, arg);
#endif
			if (true == fgEncryptedPasswords)
				destroy_local_key();
			delete pConfigFile;
			delete pPidFile;
			if (0 != iSignal)
				dprintf("exiting on signal %d\n", iSignal);
		
			exit(0);
		break;
	}
	signal(iSignal, SignalProc);			// Reset to get it again	
}

void dprintf(char *fmt, ...)
{
#ifdef DEBUG
	va_list ap;
	
	va_start(ap, fmt);
	vfprintf(stdout, fmt, ap);
	va_end(ap);
	
	fflush(stdout);
#endif
}
