/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include "config.h"
#include "passwd.hpp"
#include "strsep.hpp"

CPasswdEntry::CPasswdEntry(FILE *fpFile) : CEntry(fpFile, (void *)&strEntry, sizeof(struct passwd), &GetEntry)
{
}

CPasswdEntry::CPasswdEntry(char *pszName,
						   uid_t i,
						   gid_t j,
						   char *pszGecos,
						   char *pszDirectory,
						   char *pszShell)
{
	iErrorCode = NO_ERROR;
	
	strEntry.pw_name = strdup(pszName);
#ifdef SHADOWPWD	
	strEntry.pw_passwd = strdup("x");
#else	
	strEntry.pw_passwd = strdup("*");
#endif // SHADOWPWD
	strEntry.pw_uid = i;
	strEntry.pw_gid = j;
#ifdef ATT_CLASS
	strEntry.pw_class = strdup("");
#endif // ATT_CLASS
#ifdef ATT_CHANGE
	strEntry.pw_change = 0;
#endif // ATT_CHANGE
#ifdef ATT_EXPIRE
	strEntry.pw_expire = 0;
#endif // ATT_EXPIRE
#ifdef ATT_AGE
	strEntry.pw_age = strdup("");
#endif // ATT_AGE
#ifdef ATT_COMMENT
	strEntry.pw_comment = strdup("");
#endif // ATT_COMMENT
	strEntry.pw_gecos = strdup(pszGecos);
	strEntry.pw_dir = strdup(pszDirectory);
	strEntry.pw_shell = strdup(pszShell);
	strEntry.pw_fields = 0;
	if (NULL == strEntry.pw_name ||
		NULL == strEntry.pw_passwd ||
#ifdef ATT_AGE
		NULL == strEntry.pw_age ||
#endif // ATT_AGE
#ifdef ATT_COMMENT
		NULL == strEntry.pw_comment ||
#endif // ATT_COMMENT
		NULL == strEntry.pw_gecos ||
		NULL == strEntry.pw_dir ||
		NULL == strEntry.pw_shell)
		iErrorCode = MEMORY_ERROR;
}

CPasswdEntry::~CPasswdEntry()
{
	if (NULL != strEntry.pw_name)
		free(strEntry.pw_name);
	if (NULL != strEntry.pw_passwd)
		free(strEntry.pw_passwd);
#ifdef ATT_AGE
	if (NULL != strEntry.pw_age)
		free(strEntry.pw_age);
#endif // ATT_AGE
#ifdef ATT_COMMENT
	if (NULL != strEntry.pw_comment)
		free(strEntry.pw_comment);
#endif // ATT_COMMENT
#ifdef ATT_CLASS
	if (NULL != strEntry.pw_class)
		free(strEntry.pw_class);
#endif // ATT_CLASS
	if (NULL != strEntry.pw_gecos)
		free(strEntry.pw_gecos);
	if (NULL != strEntry.pw_dir)
		free(strEntry.pw_dir);
	if (NULL != strEntry.pw_shell)
		free(strEntry.pw_shell);
}

char *CPasswdEntry::GetName()
{
	return strEntry.pw_name;
}

char *CPasswdEntry::GetPassword()
{
	return strEntry.pw_passwd;
}

char *CPasswdEntry::GetGecos()
{
	return strEntry.pw_gecos;
}

char *CPasswdEntry::GetHome()
{
	return strEntry.pw_dir;
}

gid_t CPasswdEntry::GetGroup()
{
	return strEntry.pw_gid;
}

uid_t CPasswdEntry::GetUser()
{
	return strEntry.pw_uid;
}

bool CPasswdEntry::SetPassword(char *pszPassword, int, int, int, int)
{
	char *pszNewPassword = strdup(pszPassword);
	if (NULL == pszNewPassword)
		return false;
	free(strEntry.pw_passwd);
	strEntry.pw_passwd = pszNewPassword;
	return true;
}

bool CPasswdEntry::SetGecos(char *pszGecos)
{
	char *pszNewGecos = strdup(pszGecos);
	if (NULL == pszNewGecos)
		return false;
	free(strEntry.pw_gecos);
	strEntry.pw_gecos = pszNewGecos;
	return true;
}

bool CPasswdEntry::GetEntry(char *pszEntry, void *pstrTarget)
{
	assert(NULL != pszEntry);
	assert(NULL != pstrTarget);
	
	struct passwd *pstrEntry = (struct passwd *)pstrTarget;
	
	register int i, j;
	
	for (i = j = 0; pszEntry[i] != '\0'; i++)
		if (pszEntry[i] == ':')
	  		j += 1;

	if (6
#ifdef ATT_CLASS
	+ 1
#endif
#ifdef ATT_CHANGE
	+ 1
#endif
#ifdef ATT_EXPIRE
	+ 1
#endif
	!= j)
		return false;
	
	bool fgError = false;
	pstrEntry->pw_name = strdup(strsep(&pszEntry, ":"));
	pstrEntry->pw_passwd = strdup(strsep(&pszEntry, ":"));
	pstrEntry->pw_uid = ToLong(strsep(&pszEntry, ":"), -1, &fgError);
	pstrEntry->pw_gid = ToLong(strsep(&pszEntry, ":"), -1, &fgError);
#ifdef ATT_CLASS
	pstrEntry->pw_class = strdup(strsep(&pszEntry, ":"));
#endif // ATT_CLASS
#ifdef ATT_CHANGE
	pstrEntry->pw_change = ToLong(strsep(&pszEntry, ":"), 0, &fgError);
#endif // ATT_CHANGE
#ifdef ATT_EXPIRE
	pstrEntry->pw_expire = ToLong(strsep(&pszEntry, ":"), 0, &fgError);
#endif // ATT_EXPIRE
	pstrEntry->pw_gecos = strdup(strsep(&pszEntry, ":"));
	pstrEntry->pw_dir = strdup(strsep(&pszEntry, ":"));
	pstrEntry->pw_shell = strdup(strsep(&pszEntry, ":"));
	pstrEntry->pw_fields = 0;
	
	if ((NULL == pstrEntry->pw_name) || 
		(NULL == pstrEntry->pw_passwd) ||
#ifdef ATT_CLASS
		(NULL == pstrEntry->pw_class) ||
#endif // ATT_CLASS
		(NULL == pstrEntry->pw_gecos) ||
		(NULL == pstrEntry->pw_dir) ||
		(NULL == pstrEntry->pw_shell))
		return false;
	
#ifdef ATT_AGE
	pstrEntry->pw_age = "";
	char *p = strchr(pstrEntry->pw_passwd, ',');
	if (NULL != p) {
		*p = '\0';
		pstrEntry->pw_age = strdup(p + 1);
	}
#endif // ATT_AGE	
#ifdef ATT_COMMENT
	pstrEntry->pw_comment = "";
	char *q = strrchr(pstrEntry->pw_comment, ',');
	if (NULL != q) {
		*q = '\0';
		pstrEntry->pw_comment = strdup(q + 1);
	}
#endif // ATT_COMMENT
	
	return true;
}

bool CPasswdEntry::PutEntry(FILE *fpFile)
{
	if ((NULL == strEntry.pw_name) || 
		(NULL == strEntry.pw_passwd) ||
		(NULL == strEntry.pw_gecos) ||
		(NULL == strEntry.pw_dir) ||
		(NULL == strEntry.pw_shell))
		return false;
	
	fprintf(fpFile,
			"%s:%s"
#ifdef ATT_AGE				
			"%s%s"
#endif // ATT_AGE				
			":%d:%d"
#ifdef ATT_CLASS			  
			":%s"
#endif // ATT_CLASS
#ifdef ATT_CHANGE
			":%ld"
#endif
#ifdef ATT_EXPIRE
			":%ld"
#endif
			":%s"
#ifdef ATT_COMMENT				
			"%s%s"
#endif // ATT_COMMENT				
			":%s:%s\n",
			strEntry.pw_name,
			strEntry.pw_passwd,
#ifdef ATT_AGE				
			('\0' != *strEntry.pw_age ? "," : ""),
			strEntry.pw_age,
#endif // ATT_AGE				
			(int)strEntry.pw_uid,
			(int)strEntry.pw_gid,
#ifdef ATT_CLASS
			strEntry.pw_class,
#endif // ATT_CLASS
#ifdef ATT_CHANGE
			strEntry.pw_change,
#endif // ATT_CHANGE
#ifdef ATT_EXPIRE
			strEntry.pw_expire,
#endif // ATT_EXPIRE
			strEntry.pw_gecos,
#ifdef ATT_COMMENT				
			('\0' != *strEntry.pw_comment ? "," : ""),
			strEntry.pw_comment,
#endif // ATT_COMMENT				
			strEntry.pw_dir,
			strEntry.pw_shell);
	
	return true;
}
