/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#include "config.h"
#include "sockio.hpp"
#include "keeper.hpp"
#include "strings.hpp"
#include "conf.hpp"

extern CConfigurationFile *pConfigFile;

extern int iServerPort;

bool Postpone(CKeeper *pKeeper)
{
	char *pszScheme = pConfigFile->GetString("Global", "retry", NULL);
	if (NULL == pszScheme) {
		syslog(LOG_ERR, "configuration file error, assuming default retry scheme");
		pszScheme = "12*30,6*60,3*120,2*180";
	}
	CTokenizedString cSchemeTest(pszScheme, ";,");
	bool fgError = false;
	char *pszTime = NULL;
	char *pszToken;
	if ((pszToken = cSchemeTest.GetFirstString()))
		do {
			pszTime = strchr(pszToken, '*');
			if (NULL != pszTime) {
				*pszTime = '\0';
				pszTime += 1;
			} else {
				fgError = true;
				break;
			}
			if (atoi(pszToken) <= 0 || atoi(pszToken) >= 1000 ||
				 atoi(pszTime) <= 0 || atoi(pszTime) > 18000) {
				fgError = true;
				break;
			}
		} while ((pszToken = cSchemeTest.GetNextString()));
	
	if (true == fgError) {
		syslog(LOG_ERR, "configuration file error, assuming default retry scheme");
		pszScheme = "12*30,6*60,3*120,2*180";
	}
	
	pid_t t = fork();
	if (-1 == t) {
		syslog(LOG_ERR, "fork error: %s", strerror(errno));
		return false;
	}
	if (0 == t) {
		syslog(LOG_INFO, "child thread created");

		CTokenizedString cSchemeFinal(pszScheme, ";,");
		if ((pszToken = cSchemeFinal.GetFirstString()))
			do {
				pszTime = strchr(pszToken, '*');
				*pszTime = '\0';
				pszTime += 1;
				
				int iCount = atoi(pszToken);
				int iInterval = atoi(pszTime) * 60;
				for (int i = 0; i < iCount; i++) {
					syslog(LOG_INFO, "retrying postponed connection");
					if (true == pKeeper->Try())
						return true;
					syslog(LOG_INFO, "sleeping for %d seconds", iInterval);
					sleep(iInterval);
				}
			} while ((pszToken = cSchemeFinal.GetNextString()));
		syslog(LOG_NOTICE, "postponed transaction expired");
		return false;
		// child thread execution terminated
	}
	return false;
}

CKeeper::CKeeper(CStringArray *pHosts, char *pszSequence, ...)
{
	CKeeper::pHosts = pHosts;
	CKeeper::pszSequence = strdup(pszSequence);
	
	iParams = strlen(pszSequence) / 2;
	
	va_list ap;
	va_start(ap, pszSequence);
	if (NULL != (ppvParameters = new (void *)[iParams]))
		for (int i = 0; i < iParams; i++) {
			switch (pszSequence[i * 2 + 1]) {
				case PARAM_STRING:
				case PARAM_PASSWORD:
					ppvParameters[i] = strdup(va_arg(ap, char *));
					break;
				case PARAM_INTEGER:
					ppvParameters[i] = (void *)va_arg(ap, int);
					break;
				case PARAM_BYTE:
					ppvParameters[i] = (void *)va_arg(ap, char);
					break;
				case PARAM_PUBLICKEY:
					ppvParameters[i] = va_arg(ap, void *);
					break;
			}
		}
	va_end(ap);
}

CKeeper::~CKeeper()
{
	if (NULL != ppvParameters && NULL != pszSequence)
		for (int i = 0; i < iParams; i++)
			if (pszSequence[i * 2 + 1] == PARAM_STRING ||
				pszSequence[i * 2 + 1] == PARAM_PASSWORD)
				free(ppvParameters[i]);
	delete ppvParameters;
	if (NULL != pszSequence)
		delete pszSequence;
}

bool CKeeper::Try()
{
	bool fgResult = true;
	char *pszHost;
	if ((pszHost = pHosts->GetFirstString()))
		do {
			syslog(LOG_INFO, "(postponed) trying to connect with server: %s", pszHost);

			int sockfd;
			if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
				syslog(LOG_ERR, "(postponed) socket error: %s", strerror(errno));
				fgResult = false;
				continue;
			}
			struct hostent *pHost = gethostbyname(pszHost);
			if (NULL == pHost) {
				syslog(LOG_ERR, "(postponed) resolving error: %s", strerror(errno));
				fgResult = false;
				close(sockfd);
				continue;
			}
			struct sockaddr_in client;
			memset(&client, 0, sizeof(client));
			client.sin_family = AF_INET;
			client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
			client.sin_port = htons(iServerPort);
			if (connect(sockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
				syslog(LOG_ERR, "(postponed) connect error: %s", strerror(errno));
				fgResult = false;
				close(sockfd);
				continue;
			}
			if (false == TryHost(sockfd, pszHost, iServerPort)) {
				syslog(LOG_ERR, "(postponed) read/write error");
				fgResult = false;
				close(sockfd);
				continue;
			}
			close(sockfd);
			pHosts->DeleteString();
		} while ((pszHost = pHosts->GetNextString()));
	return fgResult;
}

bool CKeeper::TryHost(int sockfd, char *pszServer, int iServerPort)
{
	char szPublicKey[256];
	bool fgEncryptedPasswords;
	for (int i = 0; i < iParams; i++) {
		if (DIRECTION_OUT == pszSequence[i * 2]) {
			switch (pszSequence[i * 2 + 1]) {
				case PARAM_PASSWORD:
					if (false == write_password(sockfd, (char *)ppvParameters[i]))
				   	return false;
				break;
				case PARAM_STRING:
					if (-1 == write_string(sockfd, (char *)ppvParameters[i]))
				   	return false;
				break;
				case PARAM_INTEGER:
					if (!write_integer(sockfd, (int)ppvParameters[i]))
						return false;
				break;
				case PARAM_BYTE:
					if (!write_byte(sockfd, (unsigned char)(int)ppvParameters[i]))
						return false;
				break;
			}
		} else if (DIRECTION_IN == pszSequence[i * 2]) {
			switch (pszSequence[i * 2 + 1]) {
				case PARAM_PUBLICKEY:
					if (1 != read_integer(sockfd, (int *)&fgEncryptedPasswords))
						return false;

					if (true == fgEncryptedPasswords) {
						snprintf(szPublicKey,
								 sizeof(szPublicKey),
								 "%s/%s_%d.pub",
								 pConfigFile->GetString("Security", "key_directory", "/var/spool/passwdc"),
								 pszServer,
								 iServerPort);
						if (0 != recv_remote_key(sockfd, szPublicKey))
							return false;
					}
					break;
				case PARAM_STRING:
					char szString[1024];
					if (-1 == read_string(sockfd, szString, sizeof(szString)))
				   	return false;
					if (0 != strcmp((char *)ppvParameters[i], szString))
						return false;
					break;
				case PARAM_INTEGER:
					int k;
					if (!read_integer(sockfd, &k))
						return false;
					if ((int)ppvParameters[i] != k)
						return false;
					break;
				case PARAM_BYTE:
					unsigned char c;
					if (!read_byte(sockfd, &c))
						return false;
					if ((unsigned char)(int)ppvParameters[i] != c)
						return false;
					break;
			}
		} else
			return false;
	}
	return true;
}
