/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include "config.h"
#include "gshadow.hpp"
#include "strings.hpp"
#include "strsep.hpp"

CGshadowEntry::CGshadowEntry(FILE *fpFile) : CEntry(fpFile, (void *)&strEntry, sizeof(struct sgrp), &GetEntry)
{
}

CGshadowEntry::CGshadowEntry(char *pszName, char *pszAdmins, char *pszMembers)
{
	iErrorCode = NO_ERROR;
	
	strEntry.sg_name = strdup(pszName);
	strEntry.sg_passwd = strdup("!");
	strEntry.sg_adm = Tokenize(pszAdmins, ",");
	strEntry.sg_mem = Tokenize(pszMembers, ",");
}

CGshadowEntry::~CGshadowEntry()
{
	if (NULL != strEntry.sg_name)
		free(strEntry.sg_name);
	if (NULL != strEntry.sg_passwd)
		free(strEntry.sg_passwd);
	if (NULL != (char **)strEntry.sg_adm) {
		for (int i = 0; strEntry.sg_adm[i] != (char *)NULL; i++)
			delete strEntry.sg_adm[i];
		delete strEntry.sg_adm;
	}
	if (NULL != (char **)strEntry.sg_mem) {
		for (int i = 0; strEntry.sg_mem[i] != (char *)NULL; i++)
			delete strEntry.sg_mem[i];
		delete strEntry.sg_mem;
	}
}

char *CGshadowEntry::GetName()
{
	return strEntry.sg_name;
}

char *CGshadowEntry::GetPassword()
{
	return strEntry.sg_passwd;
}

char *CGshadowEntry::GetAdmins()
{
	return Concatenate(strEntry.sg_adm, ',');
}

char *CGshadowEntry::GetMembers()
{
	return Concatenate(strEntry.sg_mem, ',');
}

bool CGshadowEntry::SetPassword(char *pszPassword, int, int, int, int)
{
	char *pszNewPassword = strdup(pszPassword);
	if (NULL == pszNewPassword)
		return false;
	free(strEntry.sg_passwd);
	strEntry.sg_passwd = pszNewPassword;
	return true;
}

bool CGshadowEntry::SetAdmins(char *pszAdmins)
{
	char **pszList = Tokenize(pszAdmins, ",");
	
	if (NULL == (char **)pszList)
		return false;
	
	for (int i = 0; (char *)NULL != strEntry.sg_adm[i]; i++)
		delete strEntry.sg_adm[i];
	delete strEntry.sg_adm;
	
	strEntry.sg_adm = pszList;
	
	return true;
}

bool CGshadowEntry::SetMembers(char *pszMembers)
{
	char **pszList = Tokenize(pszMembers, ",");
	
	if (NULL == (char **)pszList)
		return false;
	
	for (int i = 0; (char *)NULL != strEntry.sg_mem[i]; i++)
		delete strEntry.sg_mem[i];
	delete strEntry.sg_mem;
	
	strEntry.sg_mem = pszList;
	
	return true;
}

bool CGshadowEntry::GetEntry(char *pszEntry, void *pstrTarget)
{
	assert(NULL != pszEntry);
	assert(NULL != pstrTarget);
	
	struct sgrp *pstrEntry = (struct sgrp *)pstrTarget;
	
	register int i, j;
	
	for (i = j = 0; pszEntry[i] != '\0'; i++)
		if (pszEntry[i] == ':')
	  		j += 1;
	
	if (3 != j)
		return false;
	
	pstrEntry->sg_name = strdup(strsep(&pszEntry, ":"));
	pstrEntry->sg_passwd = strdup(strsep(&pszEntry, ":"));
	pstrEntry->sg_adm = Tokenize(strsep(&pszEntry, ":"), ",");
	pstrEntry->sg_mem = Tokenize(strsep(&pszEntry, ":"), ",");
	
	if ((NULL == pstrEntry->sg_name) || 
		 (NULL == pstrEntry->sg_passwd) ||
		 (NULL == pstrEntry->sg_adm) ||
		 (NULL == pstrEntry->sg_mem))
		return false;
	
	return true;
}

bool CGshadowEntry::PutEntry(FILE *fpFile)
{
	if ((NULL == strEntry.sg_name) || 
		 (NULL == strEntry.sg_passwd) ||
		 (NULL == strEntry.sg_adm) ||
		 (NULL == strEntry.sg_mem))
		return false;
	
	char *pszAdmins = Concatenate(strEntry.sg_adm, ',');
	char *pszMembers = Concatenate(strEntry.sg_mem, ',');
	if (NULL != pszAdmins && NULL != pszMembers) {
		fprintf(fpFile,
				  "%s:%s:%s:%s\n",
				  strEntry.sg_name,
				  strEntry.sg_passwd,
				  pszAdmins,
				  pszMembers);
		delete pszAdmins;
		delete pszMembers;
	}

	return true;
}

bool CGshadowEntry::IsAdmin(char *pszLogin)
{
	register int i = 0;
	while (strEntry.sg_adm[i])
		if (0 == strcmp(pszLogin, strEntry.sg_adm[i++]))
			return true;
	return false;
}
