/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include "config.h"
#include "group.hpp"
#include "strsep.hpp"
#include "strings.hpp"

CGroupEntry::CGroupEntry(FILE *fpFile) : CEntry(fpFile, (void *)&strEntry, sizeof(struct group), &GetEntry)
{
}

CGroupEntry::CGroupEntry(char *pszName, gid_t j, char *pszMembers)
{
	iErrorCode = NO_ERROR;
	
	strEntry.gr_name = strdup(pszName);

#ifdef SHADOWGRP
	strEntry.gr_passwd = strdup("x");
#else	
	strEntry.gr_passwd = strdup("!");
#endif // SHADOWGRP
	
	strEntry.gr_gid = j;
	strEntry.gr_mem = Tokenize(pszMembers, ",");
}

CGroupEntry::~CGroupEntry()
{
	if (NULL != strEntry.gr_name)
		free(strEntry.gr_name);
	if (NULL != strEntry.gr_passwd)
		free(strEntry.gr_passwd);
	if (NULL != (char **)strEntry.gr_mem) {
		for (int i = 0; strEntry.gr_mem[i] != (char *)NULL; i++)
			free(strEntry.gr_mem[i]);
		free(strEntry.gr_mem);
	}
}

char *CGroupEntry::GetName()
{
	return strEntry.gr_name;
}

char *CGroupEntry::GetPassword()
{
	return strEntry.gr_passwd;
}

char *CGroupEntry::GetMembers()
{
	return Concatenate(strEntry.gr_mem, ',');
}

gid_t CGroupEntry::GetGroup()
{
	return strEntry.gr_gid;
}

bool CGroupEntry::SetPassword(char *pszPassword, int, int, int, int)
{
	char *pszNewPassword = strdup(pszPassword);
	if (NULL == pszNewPassword)
		return false;
	free(strEntry.gr_passwd);
	strEntry.gr_passwd = pszNewPassword;
	return true;
}

bool CGroupEntry::SetMembers(char *pszMembers)
{
	char **pszList = Tokenize(pszMembers, ",");
	
	if (NULL == (char **)pszList)
		return false;
	
	for (int i = 0; (char *)NULL != strEntry.gr_mem[i]; i++)
		delete strEntry.gr_mem[i];
	delete strEntry.gr_mem;
	
	strEntry.gr_mem = pszList;
	
	return true;
}

bool CGroupEntry::GetEntry(char *pszEntry, void *pstrTarget)
{
	assert(NULL != pszEntry);
	assert(NULL != pstrTarget);
	
	struct group *pstrEntry = (struct group *)pstrTarget;
	
	register int i, j;
	
	for (i = j = 0; pszEntry[i] != '\0'; i++)
		if (pszEntry[i] == ':')
	  		j += 1;
	
	if (3 != j)
		return false;
	
	bool fgError = false;
	pstrEntry->gr_name = strdup(strsep(&pszEntry, ":"));
	pstrEntry->gr_passwd = strdup(strsep(&pszEntry, ":"));
	pstrEntry->gr_gid = ToLong(strsep(&pszEntry, ":"), -1, &fgError);
	pstrEntry->gr_mem = Tokenize(strsep(&pszEntry, ":"), ",");
	
	if ((NULL == pstrEntry->gr_name) || 
		 (NULL == pstrEntry->gr_passwd) ||
		 (NULL == pstrEntry->gr_mem) ||
		 (true == fgError))
		return false;
	
	return true;
}

bool CGroupEntry::PutEntry(FILE *fpFile)
{
	if ((NULL == strEntry.gr_name) || 
		 (NULL == strEntry.gr_passwd) ||
		 (NULL == strEntry.gr_mem))
		return false;
	
	char *pszMembers = Concatenate(strEntry.gr_mem, ',');
	if (NULL != pszMembers) {
		fprintf(fpFile,
				  "%s:%s:%d:%s\n",
				  strEntry.gr_name,
				  strEntry.gr_passwd,
				  (int)strEntry.gr_gid,
				  pszMembers);
		delete pszMembers;
	}

	return true;
}