/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __DATABASE_HPP__
#define __DATABASE_HPP__

#include "config.h"
#include "passwd.hpp"
#include "shadow.hpp"
#include "group.hpp"
#include "gshadow.hpp"

#include <stdio.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>

#define MAX_PATH		1024
#define BLOCK_SIZE	1024
#define LOCK_TRIES	  15					// Tries
#define LOCK_SLEEP	   1					// Seconds

template <class TEntry> class CDatabase
{
	public:
		CDatabase(const char *, bool, bool);
		virtual ~CDatabase();
	
	public:
		virtual int GetError() { return iErrorCode; };
		virtual bool FindUser(char *);
		virtual bool GetGroup(char *, gid_t *);
		virtual bool ChangePassword(char *, char *, int, int, int, int);
		virtual bool ChangeGecos(char *, char *);
		virtual bool DeleteEntry(char *);
		virtual bool UpdateDatabase();
		virtual char *GetPassword(char *);			// Returns the password field for each database
		virtual char *GetName(gid_t);					// For group database returns group name
																// for the parameter - group id
		virtual char *GetGecos(char *);
		virtual char *GetHome(char *);
	
		virtual void CancelUpdate();
	
		inline int GetEntries() { return iEntries; }
		inline TEntry *GetEntry(int iEntry) { return ppEntry[iEntry]; }
	
	private:
		bool CreateBackup();
		bool CreateTemp();
		bool Lock();
		bool LockNow();
		void Unlock();

	private:
		char szDatabaseName[MAX_PATH];
		char szBackupName[MAX_PATH];
		char szTempName[MAX_PATH];	
	
		bool fgLocked;
		bool fgUseLckpwdf;
	
		FILE *fpDatabase;
	
	protected:
		TEntry **ppEntry;
	
		bool fgModified;
		bool fgReadOnly;	
	
		int iEntries;
		int iErrorCode;
};

#define SHADOW_FILE "/etc/shadow"
#ifdef MASTER_PASSWD
#define PASSWD_FILE "/etc/master.passwd"
#else
#define PASSWD_FILE "/etc/passwd"
#endif
#define GROUP_FILE "/etc/group"
#define GSHADOW_FILE "/etc/gshadow"

class CShadowDatabase : public CDatabase <class CShadowEntry>
{
	public:
		CShadowDatabase(bool fgReadOnly = true);
	
	public:
		bool AddEntry(char *);
};

class CPasswdDatabase : public CDatabase <class CPasswdEntry>
{
	public:
		CPasswdDatabase(bool fgReadOnly = true);
	
	public:
		bool AddEntry(char *, unsigned int, unsigned int, char *, char *, char *);
		int GetMaxUserID();
};

class CGroupDatabase : public CDatabase <class CGroupEntry>
{
	public:
		CGroupDatabase(bool fgReadOnly = true);

	public:
		bool AddEntry(char *, unsigned int, char *);
		int GetMaxGroupID();
};

class CGshadowDatabase : public CDatabase <class CGshadowEntry>
{
	public:
		CGshadowDatabase(bool fgReadOnly = true);

	public:
		bool AddEntry(char *, char *, char *);
		bool IsAdministrator(char *, char *);
};


#endif // __DATABASE_HPP__
