/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "allow.hpp"

#ifdef HAVE_LIBCRYPT
	extern "C" char *crypt(const char *, const char *);
#endif

extern CConfigurationFile *pConfigFile;

bool to_long(char *pszEntry, long *plValue)
{
	if ((NULL == pszEntry) || '\0' == pszEntry[0])
		return false;
	char *pszInvalid;
	long lResult = strtol(pszEntry, &pszInvalid, 10);
	if ('\0' == *pszInvalid) {				// We want the whole string to be valid
		*plValue = lResult;
		return true;		
	}
	return false;
}

// This function returns true if the given password matches the appropriate
// entry for the user who is in the system databases.
bool check_login(char *pszLogin, char *pszPassword)
{
// Check to see if there is a valid login name
	CPasswdDatabase cPasswdDatabase(true);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	if (false == cPasswdDatabase.FindUser(pszLogin)) {	
		syslog(LOG_NOTICE, "invalid user: %s", pszLogin);
		return false;
	}
#ifdef SHADOWPWD
	CShadowDatabase cShadowDatabase(true);
	if (NO_ERROR != cShadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading shadow file");
		return false;
	}
	if (false == cShadowDatabase.FindUser(pszLogin)) {
		syslog(LOG_NOTICE, "invalid user: %s", pszLogin);
		return false;
	}
	char *pszUserPassword = cShadowDatabase.GetPassword(pszLogin);
#else
	char *pszUserPassword = cPasswdDatabase.GetPassword(pszLogin);
#endif
	if (0 != strcmp(pszUserPassword, crypt(pszPassword, pszUserPassword))) { 
		syslog(LOG_NOTICE, "invalid password of user: %s", pszLogin);
		return false;
	}

	return true;
}

// This function returns true if there exists such user in the system
// databases.
bool check_target(char *pszTarget)
{
// Check to see if there is a valid login name
	CPasswdDatabase cPasswdDatabase(true);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	if (false == cPasswdDatabase.FindUser(pszTarget))
		return false;
#ifdef SHADOWPWD
	CShadowDatabase cShadowDatabase(true);
	if (NO_ERROR != cShadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading shadow file");
		return false;
	}
	if (false == cShadowDatabase.FindUser(pszTarget))
		return false;
#endif	
	return true;
}

// This function returns true if there exists such group in the system
// databases.
bool check_target_group(char *pszTarget)
{
// Check to see if there is a valid login name
	CGroupDatabase cGroupDatabase(true);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return false;
	}
	if (false == cGroupDatabase.FindUser(pszTarget))
		return false;							// There is no such group in /etc/group
#ifdef SHADOWGRP
	CGshadowDatabase cGshadowDatabase(true);
	if (NO_ERROR != cGshadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading gshadow file");
		return false;
	}
	if (false == cGshadowDatabase.FindUser(pszTarget))
		return false;							// There is no such group in /etc/gshadow.
#endif	
	return true;
}

// This function returns true if the supplied user has the right to add new
// accounts.
bool allow_add(char *pszLogin)
{
	CPasswdDatabase cPasswdDatabase(true);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");
		return false;
	}
	CGroupDatabase cGroupDatabase(true);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");
		return false;
	}
	CTokenizedString cAllowedUsers(pConfigFile->GetString("Add", "master", ""), ",;");
	
	char *pszAllowed;
	char *pszGroup;

	gid_t iLoginGroup;
	if (false == cPasswdDatabase.GetGroup(pszLogin, &iLoginGroup)) {
		syslog(LOG_WARNING, "invalid group for user %s", pszLogin);
		return false;
	}
	pszGroup = cGroupDatabase.GetName(iLoginGroup);
	if (NULL == pszGroup) {
		syslog(LOG_WARNING, "invalid group for user %s", pszLogin);
		return false;
	}
	
	if ((pszAllowed = cAllowedUsers.GetFirstString()))
		do {
			pszAllowed = TrimBoth(pszAllowed, " \t\x0a\x0d");
			if ('@' == pszAllowed[0]) {
			  	if (0 == strcmp(pszGroup, pszAllowed + 1))
					return true;
			} else {
			   if (0 == strcmp(pszLogin, pszAllowed))
					return true;
			}
		} while ((pszAllowed = cAllowedUsers.GetNextString()));
	
	syslog(LOG_NOTICE, "unauthorized add request from user %s", pszLogin);

	return false;
}

// If login and target are the same, the function returns true if the given
// target is allowed to change her password.
// If login differs from the target then the function returns true if the
// password of the target is allowed to be changed AND the user with the given
// login has the right to do this.
bool allow_change(char *pszLogin, char *pszTarget)
{
	char *pszAllowed = NULL;
	char *pszPowered;
	char *pszSheep;
	bool fgAllowed = false;
	bool fgPowered = false;
	bool fgSheep = false;
	
	CPasswdDatabase cPasswdDatabase(true);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");		
		return false;
	}
	CGroupDatabase cGroupDatabase(true);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");		
		return false;
	}
	
	CTokenizedString cAllowedUsers(pConfigFile->GetString("Permissions", "users_allow", ""), ";,");
	CTokenizedString cPoweredUsers(pConfigFile->GetString("Permissions", "master", ""), ";,");
	
	gid_t iLoginGroup;
	gid_t iTargetGroup;
	
	if (false == cPasswdDatabase.GetGroup(pszLogin, &iLoginGroup)) {
		syslog(LOG_WARNING, "invalid group for user %s", pszLogin);
		return false;
	}
	if (false == cPasswdDatabase.GetGroup(pszTarget, &iTargetGroup)) {
		syslog(LOG_WARNING, "invalid group for user %s", pszTarget);
		return false;
	}
	
	if ((pszAllowed = cAllowedUsers.GetFirstString()))
		do {
			pszAllowed = TrimBoth(pszAllowed, " \t\x0a\x0d");
			if ('@' == pszAllowed[0] || '>' == pszAllowed[0]) {
				gid_t iAllowedGroup;
				if (false == to_long(pszAllowed + 1, (long *)&iAllowedGroup))
					if (false == cGroupDatabase.GetGroup(pszAllowed + 1, &iAllowedGroup)) {
						syslog(LOG_ERR, "invalid group '%s', specified in the configuration file", pszAllowed + 1);
						return false;
					}
				if ((('@' == pszAllowed[0]) && (iTargetGroup == iAllowedGroup)) ||
					 (('>' == pszAllowed[0]) && (iTargetGroup > iAllowedGroup))) {
					fgAllowed = true;
					break;
				}
			} else {
			   if (0 == strcmp(pszTarget, pszAllowed)) {
					fgAllowed = true;
					break;
				}
			}
		} while ((pszAllowed = cAllowedUsers.GetNextString()));

	if (false == fgAllowed) {
		syslog(LOG_NOTICE, "unauthorized user %s", pszLogin);
		return false;
	}
	if (0 == strcmp(pszLogin, pszTarget))
		return true;

	if ((pszPowered = cPoweredUsers.GetFirstString()))
		do {
			pszPowered = TrimBoth(pszPowered, " \t\x0a\x0d");
			if (NULL == (pszSheep = strchr(pszPowered, ':'))) {
				syslog(LOG_WARNING, "configuration file error, skipping power users");
				continue;
			}
			*pszSheep = '\0';
			pszSheep += 1;
			if ('@' == pszPowered[0] || '>' == pszPowered[0]) {
				gid_t iPoweredGroup;
				if (false == to_long(pszPowered + 1, (long *)&iPoweredGroup))
					if (false == cGroupDatabase.GetGroup(pszPowered + 1, &iPoweredGroup)) {
						syslog(LOG_ERR, "invalid group '%s', specified in the configuration file", pszAllowed + 1);
						return false;
					}
				if ((('@' == pszPowered[0]) && (iLoginGroup == iPoweredGroup)) ||
					 (('>' == pszPowered[0]) && (iLoginGroup > iPoweredGroup)))
					fgPowered = true;
			} else {
				if (0 == strcmp(pszPowered, pszLogin))
					fgPowered = true;
			}
			if (fgPowered && ('@' == pszSheep[0] || '>' == pszSheep[0])) {
				gid_t iSheepGroup;
				if (false == to_long(pszSheep + 1, (long *)&iSheepGroup))
					if (false == cGroupDatabase.GetGroup(pszSheep + 1, &iSheepGroup)) {
						syslog(LOG_ERR, "invalid group '%s', specified in the configuration file", pszAllowed + 1);
						return false;
					}
				if ((('@' == pszSheep[0]) && (iTargetGroup == iSheepGroup)) ||
					 (('>' == pszSheep[0]) && (iTargetGroup > iSheepGroup))) {
					fgSheep = true;
					break;
				}
			} else {
				if (0 == strcmp(pszSheep, pszTarget)) {
					fgSheep = true;
					break;
				}
			}
		} while ((pszPowered = cPoweredUsers.GetNextString()));
	
	if (true == fgSheep)
		return true;
	
	syslog(LOG_NOTICE, "%s tried to change the password of %s without rights", pszLogin, pszTarget);
	
	return false;
}

// This function returns true if the given login has the right to change the
// fields of the supplied target group. This may happen if the user is master
// for the group in the passwdd.conf or if the user is master of the group in
// the administrators field in /etc/gshadow.
bool allow_change_group(char *pszLogin, char *pszTarget)
{
	char *pszPowered;
	char *pszSheep;
	bool fgPowered = false;
	bool fgSheep = false;
	
	CPasswdDatabase cPasswdDatabase(true);
	if (NO_ERROR != cPasswdDatabase.GetError()) {
		syslog(LOG_ERR, "error reading passwd file");		
		return false;
	}
	CGroupDatabase cGroupDatabase(true);
	if (NO_ERROR != cGroupDatabase.GetError()) {
		syslog(LOG_ERR, "error reading group file");		
		return false;
	}
	
	gid_t iLoginGroup;
	
	if (false == cPasswdDatabase.GetGroup(pszLogin, &iLoginGroup)) {
		syslog(LOG_WARNING, "invalid group for user %s", pszLogin);
		return false;
	}
	
	char *pszLoginGroup = cGroupDatabase.GetName(iLoginGroup);
	
	if (NULL == pszLoginGroup) {
		syslog(LOG_WARNING, "invalid group for user %s", pszLogin);
		return false;
	}
	
	CTokenizedString cPoweredUsers(pConfigFile->GetString("Permissions", "master", ""), ";,");
	
	if ((pszPowered = cPoweredUsers.GetFirstString()))
		do {
			pszPowered = TrimBoth(pszPowered, " \t\x0a\x0d");
			if (NULL == (pszSheep = strchr(pszPowered, ':'))) {
				syslog(LOG_WARNING, "configuration file error, skipping power users");
				continue;
			}
			*pszSheep = '\0';
			pszSheep += 1;
			if ('@' == pszPowered[0]) {
				if (0 == strcmp(pszPowered + 1, pszLoginGroup))
					fgPowered = true;
			} else {
				if (0 == strcmp(pszPowered, pszLogin))
					fgPowered = true;
			}
			if (fgPowered && ('@' == pszSheep[0]))
				if (0 == strcmp(pszSheep + 1, pszTarget)) {
					fgSheep = true;
					break;
				}
		} while ((pszPowered = cPoweredUsers.GetNextString()));
	
	if (true == fgSheep)
		return true;
	
#ifdef SHADOWGRP
	CGshadowDatabase cGshadowDatabase(true);
	if (NO_ERROR != cGshadowDatabase.GetError()) {
		syslog(LOG_ERR, "error reading gshadow file");
		return false;
	}
	if (true == cGshadowDatabase.IsAdministrator(pszTarget, pszLogin))
		return true;
#endif	
	
	syslog(LOG_NOTICE, "%s tried to change the group password of %s without rights", pszLogin, pszTarget);
	
	return false;
}
