/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "der.hpp"

CBase64Coder cBase64Coder;		

CDEREncodedPrimitive::CDEREncodedPrimitive() : pbValue(NULL), wTypeSize(1), wLengthSize(1), wValueSize(0)
{
	bType[0] = ASN1_TAG_NULL | ASN1_ID_PRIMITIVE;
	bLength[0] = 0;
}

CDEREncodedPrimitive::~CDEREncodedPrimitive()
{
	if (NULL != pbValue)
		free(pbValue);
}

void CDEREncodedPrimitive::EncodeLength(Word wLength)
{
	Word i;
	if (wLength > 0x7f) {
#ifdef LITTLE_ENDIAN_HOST
		wLength = REVERSEWORD(wLength);
#endif
		wLengthSize = 1;										// Leave space
		for (i = 0; 0 == ((Byte *)&wLength)[i]; i++);
		for ( ; i < BYTESINWORD; i++)
			bLength[wLengthSize++] = ((Byte *)&wLength)[i];
		bLength[0] = (Byte)(wLengthSize - 1) | 0x80;
	} else {
		bLength[0] = (Byte)wLength;								// Value
		wLengthSize = 1;
	}
}

Word CDEREncodedPrimitive::DecodeLength()
{
	if (wLengthSize == 1)
		return (Word)bLength[0];
	
	Word wResult = 0;
	
	for (Word i = 0; i < wLengthSize - 1; i++)
		((Byte *)&wResult)[i] = bLength[wLengthSize - 1 - i];
	
#ifdef BIG_ENDIAN_HOST
	wResult = REVERSEWORD(wResult);
#endif

	return wResult;
}

void CDEREncodedPrimitive::Write(int iOut)
{
	if ((wTypeSize != (Word)scl_write(iOut, bType, wTypeSize)) ||
		(wLengthSize != (Word)scl_write(iOut, bLength, wLengthSize)) ||
		(wValueSize != (Word)scl_write(iOut, pbValue, wValueSize)))
		throw(SCL_WRITE_ERROR);
}

void CDEREncodedPrimitive::Read(int iIn)
{
	Byte bNewType[16];
	if (wTypeSize != (Word)scl_read(iIn, bNewType, wTypeSize))
		throw(SCL_READ_ERROR);
	if (0 != memcmp(bType, bNewType, wTypeSize))
		throw(SCL_UNEXPECTEDTYPE_ERROR);
	if (1 != scl_read(iIn, bLength, 1))
		throw(SCL_READ_ERROR);
	if ((0x80 & bLength[0]) == 0) {
		wLengthSize = 1;
	} else {
		wLengthSize = (bLength[0] & 0x7f) + 1;
		if (wLengthSize - 1 != (Word)scl_read(iIn, bLength + 1, wLengthSize - 1))
			throw(SCL_READ_ERROR);
		if (wLengthSize > BYTESINWORD + 1)
			throw(SCL_UNEXPECTEDLENGTH_ERROR);
	}
	wValueSize = DecodeLength();
	pbValue = (Byte *)calloc(wValueSize, sizeof(Byte));
	if (NULL == pbValue)
		throw(SCL_MEMORYALLOCATION_ERROR);
	if (wValueSize != (Word)scl_read(iIn, pbValue, wValueSize))
		throw(SCL_READ_ERROR);
}

void CDEREncodedPrimitive::WriteBase64(int iOut, bool fgPad)
{
	Word i;
	Word j = wTypeSize + wLengthSize + wValueSize;
	char *pszBuffer = (char *)calloc(cBase64Coder.EncodedSize(j) + j / BASE64_MAXLINESIZE + 64, sizeof(Byte));
	if (NULL == pszBuffer)
		throw(SCL_MEMORYALLOCATION_ERROR);
	i = cBase64Coder.EncodeData(bType, wTypeSize, pszBuffer, false);
	i += cBase64Coder.EncodeData(bLength, wLengthSize, pszBuffer + i, false);
	i += cBase64Coder.EncodeData(pbValue, wValueSize, pszBuffer + i, fgPad);
	if (i != (Word)scl_write(iOut, pszBuffer, i))
		throw(SCL_WRITE_ERROR);
	free(pszBuffer);
}

void CDEREncodedPrimitive::ReadBase64(int iIn)
{
	Byte bNewType[16];
	if (wTypeSize != cBase64Coder.DecodeData(bNewType, wTypeSize, iIn))
		throw(SCL_READ_ERROR);
	if (0 != memcmp(bType, bNewType, wTypeSize))
		throw(SCL_UNEXPECTEDTYPE_ERROR);
	if (1 != cBase64Coder.DecodeData(bLength, 1, iIn))
		throw(SCL_READ_ERROR);
	if ((0x80 & bLength[0]) == 0) {
		wLengthSize = 1;
	} else {
		wLengthSize = (bLength[0] & 0x7f) + 1;
		if (wLengthSize - 1 != cBase64Coder.DecodeData(bLength + 1, wLengthSize - 1, iIn))
			throw(SCL_READ_ERROR);
		if (wLengthSize > BYTESINWORD + 1)
			throw(SCL_UNEXPECTEDLENGTH_ERROR);
	}
	wValueSize = DecodeLength();
	pbValue = (Byte *)calloc(wValueSize, sizeof(Byte));
	if (NULL == pbValue)
		throw(SCL_MEMORYALLOCATION_ERROR);
	if (wValueSize != cBase64Coder.DecodeData(pbValue, wValueSize, iIn))
		throw(SCL_READ_ERROR);
}

void CDEREncodedPrimitive::FromSequence(CDEREncodedSequence &cDEREncodedSequence)
{
	if (cDEREncodedSequence.wPointer + wTypeSize > cDEREncodedSequence.wValueSize)
		throw(SCL_PARSE_ERROR);
	if (0 != memcmp(cDEREncodedSequence.pbValue + cDEREncodedSequence.wPointer, bType, wTypeSize))
		throw(SCL_UNEXPECTEDTYPE_ERROR);
	cDEREncodedSequence.wPointer += wTypeSize;

	if (cDEREncodedSequence.wPointer + 1 > cDEREncodedSequence.wValueSize)
		throw(SCL_PARSE_ERROR);
	bLength[0] = cDEREncodedSequence.pbValue[cDEREncodedSequence.wPointer];
	cDEREncodedSequence.wPointer += 1;
	if ((0x80 & bLength[0]) == 0) {
		wLengthSize = 1;
	} else {
		wLengthSize = (bLength[0] & 0x7f) + 1;
		if (cDEREncodedSequence.wPointer + wLengthSize - 1 > cDEREncodedSequence.wValueSize)
			throw(SCL_PARSE_ERROR);
		memcpy(bLength + 1, cDEREncodedSequence.pbValue + cDEREncodedSequence.wPointer, wLengthSize - 1);
		cDEREncodedSequence.wPointer += (wLengthSize - 1);
		if (wLengthSize > BYTESINWORD + 1)
			throw(SCL_UNEXPECTEDLENGTH_ERROR);
	}
	wValueSize = DecodeLength();
	pbValue = (Byte *)calloc(wValueSize, sizeof(Byte));
	if (NULL == pbValue)
		throw(SCL_MEMORYALLOCATION_ERROR);
	if (cDEREncodedSequence.wPointer + wValueSize > cDEREncodedSequence.wValueSize)
		throw(SCL_PARSE_ERROR);
	memcpy(pbValue, cDEREncodedSequence.pbValue + cDEREncodedSequence.wPointer, wValueSize);
	cDEREncodedSequence.wPointer += wValueSize;
}

CDEREncodedSequence::CDEREncodedSequence() : CDEREncodedPrimitive(), wPointer(0)
{
	cBase64Coder.Reset();
	bType[0] = ASN1_TAG_SEQUENCE | ASN1_ID_CONSTRUCTED;
}

void CDEREncodedSequence::AddPrimitive(const CDEREncodedPrimitive &cDEREncodedPrimitive)
{
	Byte *pbNewValue = (Byte *)realloc(pbValue, wValueSize + cDEREncodedPrimitive.GetSize());
	if (NULL == pbNewValue)
		throw(SCL_MEMORYALLOCATION_ERROR);
	pbValue = pbNewValue;
	memcpy(pbValue + wValueSize, cDEREncodedPrimitive.GetType(), cDEREncodedPrimitive.GetTypeSize());
	wValueSize += cDEREncodedPrimitive.GetTypeSize();
	memcpy(pbValue + wValueSize, cDEREncodedPrimitive.GetLength(), cDEREncodedPrimitive.GetLengthSize());
	wValueSize += cDEREncodedPrimitive.GetLengthSize();
	memcpy(pbValue + wValueSize, cDEREncodedPrimitive.GetValue(), cDEREncodedPrimitive.GetValueSize());
	wValueSize += cDEREncodedPrimitive.GetValueSize();
	EncodeLength(wValueSize);
}

CDEREncodedBigNumber::CDEREncodedBigNumber() : CBigNumber(), CDEREncodedPrimitive()
{
	bType[0] = ASN1_TAG_INTEGER | ASN1_ID_PRIMITIVE;
}

CDEREncodedBigNumber::CDEREncodedBigNumber(const CBigNumber &cBigNumber) : CBigNumber(cBigNumber), CDEREncodedPrimitive()
{
	bType[0] = ASN1_TAG_INTEGER | ASN1_ID_PRIMITIVE;

	pbValue = (Byte *)calloc(wWords + 1, sizeof(Word));
	if (NULL == pbValue)
		throw(SCL_MEMORYALLOCATION_ERROR);

	if (IsZero()) {
		bLength[0] = 1;
		wLengthSize = 1;
		pbValue[0] = 0;
		wValueSize = 1;
		return;
	}

#ifdef BIG_ENDIAN_HOST
	ReverseData();
#endif

	Word wFirst = REVERSEWORD(pwBigNumber[wWords - 1]);
	Word wSkip = 0;
	while (0 == ((Byte *)&wFirst)[wSkip])
		wSkip += 1;

	wValueSize = wWords * BYTESINWORD - wSkip;
	Word wPadSize = 0;
	if (false == fgSign) {
		if ((0x80 & ((Byte *)&wFirst)[wSkip]) == 0x80) {
			pbValue[0] = 0;
			wPadSize = 1;
		}
	} else {
		for (Word j = 0; j < wValueSize; j++)
			((Byte *)pwBigNumber)[j] = ~(((Byte *)pwBigNumber)[j]);
		Add(1);
		if ((0x80 & ((Byte *)pwBigNumber)[wValueSize - 1]) != 0x80) {
			pbValue[0] = 0xff;
			wPadSize = 1;
		}
	}
	wValueSize += wPadSize;

	for (Word i = wPadSize; i < wValueSize; i++)
		pbValue[i] = ((Byte *)pwBigNumber)[wValueSize - i - 1];

#ifdef BIG_ENDIAN_HOST
	ReverseData();
#endif

	EncodeLength(wValueSize);
}

CDEREncodedBigNumber::CDEREncodedBigNumber(CDEREncodedSequence &cSequence) : CBigNumber(), CDEREncodedPrimitive()
{
	bType[0] = ASN1_TAG_INTEGER | ASN1_ID_PRIMITIVE;
	FromSequence(cSequence);
	
	wWords = (wValueSize + BYTESINWORD - 1) / BYTESINWORD;
	pwBigNumber = (Word *)calloc(wWords, sizeof(Word));
	if (NULL == pwBigNumber)
		throw(SCL_MEMORYALLOCATION_ERROR);
	Byte *pbDest = (Byte *)pwBigNumber + wValueSize - 1;
	if ((wValueSize >= 1) && ((0x80 & pbValue[0]) == 0x80))
		fgSign = true;
	for (Word i = 0; i < wValueSize; i++)
		(*pbDest--) = pbValue[i];
	if (true == fgSign) {
		Sub(1);
		for (Word i = 0; i < wValueSize; i++)
			((Byte *)pwBigNumber)[i] = ~(((Byte *)pwBigNumber)[i]);
	}
#ifdef BIG_ENDIAN_HOST
	ReverseData();
#endif
	StripLeadingZeroes();
}

void CDEREncodedBigNumber::Read(int iIn)
{
	CDEREncodedPrimitive::Read(iIn);

	wWords = (wValueSize + BYTESINWORD - 1) / BYTESINWORD;
	pwBigNumber = (Word *)calloc(wWords, sizeof(Word));
	if (NULL == pwBigNumber)
		throw(SCL_MEMORYALLOCATION_ERROR);
	Byte *pbDest = (Byte *)pwBigNumber + wValueSize - 1;
	if ((wValueSize >= 1) && ((0x80 & pbValue[0]) == 0x80))
		fgSign = true;
	for (Word i = 0; i < wValueSize; i++)
		(*pbDest--) = pbValue[i];
	if (true == fgSign) {
		Sub(1);
		for (Word i = 0; i < wValueSize; i++)
			((Byte *)pwBigNumber)[i] = ~(((Byte *)pwBigNumber)[i]);
	}
#ifdef BIG_ENDIAN_HOST
	ReverseData();
#endif
	StripLeadingZeroes();
}
