/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "block.hpp"

CECBCryptographer::CECBCryptographer(CBlockObject &cBlock, CBlockKey &cKey) : cBlockObject(cBlock), cBlockKey(cKey)
{
}

CECBCryptographer::~CECBCryptographer()
{
	memset(bLastBlock, 0, sizeof(bLastBlock));
}

Word CECBCryptographer::EncryptData(Byte *pbData, Word wLength, bool fgPad, bool fgCopy)
{
	return EncryptData(pbData, pbData, wLength, fgPad, fgCopy);
}

Word CECBCryptographer::DecryptData(Byte *pbData, Word wLength, bool fgPad)
{
	return DecryptData(pbData, pbData, wLength, fgPad);
}

Word CECBCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength, bool fgPad, bool fgCopy)
{
	Word wBlockSize = cBlockObject.GetBlockSize();
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize;
	Word i;
	for (i = 0; i < wFullBlocks; i++) {
		cBlockObject.SetData(pbData + i * wBlockSize, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	if (true == fgPad) {
		memcpy(bLastBlock, pbData + i * wBlockSize, wRemainder);
		memset(bLastBlock + wRemainder, (Byte)wRemainder, wBlockSize - wRemainder);
		cBlockObject.SetData(bLastBlock, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		if (true == fgCopy)
			memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	return wFullBlocks * wBlockSize + ((fgPad && fgCopy) ? wBlockSize : 0);
}

Word CECBCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength, bool fgPad)
{
	Word wBlockSize = cBlockObject.GetBlockSize();
	Word wFullBlocks = wLength / wBlockSize;
	Word i, wResult = 0;
	for (i = 0; i < wFullBlocks; i++) {
		cBlockObject.SetData(pbData + i * wBlockSize, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		if (i != wFullBlocks - 1 || false == fgPad) {
			memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
			wResult += wBlockSize;
		} else {
			memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), cBlockObject.GetData()[wBlockSize - 1]);
			wResult += cBlockObject.GetData()[wBlockSize - 1];
		}
	}
	return wResult;
}

CECBCTSCryptographer::CECBCTSCryptographer(CBlockObject &cBlock, CBlockKey &cKey) : cBlockObject(cBlock), cBlockKey(cKey)
{
}

CECBCTSCryptographer::~CECBCTSCryptographer()
{
	memset(bLastBlock, 0, sizeof(bLastBlock));
}

Word CECBCTSCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	Word wBlockSize = cBlockObject.GetBlockSize();
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize;
	Word i, wSteal = wBlockSize - wRemainder;
	for (i = 0; i < wFullBlocks; i++) {
		cBlockObject.SetData(pbData + i * wBlockSize, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	if (0 != wRemainder) {
		memcpy(bLastBlock, pbData + i * wBlockSize, wRemainder);
		memcpy(bLastBlock + wRemainder, pbStorage + i * wBlockSize - wSteal, wSteal);
		cBlockObject.SetData(bLastBlock, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, pbStorage + (i - 1) * wBlockSize, wRemainder);
		memcpy(pbStorage + (i - 1) * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	return wFullBlocks * wBlockSize + wRemainder;
}

Word CECBCTSCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	Word wBlockSize = cBlockObject.GetBlockSize();
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize;
	Word i, wSteal = wBlockSize - wRemainder;
	for (i = 0; i < wFullBlocks; i++) {
		cBlockObject.SetData(pbData + i * wBlockSize, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	if (0 != wRemainder) {
		memcpy(bLastBlock, pbData + i * wBlockSize, wRemainder);
		memcpy(bLastBlock + wRemainder, pbStorage + i * wBlockSize - wSteal, wSteal);
		cBlockObject.SetData(bLastBlock, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, pbStorage + (i - 1) * wBlockSize, wRemainder);
		memcpy(pbStorage + (i - 1) * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	return wFullBlocks * wBlockSize + wRemainder;
}

CCBCCryptographer::CCBCCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV) : cBlockObject(cBlock), cBlockKey(cKey)
{
	wBlockSize = cBlockObject.GetBlockSize();
	memcpy(bIV, pbIV, cBlockObject.GetBlockSize());
}

CCBCCryptographer::~CCBCCryptographer()
{
	memset(bFeedBack, 0, sizeof(bFeedBack));
	memset(bBuffer, 0, sizeof(bBuffer));
	memset(bLastBlock, 0, sizeof(bLastBlock));
	memset(bIV, 0, sizeof(bIV));
}

void CCBCCryptographer::SetIV(const Byte *pbIV)
{
	memcpy(bIV, pbIV, wBlockSize);
}

Word CCBCCryptographer::EncryptData(Byte *pbData, Word wLength, bool fgPad, bool fgCopy)
{
	return EncryptData(pbData, pbData, wLength, fgPad, fgCopy);
}

Word CCBCCryptographer::DecryptData(Byte *pbData, Word wLength, bool fgPad)
{
	return DecryptData(pbData, pbData, wLength, fgPad);
}

Word CCBCCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength, bool fgPad, bool fgCopy)
{
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize;
	Word i;
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(bBuffer, pbData + i * wBlockSize, wBlockSize);
		for (Word l = 0; l < wBlockSize; l++)
			bBuffer[l] ^= ((i == 0) ? bIV[l] : pbStorage[(i - 1) * wBlockSize + l]);
		cBlockObject.SetData(bBuffer, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	if (true == fgPad) {
		memcpy(bLastBlock, pbData + i * wBlockSize, wRemainder);
		memset(bLastBlock + wRemainder, (Byte)wRemainder, wBlockSize - wRemainder);
		for (Word l = 0; l < wBlockSize; l++)
			bLastBlock[l] ^= ((i == 0) ? bIV[l] : pbStorage[(i - 1) * wBlockSize + l]);
		cBlockObject.SetData(bLastBlock, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		if (true == fgCopy)
			memcpy(pbStorage + i * wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	return wFullBlocks * wBlockSize + ((fgPad && fgCopy) ? wBlockSize : 0);
}

Word CCBCCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength, bool fgPad)
{
	Word wFullBlocks = wLength / wBlockSize;
	Word i, wResult = 0;
	memcpy(bFeedBack, bIV, wBlockSize);
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(bFeedBack + wBlockSize, pbData + i * wBlockSize, wBlockSize);
		cBlockObject.SetData(bFeedBack + wBlockSize, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
		for (Word l = 0; l < wBlockSize; l++)
			bBuffer[l] ^= bFeedBack[l];
		memcpy(bFeedBack, bFeedBack + wBlockSize, wBlockSize);
		if (i != wFullBlocks - 1 || false == fgPad) {
			memcpy(pbStorage + i * wBlockSize, bBuffer, wBlockSize);
			wResult += wBlockSize;
		} else {
			memcpy(pbStorage + i * wBlockSize, bBuffer, bBuffer[wBlockSize - 1]);
			wResult += bBuffer[wBlockSize - 1];
		}
	}
	return wResult;
}

CCBCCTSCryptographer::CCBCCTSCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV) : cBlockObject(cBlock), cBlockKey(cKey)
{
	wBlockSize = cBlockObject.GetBlockSize();
	memcpy(bIV, pbIV, cBlockObject.GetBlockSize());
}

CCBCCTSCryptographer::~CCBCCTSCryptographer()
{
	memset(bFeedBack, 0, sizeof(bFeedBack));
	memset(bBuffer, 0, sizeof(bBuffer));
	memset(bLastBlock, 0, sizeof(bLastBlock));
	memset(bIV, 0, sizeof(bIV));
}

void CCBCCTSCryptographer::SetIV(const Byte *pbIV)
{
	memcpy(bIV, pbIV, wBlockSize);
}

Word CCBCCTSCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize;
	for (Word i = 0; i < wFullBlocks; i++, pbStorage += wBlockSize, pbData += wBlockSize) {
		memcpy(bBuffer, pbData, wBlockSize);
		for (Word l = 0; l < wBlockSize; l++)
			bBuffer[l] ^= ((i == 0) ? bIV[l] : (pbStorage - wBlockSize)[l]);
		cBlockObject.SetData(bBuffer, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage, cBlockObject.GetData(), wBlockSize);
	}
	if (0 != wRemainder) {
		memcpy(bBuffer, pbStorage - wBlockSize, wBlockSize);
		for (Word l = 0; l < wRemainder; l++)
			bBuffer[l] ^= pbData[l];
		cBlockObject.SetData(bBuffer, wBlockSize);
		cBlockObject.Encrypt(cBlockKey);
		memcpy(pbStorage, pbStorage - wBlockSize, wRemainder);
		memcpy(pbStorage - wBlockSize, cBlockObject.GetData(), wBlockSize);
	}
	return wLength;
}

Word CCBCCTSCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	Word wFullBlocks = wLength / wBlockSize;
	Word wRemainder = wLength % wBlockSize, l;
	memcpy(bFeedBack, bIV, wBlockSize);
	for (Word i = 0; i < wFullBlocks; i++, pbStorage += wBlockSize, pbData += wBlockSize) {
		memcpy(bFeedBack + wBlockSize, pbData, wBlockSize);
		cBlockObject.SetData(pbData, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
		if ((i != wFullBlocks - 1) || (wLength == wBlockSize)) {
			for (l = 0; l < wBlockSize; l++)
				bBuffer[l] ^= bFeedBack[l];
			memcpy(bFeedBack, bFeedBack + wBlockSize, wBlockSize);
			memcpy(pbStorage, bBuffer, wBlockSize);
		}
	}
	if (0 != wRemainder) {
		for (l = 0; l < wRemainder; l++)
			bLastBlock[l] = bBuffer[l] ^ pbData[l];

		memcpy(bBuffer, pbData, wRemainder);
		cBlockObject.SetData(bBuffer, wBlockSize);
		cBlockObject.Decrypt(cBlockKey);
		memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
		for (l = 0; l < wBlockSize; l++)
			bBuffer[l] ^= bFeedBack[l];
		memcpy(pbStorage - wBlockSize, bBuffer, wBlockSize);
		memcpy(pbStorage, bLastBlock, wRemainder);
	}

	return wLength;
}

CFBCryptographer::CFBCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV, Word wFBS) : cBlockObject(cBlock), cBlockKey(cKey)
{
	wBlockSize = cBlock.GetBlockSize();
	wFeedBackSize = (0 == wFBS ? wBlockSize : wFBS);
	wFeedBackPointer = wFeedBackSize;
	memcpy(bIV, pbIV, wBlockSize);
	memcpy(bRegister, pbIV, wBlockSize);
}

CFBCryptographer::~CFBCryptographer()
{
	memset(bIV, 0, sizeof(bIV));
	memset(bRegister, 0, sizeof(bRegister));
}

void CFBCryptographer::SetIV(const Byte *pbIV)
{
	memcpy(bIV, pbIV, wBlockSize);
}

CCFBCryptographer::CCFBCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV, Word wFBS) : CFBCryptographer(cBlock, cKey, pbIV, wFBS)
{
}

Word CCFBCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	for (Word i = 0; i < wLength; i++, wFeedBackPointer++) {
		if (wFeedBackPointer == wFeedBackSize) {
			cBlockObject.SetData(bRegister, wBlockSize);
			cBlockObject.Encrypt(cBlockKey);
			memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
			wFeedBackPointer = 0;
		}
		pbStorage[i] = pbData[i] ^ bBuffer[wFeedBackPointer];
		for (Word j = 1; j < wBlockSize; j++)
			bRegister[j - 1] = bRegister[j];
		bRegister[wBlockSize - 1] = pbStorage[i];
	}
	
	return wLength;
}

Word CCFBCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	for (Word i = 0; i < wLength; i++, wFeedBackPointer++) {
		if (wFeedBackPointer == wFeedBackSize) {
			cBlockObject.SetData(bRegister, wBlockSize);
			cBlockObject.Encrypt(cBlockKey);
			memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
			wFeedBackPointer = 0;
		}
		for (Word j = 1; j < wBlockSize; j++)
			bRegister[j - 1] = bRegister[j];
		bRegister[wBlockSize - 1] = pbStorage[i];
		pbStorage[i] = pbData[i] ^ bBuffer[wFeedBackPointer];
	}
	
	return wLength;
}

COFBCryptographer::COFBCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV, Word wFBS) : CFBCryptographer(cBlock, cKey, pbIV, wFBS)
{
}

Word COFBCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	for (Word i = 0; i < wLength; i++, wFeedBackPointer++) {
		if (wFeedBackPointer == wFeedBackSize) {
			if (0 != i) {
				memmove(bRegister, bRegister + wBlockSize, wFeedBackSize - wFeedBackSize);
				memcpy(bRegister + wBlockSize - wFeedBackSize, bBuffer, wFeedBackSize);
			}
			cBlockObject.SetData(bRegister, wBlockSize);
			cBlockObject.Encrypt(cBlockKey);
			memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
			wFeedBackPointer = 0;
		}
		pbStorage[i] = pbData[i] ^ bBuffer[wFeedBackPointer];
	}
	
	return wLength;
}

Word COFBCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	return EncryptData(pbData, pbStorage, wLength);
}

COFBCCryptographer::COFBCCryptographer(CBlockObject &cBlock, CBlockKey &cKey, const Byte *pbIV, Word wFBS) : CFBCryptographer(cBlock, cKey, pbIV, wFBS)
{
}

Word COFBCCryptographer::EncryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	for (Word i = 0; i < wLength; i++, wFeedBackPointer++) {
		if (wFeedBackPointer == wFeedBackSize) {
			if (0 != i)
				for (Word j = wBlockSize - 1; j < wBlockSize; j--) {
					bBuffer[j] += 1;
					if (bBuffer[j] != 0)
						break;
				}
			cBlockObject.SetData(bRegister, wBlockSize);
			cBlockObject.Encrypt(cBlockKey);
			memcpy(bBuffer, cBlockObject.GetData(), wBlockSize);
			wFeedBackPointer = 0;
		}
		pbStorage[i] = pbData[i] ^ bBuffer[wFeedBackPointer];
	}
	
	return wLength;
}

Word COFBCCryptographer::DecryptData(const Byte *pbData, Byte *pbStorage, Word wLength)
{
	return EncryptData(pbData, pbStorage, wLength);
}
