/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#define MIN_PARAMS 0

#include "config.h"
#include "getdef.hpp"
#else
#include "getpass.hpp"

#define MIN_PARAMS 1
#endif

#include "passwddio.hpp"
#include "globals.hpp"
#include "strings.hpp"
#include "getarg.h"
#include "conf.hpp"
#include "rsa.hpp"

int iMax = -2;
int iMin = -2;
int iWarn = -2;
int iInact = -2;

bool fgGroup = false;
bool fgEncryptedPasswords;

// Function prototypes
int option_value(char *, bool *, int, int, char *);

char szConfigFile[256] = DEFAULT_CLIENTCONFFILE;

void usage()
{
#ifdef WIN32
	printf("Usage: passwdc <target> [<user>] [-c <file>]\n"
#else
	printf("Usage: passwdc [<target>] [<user>] [-c <file>]\n"
#endif
	       "       [-x <max>] [-n <min>] [-w <warn>] [-i <inact>]\n"
	       "       passwdc <target> [<user>] -g\n"
	       "Where:\n"
	       "  <target>     The user whose password you want to change.\n"
	       "  <login>      Your login if you are master user for the given target.\n"
	       "Options:\n"
	       "  -c <file>    Use alternate config file instead the default\n"
	       "               " DEFAULT_CLIENTCONFFILE " system wide config file.\n"
	       "  -x <max>     Sets the maximum number of days a password remains valid.\n"
	       "  -n <min>     Sets the minimum number of days before a password may be\n"
	       "               changed.\n"
	       "  -w <warn>    Sets the number of days before a password expiration an user\n"
	       "               will receive a warning.\n"
	       "  -i <inact>   Sets the number of days after password expiration, the\n"
	       "               account will be marked as inactive.\n"
	       "  -g           Change group password.\n"
	       "Written by: Alexander Feldman <alex@varna.net>\n\n");
}

int main(int argc, char **argv)
{
	CStringArray cFailed;
	
	static char szServers[512];
	
	static char szPassword[128];
	static char szNewPassword1[128];
	static char szNewPassword2[128];
	
	struct sockaddr_in client;
	struct hostent *pHost;
	
	char *pszLogin = (char *)NULL;
	char *pszTarget = (char *)NULL;
	char *pszServer;
	char *pszPassword;
	char *pszNewPassword1;
	char *pszNewPassword2;
	char *pszObscure = NULL;
	
	int sockfd;
	int passed = -1;
	
	int iMaxLen = 8;
	int iMinLen = 5;
	
	int iServerPort;
	
	int iResult = 0;
	
	char *pszConfig = getenv("PASSWDCCONF");
	
	bool fgError = false;
	
	int w;
	while (-1 != (w = getopt(argc, argv, "c:x:n:w:i:g")))
		switch (w) {
			case 'c':
				strncpy(szConfigFile, optarg, sizeof(szConfigFile));
			break;
			case 'x':
				iMax = option_value(optarg, &fgError, 0, 99999, "-x");
			break;
			case 'n':
				iMin = option_value(optarg, &fgError, 0, 99999, "-n");
			break;
			case 'w':
				iWarn = option_value(optarg, &fgError, 0, 99999, "-w");
			break;
			case 'i':
				iInact = option_value(optarg, &fgError, 0, 99999, "-i");
			break;
			case 'g':
				fgGroup = true;
			break;
			default:
				usage();
				return 0;
			break;
		}
	argc -= optind;
	argv += optind;
	
	if ((argc < MIN_PARAMS) || (argc > 2) || (true == fgError) || ((true == fgGroup) && (argc == 0))) {
		usage();
		return 0;
	}
	if (argc >= 1)
		pszTarget = argv[0];
	if (argc == 2)
		pszLogin = argv[1];
	
#ifdef WIN32
	GetWindowsDirectory(szConfigFile, sizeof(szConfigFile));
	strncat(szConfigFile, SEPARATOR DEFAULT_CLIENTCONFFILE, sizeof(szConfigFile));
#endif
	if (NULL == pszConfig)
		pszConfig = szConfigFile;

	char *pszMessage = NULL;
	CConfigurationFile *pConfigFile = new CConfigurationFile(pszConfig, &pszMessage);
	if (NULL == pConfigFile) {
		fprintf(stderr, "Error allocating memory...\n");
		return 1;
	}
	if (NULL != pszMessage) {
		fprintf(stderr, "%s\n", pszMessage);
		delete pConfigFile;
		return 1;
	}

#ifdef WIN32
	WSADATA WSAData;
	int iStatus;
	if (0 != (iStatus = WSAStartup(MAKEWORD(2, 2), &WSAData))) {
		printf("Socket init error: %s\n%s.\n", WSAData.szDescription, WSAData.szSystemStatus);
		return 1;
	}
#endif

	iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
	memzero(szServers);
	strncpy(szServers, pConfigFile->GetString("Global", "targets", "localhost"), sizeof(szServers) - 1);

#ifndef WIN32
	if (NULL == pszLogin)
//		pszLogin = getlogin();				// This returns NULL in some glibc
		pszLogin = getenv("LOGNAME");		// Thanks to Mark Siegler <tech@netalia.com>

	if (NULL == pszLogin) {
		fprintf(stderr, "Error getting login from the environment...\n");
		return 1;
	}
	if (NULL == pszTarget)
		pszTarget = pszLogin;
#else
	if (NULL == pszLogin)
		pszLogin = pszTarget;
#endif
	
	pszPassword = getpass("Password:");
	strcpy(szPassword, pszPassword);
	strzero(pszPassword);

// In fact we don't need to check for login lengths on the source machine but
// on the target but it always better to have sample restrictions for the
// min and max lengths of a string than not to have.
#if !defined(WIN32) && defined(LOGIN_DEFS)
	iMinLen = getdef_num("PASS_MIN_LEN", 5);
	iMaxLen = getdef_num("PASS_MAX_LEN", 32);
#endif
	bool fgCorrect = false;
	for (int i = 0; i < 3; i++) {
		printf("\nEnter the new password (minimum of %d, maximum of %d characters)\n", iMinLen, iMaxLen);
		printf("Please use combination of upper and lower case letters and numbers.\n");
	
		pszNewPassword1 = getpass("New password:");
		strcpy(szNewPassword1, pszNewPassword1);	
		strzero(pszNewPassword1);
		pszNewPassword2 = getpass("Re-enter new password:");
		strcpy(szNewPassword2, pszNewPassword2);
		strzero(pszNewPassword2);
	
// We can skip overflow and error checking from getpass(), because in all
// cases it returns valid pointer and the internal buffer is 128 bytes and
// null terminated.	
	
		if (0 != strcmp(szNewPassword1, szNewPassword2)) {
			printf("They don't match; try again.\n");
			continue;
		}
	
		if ((int)strlen(szNewPassword1) < iMinLen ||
			(int)strlen(szNewPassword1) > iMaxLen) {
			printf("They don't meet length criteria; try again.\n");
			continue;
		}
		fgCorrect = true;
		break;
	}
	printf("\n");

	pszServer = strtok(szServers, ";,");
	do {
		if (false == fgCorrect)
			break;
		
		bool fgPassed = false;
		
		pszServer += strspn(pszServer, " \t");			
		printf("Changing password for %s on %s.\n", pszTarget, pszServer);

		if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
			fprintf(stderr, "Socket error: %s\n", strerror(errno));
			cFailed.AddString(pszServer);
			continue;
		}

		pHost = gethostbyname(pszServer);
		if (NULL == pHost) {
			fprintf(stderr, "Resolving error: %s\n", pszServer);
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}
		memset(&client, 0, sizeof(client));
		client.sin_family = AF_INET;
		client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
		client.sin_port = htons(iServerPort);
		if (connect(sockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
			fprintf(stderr, "Connect error: %s\n", strerror(errno));
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}

		unsigned char c = '\0';
		if (1 != read_byte(sockfd, &c)) {
			fprintf(stderr, "Recv error.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}
		if (WELCOME != c) {
			fprintf(stderr, "Invalid response from server.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}
		if (1 != read_integer(sockfd, (int *)&fgEncryptedPasswords)) {
			fprintf(stderr, "Recieve data error.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		}

		if (true == fgEncryptedPasswords) {
			char szPublicKey[256];
			snprintf(szPublicKey,
					 sizeof(szPublicKey),
					 "%s/%s_%d.pub",
					 pConfigFile->GetString("Security", "key_directory", "/var/spool/passwdc"),
					 pszServer,
					 iServerPort);
			int i = recv_remote_key(sockfd, szPublicKey);
			if (1 == i)
				fprintf(stderr, "Error allocating memory.\n");
			if (2 == i)
				fprintf(stderr, "Receive public key error.\n");
			if (3 == i)
				fprintf(stderr, "Read public key error.\n");
			if (4 == i)
				fprintf(stderr, "Public key mismatch! Please remove manually the file '%s' if this is not an attack.\n", szPublicKey);
			if (5 == i)
				fprintf(stderr, "Write public key error.\n");
			if (0 != i) {
				write_byte(sockfd, FAILURE);
				close(sockfd);
				cFailed.AddString(pszServer);
				continue;
			}
			write_byte(sockfd, OK);
		}

		if (((int)strlen(pszLogin) + 1 != write_string(sockfd, pszLogin)) ||
			(false == write_password(sockfd, szPassword))) {
			fprintf(stderr, "Write data error.\n");
			close(sockfd);
			cFailed.AddString(pszServer);
			continue;
		} 
		read_byte(sockfd, &c);			
		switch (c) {
			case GO_AHEAD:
				write_byte(sockfd, fgGroup ? ACTION_CHANGEGROUPPASSWORD : ACTION_CHANGEPASSWORD);
				if (((int)strlen(pszTarget) + 1 != write_string(sockfd, pszTarget)) ||
					 (0 == write_password(sockfd, szNewPassword1)))
					fprintf(stderr, "Write error.\n");
				if (!fgGroup)
					if ((false == write_integer(sockfd, iMin)) ||
						(false == write_integer(sockfd, iMax)) ||
						(false == write_integer(sockfd, iWarn)) ||
						(false == write_integer(sockfd, iInact)))
						fprintf(stderr, "Write error.\n");
				read_byte(sockfd, &c);
				switch (c) {
					case OK:
						printf("Password successfully changed.\n");
						fgPassed = true;
					break;
					case INVALID_USER:
						fprintf(stderr, "Invalid target specified.\n");
					break;
					case FAILURE:
						fprintf(stderr, "Password still unchanged. Server returned error.\n");
					break;
					case NO_RIGHTS:
						fprintf(stderr, "Access denied.\n");
					break;
					case WEAK_PASSWORD:
						if (-1 != read_string(sockfd, &pszObscure))
							fprintf(stderr, "Weak password: %s.\n", pszObscure);
						if (NULL != pszObscure)
							free(pszObscure);
					break;
					default:
						fprintf(stderr, "Invalid response from server.\n");
				}
			break;
			case LOGIN_FAILED:
				fprintf(stderr, "Incorrect login or password for %s on %s.\n", pszLogin, pszServer);				
			break;
			default:
				fprintf(stderr, "Invalid response from server.\n");
		}
		if (true == fgPassed) {
			if (-1 != passed) {
				write_string(passed, "");
				close(passed);
			}
			passed = sockfd;
		} else {
			cFailed.AddString(pszServer);
			close(sockfd);
		}
	} while (NULL != (pszServer = strtok(NULL, ";,")));

	if (-1 != passed) {
		char *pszFailed;
		if ((pszFailed = cFailed.GetFirstString()))
			do {
				write_string(passed, pszFailed);
			} while ((pszFailed = cFailed.GetNextString()));
		write_string(passed, "");
		close(passed);
	} else {
		fprintf(stderr,
				"\nThe password of user %s could not be changed on no one server.\n"
				"Therefore no further attempts for automatic change of the password will be made.\n",
				pszTarget);
	}

	strzero(szPassword);
	strzero(szNewPassword1);
	strzero(szNewPassword2);

#ifdef WIN32
	WSACleanup();
#endif

	delete pConfigFile;

	destroy_remote_key();

	return iResult;
}

int option_value(char *pszValue, bool *pfgError, int iMin, int iMax, char *pszInfo)
{
	int iResult = 0;
	
	if ((NULL == pszValue) || ('\0' == *pszValue)) {
		fprintf(stderr, "Invalid value for option: %s...\n", pszInfo);
		*pfgError = true;
		return iResult;
	}
	char *pszInvalid;
	iResult = strtol(pszValue, &pszInvalid, 10);
	if ('\0' != *pszInvalid) {
		fprintf(stderr, "Invalid value for option: %s...\n", pszInfo);
		*pfgError = true;
	}
	if (iResult < iMin || iResult > iMax) {
		fprintf(stderr, "Out of range value for option: %s...\n", pszInfo);
		*pfgError = true;
	}
	
	return iResult;
}
