/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  grg_widgets.c - various "custom" widgets used in Gringotts
 *  Author: Germano Rizzo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <string.h>

#include <gtk/gtk.h>

#include "grg_print.h"
#include "grg_defs.h"
#include "grg_utils.h"

/**
 * grg_msg:
 * @text: the text of the message to display
 * @msgt: the GtkMessageType of the message
 * @parent: the parent of the dialog to create
 *
 * Displays a dialog that shows a message.
 */
void grg_msg(gchar * text, GtkMessageType msgt, GtkWidget * parent)
{
    GtkWidget *dialog =
	gtk_message_dialog_new(GTK_WINDOW(parent), GTK_DIALOG_MODAL, msgt,
			       GTK_BUTTONS_OK, text);

    gtk_dialog_set_default_response(GTK_DIALOG(dialog),
				    GTK_RESPONSE_ACCEPT);
    gtk_dialog_run(GTK_DIALOG(dialog));

    gtk_widget_destroy(dialog);
}

/**
 * grg_input_dialog:
 * @title: the title of the input dialog
 * @qtext: the question to ask
 * @preset: the predefined answer, NULL if none
 * @is_pass: tells if the requested string is a password, to be obfuscated
 * @parent: the parent of the dialog to create
 *
 * Shows and manages a dialog that asks for a string.
 *
 * Returns: the answer string, or NULL if user pressed Cancel.
 */
gchar *grg_input_dialog(gchar * title, gchar * qtext, gchar * preset,
			gboolean is_pass, GtkWidget * parent)
{
    GtkWidget *dialog, *question, *label;
    gchar *ret;
    gint res;

    dialog =
	gtk_dialog_new_with_buttons(title, GTK_WINDOW(parent),
				    GTK_DIALOG_MODAL, GTK_STOCK_OK,
				    GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL,
				    GTK_RESPONSE_REJECT, NULL);

    gtk_dialog_set_default_response(GTK_DIALOG(dialog),
				    GTK_RESPONSE_ACCEPT);

    gtk_container_set_border_width(GTK_CONTAINER(dialog), GRG_PAD);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), GRG_PAD);

    label = gtk_label_new(qtext);

    question = gtk_entry_new_with_max_length(32);
    if (preset != NULL)
	gtk_entry_set_text(GTK_ENTRY(question), preset);
    gtk_entry_set_visibility(GTK_ENTRY(question), !is_pass);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
				question);
    gtk_widget_show_all(dialog);

    res = gtk_dialog_run(GTK_DIALOG(dialog));

    if (res == GTK_RESPONSE_ACCEPT)
	ret = g_strdup((gchar *) gtk_entry_get_text(GTK_ENTRY(question)));
    else
	ret = NULL;

    gtk_widget_destroy(dialog);

    return ret;
}

/**
 * grg_new_pwd_dialog:
 * @parent: the parent of the dialog to create.
 *
 * Asks for a new password, validating it (>4 chars).
 *
 * Returns: the new password, or NULL if user Cancel-ed
 */
gchar *grg_new_pwd_dialog(GtkWidget * parent)
{
    GtkWidget *dialog, *question, *label, *label2, *question2;
    gchar *ret;

    dialog =
	gtk_dialog_new_with_buttons(_("New password"), GTK_WINDOW(parent),
				    GTK_DIALOG_MODAL, GTK_STOCK_OK,
				    GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL,
				    GTK_RESPONSE_REJECT, NULL);

    gtk_dialog_set_default_response(GTK_DIALOG(dialog),
				    GTK_RESPONSE_ACCEPT);

    gtk_container_set_border_width(GTK_CONTAINER(dialog), 3);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), 3);

    label = gtk_label_new(_("Enter new password:"));
    label2 = gtk_label_new(_("Enter it again for confirmation:"));

    question = gtk_entry_new_with_max_length(32);
    gtk_entry_set_visibility(GTK_ENTRY(question), FALSE);

    question2 = gtk_entry_new_with_max_length(32);
    gtk_entry_set_visibility(GTK_ENTRY(question2), FALSE);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
				question);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label2);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
				question2);

    gtk_widget_show_all(dialog);

    do {
	gint res = gtk_dialog_run(GTK_DIALOG(dialog));

	if (res == GTK_RESPONSE_ACCEPT) {
	    gchar *ret1, *ret2;

	    ret1 = g_strdup((gchar *)
			    gtk_entry_get_text(GTK_ENTRY(question)));
	    ret2 = g_strdup((gchar *)
			    gtk_entry_get_text(GTK_ENTRY(question2)));

	    if (strcmp(ret1, ret2)) {
		grg_msg(_("The two passwords are different"),
			GTK_MESSAGE_ERROR, dialog);

		gtk_entry_set_text(GTK_ENTRY(question), "");
		gtk_entry_set_text(GTK_ENTRY(question2), "");

		grg_free(ret1, -1);
		grg_free(ret2, -1);
		continue;
	    }

	    if (strlen(ret1) < 4) {
		grg_msg
		    (_
		     ("The password is too short, it must be at least 4 chars"),
		     GTK_MESSAGE_ERROR, dialog);

		gtk_entry_set_text(GTK_ENTRY(question), "");
		gtk_entry_set_text(GTK_ENTRY(question2), "");

		grg_free(ret1, -1);
		grg_free(ret2, -1);
		continue;
	    }

	    ret = g_strdup(ret1);

	    grg_free(ret1, -1);
	    grg_free(ret2, -1);

	    break;
	} else {
	    ret = NULL;
	    break;
	}
    }
    while (TRUE);

    gtk_widget_destroy(dialog);

    return ret;
}

/**
 * grg_ask_dialog:
 * @title: the title of the dialog
 * @question: the question to ask
 * @parent: the parent of the dialog to create.
 *
 * Asks a question for a boolean answer.
 *
 * Returns: TRUE if yes, FALSE if no.
 */
gboolean
grg_ask_dialog(gchar * title, gchar * question, GtkWidget * parent)
{
    GtkWidget *dialog, *label;
    gboolean ret;
    gint res;

    dialog =
	gtk_dialog_new_with_buttons(title, GTK_WINDOW(parent),
				    GTK_DIALOG_MODAL, GTK_STOCK_YES,
				    GTK_RESPONSE_YES, GTK_STOCK_NO,
				    GTK_RESPONSE_NO, NULL);

    gtk_dialog_set_default_response(GTK_DIALOG(dialog),
				    GTK_RESPONSE_ACCEPT);

    gtk_container_set_border_width(GTK_CONTAINER(dialog), GRG_PAD);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), GRG_PAD);

    label = gtk_label_new(question);

    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);

    gtk_widget_show_all(dialog);

    res = gtk_dialog_run(GTK_DIALOG(dialog));

    if (res == GTK_RESPONSE_YES)
	ret = TRUE;
    else
	ret = FALSE;

    gtk_widget_destroy(dialog);

    return ret;
}

/**
 * grg_window_set_icon:
 * @w: the window to set the icon to
 *
 * Associates the Gringotts icon with the given window.
 */
void grg_window_set_icon(GtkWindow * w)
{
    GdkPixbuf *gp = NULL;
    gp = gdk_pixbuf_new_from_file(PIX_DIR "gringotts.xpm", NULL);
    if (gp != NULL) {
	GList *gl = NULL;

	g_list_append(gl, gp);
	gtk_window_set_icon(w, gp);
	gtk_window_set_default_icon_list(gl);

	g_list_free(gl);
	g_object_unref(G_OBJECT(gp));
    } else {
	gchar *out;

	out =
	    g_strdup_printf(_("Icon file %s not found"),
			    PIX_DIR "gringotts.xpm");
	grg_print_warn(out);
	g_free(out);
    }
}
