/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  grg_entries.c - functions to manage the GList of Gringotts' entries
 *  Author: Germano Rizzo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <gtk/gtktextiter.h>
#include <gtk/gtktextbuffer.h>
#include <stdlib.h>

#include "grg_defs.h"
#include "grg_crypt.h"
#include "grg_utils.h"
#include "gringotts.h"

struct grg_entry {
    gchar *entryID;
    GtkTextBuffer *entryBody;
};

GList *entries = NULL;
GList *current = NULL;
guchar *serialized;

/**
 * meta_free:
 * @data: the callback's data
 * @user_data: the callback's user-defined data
 *
 * Frees a single node. Used only by grg_entries_free()
 */
void meta_free(gpointer data, gpointer user_data)
{
    struct grg_entry *entry = (struct grg_entry *) data;

    grg_free(entry->entryID, -1);
    g_free(entry);
}

/**
 * grg_entries_append:
 * @ID: the title of the new entry
 *
 * Appends a new entry to the list
 */
void grg_entries_append(gchar * ID)
{
    struct grg_entry *entry;
    GtkTextTagTable *gttt = gtk_text_tag_table_new();

    entry = malloc(sizeof(struct grg_entry));
    entry->entryID = g_strdup(ID);

    entry->entryBody = gtk_text_buffer_new(gttt);
    g_signal_connect(G_OBJECT(entry->entryBody), "changed",
		     G_CALLBACK(meta_saveable),
		     GINT_TO_POINTER(GRG_SAVE_ACTIVE));

    entries = g_list_append(entries, entry);
    current = g_list_last(entries);
}

/**
 * grg_entries_remove:
 *
 * Removes the current entry and frees it
 */
void grg_entries_remove()
{
    gint pos;

    if (current == NULL)
	return;

    pos = g_list_position(entries, current);
    meta_free(current->data, NULL);
    entries = g_list_remove_link(entries, current);
    g_list_free_1(current);
    current = g_list_nth(entries, pos);
    if (current == NULL)
	current = g_list_nth(entries, pos - 1);
}

/**
 * grg_entries_is_first:
 *
 * Tells if the current entry is the first one
 *
 * Returns: GRG_TRUE if true, GRG_FALSE if false.
 */
gboolean grg_entries_is_first()
{
    if (current == NULL)
	return GRG_TRUE;
    if (g_list_previous(current) == NULL)
	return GRG_TRUE;
    return GRG_FALSE;
}

/**
 * grg_entries_is_last:
 *
 * Tells if the current entry is the last one
 *
 * Returns: GRG_TRUE if true, GRG_FALSE if false.
 */
gboolean grg_entries_is_last()
{
    if (current == NULL)
	return GRG_TRUE;

    if (g_list_next(current) == NULL)
	return GRG_TRUE;

    return GRG_FALSE;
}

/**
 * grg_entries_is_empty:
 *
 * Tells if the current list is empty
 *
 * Returns: GRG_TRUE if true, GRG_FALSE if false.
 */
gboolean grg_entries_is_empty()
{
    return (entries == NULL);
}

/**
 * grg_entries_first:
 *
 * Goes to the first entry
 */
void grg_entries_first()
{
    current = g_list_first(entries);
}

/**
 * grg_entries_prev:
 *
 * Goes to the previous entry
 */
void grg_entries_prev()
{
    if (!grg_entries_is_first())
	current = g_list_previous(current);
}

/**
 * grg_entries_next:
 *
 * Goes to the next entry
 */
void grg_entries_next()
{
    if (!grg_entries_is_last())
	current = g_list_next(current);
}

/**
 * grg_entries_last:
 *
 * Goes to the last entry
 */
void grg_entries_last()
{
    current = g_list_last(entries);
}

/**
 * grg_entries_nth:
 * @pos: the position to go to
 *
 * Goes to the specified entry, if possible
 */
void grg_entries_nth(gint pos)
{
    current = g_list_nth(entries, pos);
}

/**
 * grg_entries_position:
 *
 * Tells the position of the current entry in the list
 *
 * Returns: a gint with the position
 */
gint grg_entries_position()
{
    if (current == NULL)
	return -1;

    return g_list_position(entries, current);
}

/**
 * grg_entries_raise
 *
 * Shifts one position up
 */
void grg_entries_raise()
{
    gpointer ent;

    if (!current->prev)
	return;

    ent = current->data;
    current->data = current->prev->data;
    current->prev->data = ent;
}

/**
 * grg_entries_sink
 *
 * Shifts one position down
 */
void grg_entries_sink()
{
    gpointer ent;

    if (!current->next)
	return;

    ent = current->data;
    current->data = current->next->data;
    current->next->data = ent;
}

/**
 * grg_entries_get_ID
 *
 * Returns the title of the current entry.
 *
 * Returns: a gchar* with the title (NOT a copy), or NULL if the list is empty
 */
gchar *grg_entries_get_ID()
{
    if (entries == NULL)
	return NULL;
    else
	return ((struct grg_entry *) current->data)->entryID;
}

/**
 * grg_entries_get_Body
 *
 * Returns the body of the current entry.
 *
 * Returns: a grg_entry struct with the body, or NULL if the list is empty
 */
GtkTextBuffer *grg_entries_get_Body()
{
    if (entries == NULL)
	return NULL;
    else
	return ((struct grg_entry *) current->data)->entryBody;
}

/**
 * grg_entries_get_Body_text
 *
 * Returns the text contained in the body of the current entry.
 *
 * Returns: a gchar* with the text, or NULL if the list is empty
 */
gchar *grg_entries_get_Body_text()
{
    GtkTextBuffer *buf = NULL;
    static GtkTextIter s, e;

    if (current == NULL)
	return NULL;

    buf = ((struct grg_entry *) current->data)->entryBody;

    gtk_text_buffer_get_bounds(buf, &s, &e);
    return gtk_text_buffer_get_text(buf, &s, &e, FALSE);
}

/**
 * grg_entries_set_ID
 * @ID: the text to store
 *
 * Stores the given title in the current entry.
 */
void grg_entries_set_ID(gchar * ID)
{
    ((struct grg_entry *) current->data)->entryID = g_strdup(ID);
}

/**
 * grg_entries_set_Body
 * @Body: the grg_entry struct with the body to store
 *
 * Stores the given body in the current entry.
 */
void grg_entries_set_Body(GtkTextBuffer * Body)
{
    ((struct grg_entry *) current->data)->entryBody = Body;
}

/**
 * grg_entries_set_Body_text
 * @Body: the text to store
 *
 * Stores the given text in the body of the current entry.
 */
void grg_entries_set_Body_text(gchar * Body)
{
    GtkTextBuffer *buf = ((struct grg_entry *) current->data)->entryBody;

    gtk_text_buffer_set_text(buf, Body, -1);
}

/**
 * grg_entries_free:
 *
 * Deletes and frees all the list
 */
void grg_entries_free()
{
    g_list_foreach(entries, meta_free, NULL);
    g_list_free(entries);
    entries = NULL;
    current = NULL;
}

/**
 * meta_save:
 * @data: the callback's data
 * @user_data: the callback's user-defined data
 *
 * "serializes" a single node. Used only by grg_entries_save()
 */
void meta_save(gpointer data, gpointer user_data)
{
    struct grg_entry *entry = (struct grg_entry *) data;
    gchar *Body, *eBody, *res;
    GtkTextBuffer *buf = entry->entryBody;
    static GtkTextIter s, e;

    gtk_text_buffer_get_bounds(buf, &s, &e);
    Body = gtk_text_buffer_get_text(buf, &s, &e, FALSE);

    eBody = g_strescape(Body, "");

    res = g_strconcat(serialized, "\n", entry->entryID, "\n", eBody, NULL);

    grg_free(Body, -1);
    grg_free(eBody, -1);
    grg_free(serialized, -1);

    serialized = g_strdup(res);

    grg_free(res, -1);
}

/**
 * grg_entries_save:
 * @file: the path of the file to save
 * @pwd: the password to use
 *
 * Saves the list into an encrypted file
 *
 * Returns: GRG_OK if all is well, an error if not
 */
gint grg_entries_save(gchar * file, gchar * pwd)
{
    gint err;

    serialized = g_strdup("");
    g_list_foreach(entries, meta_save, NULL);
    //the "+1" strips off the initial separator
    err = grg_save_crypted(serialized + 1, pwd, file);
    if (err != GRG_OK)
	return err;

    grg_free(serialized, -1);
    serialized = NULL;

    return GRG_OK;
}

/**
 * grg_entries_load_from_string:
 * @str: the string which contains the data
 *
 * "de-serializes" a string into an entry list
 */
void grg_entries_load_from_string(gchar * str)
{
    gchar **fields;
    gint num;

    fields = g_strsplit(str, "\n", 0);

    num = 0;
    while (fields[num * 2] != NULL) {
	gchar *Body;

	grg_entries_append(fields[num * 2]);
	Body = g_strcompress(fields[num * 2 + 1]);
	grg_entries_set_Body_text(Body);

	grg_free(Body, -1);
	num++;
    }

    grg_entries_first();

    g_strfreev(fields);
}

/**
 * grg_load_wrapper:
 * @txt: a pointer to a byte sequence to store the data in. It must be freed after use!
 * @pwd: the password to decode data
 * @file: path of the file to read data from
 * @just_validate: if 1, it only checks for file validity, without loading it.
 * 
 * Wrapper to grg_load_crypted, to add UTF-8 validation.
 *
 * Returns: 0 if OK; an error code otherwise (see grg_crypt.h)
 */
gint grg_load_wrapper(guchar **txt, gchar *pwd, gchar *file, guint just_validate)
{
	gint err;
	
	err = grg_load_crypted(txt, pwd, file, just_validate);
    if (err != GRG_OK)
		return err;
	
	if(just_validate)
		return GRG_OK;

	if(!g_utf8_validate(*txt, -1, NULL))
		return GRG_READ_INVALID_CHARSET_ERR;
	
	return GRG_OK;
}

/**
 * grg_entries_load:
 * @file: the path of the file to load
 * @pwd: the password to use
 *
 * Reads a file into an entry list
 *
 * Returns: GRG_OK if all is well, an error if not
 */
gint grg_entries_load(gchar * file, gchar * pwd)
{
    gint err;

    if (entries != NULL)
	grg_entries_free();

    err = grg_load_wrapper(&serialized, pwd, file, 0);
    if (err != GRG_OK)
	return err;

    grg_entries_load_from_string(serialized);

    grg_free(serialized, -1);
    serialized = NULL;

    return GRG_OK;
}

/**
 * grg_entries_n_el
 *
 * Tells how many elements does the list contain
 *
 * Returns: a guint with the elements' number
 */
guint grg_entries_n_el()
{
    if (entries == NULL)
	return 0;
    else
	return g_list_length(entries);
}
