/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  grg_crypt.c - routines for data encryption and writing
 *  Author: Germano Rizzo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <mcrypt.h>
#include <mhash.h>
#include <zlib.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>

#include "grg_crypt.h"
#include "grg_utils.h"
#include "grg_defs.h"

/**
 * get_key:
 * @pwd: the password to generate hash key from
 *
 * Produces a key, for use with libmcrypt's encryption engine, using the
 * 256-bit SHA hash of a password
 *
 * Returns: a string (KEY_SIZE byte) with the key
 */
unsigned char *
get_key (unsigned char *pwd)
{
	unsigned char *ret;

	ret = calloc (1, GRG_KEY_SIZE);

	mhash_keygen (KEYGEN_MCRYPT, MHASH_SHA256, 0, ret, GRG_KEY_SIZE, NULL,
		      0, pwd, strlen (pwd));

	return ret;
}

/**
 * get_CRC32:
 * @string: a byte sequence
 * @strlen: the length of the string
 *
 * Computes the CRC32 checksum of a byte sequence.
 *
 * Returns: the checksum
 */
unsigned char *
get_CRC32 (unsigned char *string, unsigned long strlen)
{
	MHASH td;
	unsigned char *ret;

	td = mhash_init (MHASH_CRC32);

	if (td == MHASH_FAILED)
		exit (1);

	mhash (td, string, strlen);

	ret = mhash_end (td);

	return ret;
}

/**
 * compare_CRC32:
 * @CRC: the CRC to compare to
 * @toCheck: the byte sequence to compare the CRC to
 * @len: the byte sequence length
 *
 * Tells if a byte sequence has the provided CRC32, in other words if
 * it's equal to the previously CRC'ed one.
 *
 * Returns: TRUE or FALSE
 */
int
compare_CRC32 (unsigned char *CRC, unsigned char *toCheck, unsigned long len)
{
	unsigned char *CRC2;
	int ret;

	CRC2 = get_CRC32 (toCheck, len);

	ret = !memcmp (CRC, CRC2, GRG_CRC_LEN);

	grg_free (CRC2, 4);

	return ret;
}

/**
 * get_IV:
 * @IV_size: the size of the IV to produce
 * @salt: the salt to produce the IV with
 * @salt_size: the size of the salt
 * @pwd: the password of the encrypted data
 *
 * Generates an `IV' token. See libmcrypt docs to know what it is... ;)
 *
 * Returns: a byte sequence with the IV
 */
unsigned char *
get_IV (unsigned int IV_size, unsigned char *salt,
	unsigned int salt_size, unsigned char *pwd)
{
	unsigned char *ret;

	ret = calloc (1, IV_size);

	mhash_keygen (KEYGEN_S2K_SALTED, MHASH_GOST, 0, ret, IV_size, salt,
		      salt_size, pwd, strlen (pwd));

	return ret;
}

/**
 * grg_save_crypted:
 * @origData: a byte sequence to read the data to encrypt and write in the file.
 * @pwd: the password to decode data
 * @fpath: path of the file to read data from
 *
 * Stores data in a Gringotts file
 *
 * Returns: 0 if OK; an error code otherwise (see grg_crypt.h)
 */
int
grg_save_crypted (unsigned char *origData, unsigned char *pwd,
		  unsigned char *fpath)
{
	unsigned char *compData, *chunk, *toCRC1, *CRC1, *toEnc, *key, *IV,
		*toCRC2, *CRC2, *salt;
	unsigned int dIV, err, i, salt_size;
	unsigned long uncDim, compDim;
	FILE *fp;
	MCRYPT mod;

	uncDim = strlen (origData);
	compDim = (unsigned long) ((((float) uncDim) + 12.0) / 100.0 * 101.0);

	compData =
		(unsigned char *) malloc (compDim * sizeof (unsigned char));

	//compress the data
	err = compress2 (compData, &compDim, origData, uncDim,
			 Z_BEST_COMPRESSION);

	if (err < 0)
		return GRG_WRITE_COMP_ERR;

	if (compDim > MAX_COMP_SIZE * 1048576l)
		return GRG_WRITE_TOO_BIG_ERR;

	chunk = grg_long2char (uncDim);

	//adds the CRC32 and DATA_LEN field
	toCRC1 = grg_strconcat (chunk, compData, NULL, GRG_DATA_DIM_LEN,
				compDim, 0);

	grg_free (chunk, GRG_DATA_DIM_LEN);
	grg_free (compData, compDim * sizeof (unsigned char));

	compDim += GRG_DATA_DIM_LEN;

	CRC1 = get_CRC32 (toCRC1, compDim);

	toEnc = grg_strconcat (CRC1, toCRC1, NULL, GRG_CRC_LEN, compDim, 0);

	grg_free (CRC1, GRG_CRC_LEN);
	grg_free (toCRC1, compDim);

	compDim += GRG_CRC_LEN;

	//encrypts the data
	mod = mcrypt_module_open (MCRYPT_SERPENT, NULL, MCRYPT_CFB, NULL);

	if (mod == MCRYPT_FAILED)
	{
		grg_free (toEnc, compDim);
		return GRG_WRITE_ENC_INIT_ERR;
	}

	key = get_key (pwd);
	dIV = mcrypt_enc_get_iv_size (mod);
	salt_size = mhash_get_keygen_salt_size (KEYGEN_S2K_SALTED);
	salt = grg_rnd_seq (salt_size);
	IV = get_IV (dIV, salt, salt_size, pwd);
	err = mcrypt_generic_init (mod, key, GRG_KEY_SIZE, IV);
	grg_free (key, GRG_KEY_SIZE);
	grg_free (IV, dIV);
	if (err < 0)
	{
		grg_free (salt, salt_size);
		grg_free (toEnc, compDim);
		return GRG_WRITE_ENC_INIT_ERR;
	}

	for (i = 0; i < compDim; i++)
		mcrypt_generic (mod, &toEnc[i], 1);

	mcrypt_generic_end (mod);

	//adds algorythm and salt 

	toCRC2 = grg_strconcat ("1", salt, toEnc, GRG_ALGO_LEN, salt_size,
				compDim);

	grg_free (toEnc, compDim);
	grg_free (salt, salt_size);

	compDim += GRG_ALGO_LEN + salt_size;

	//calculates the CRC32

	CRC2 = get_CRC32 (toCRC2, compDim);

	//writes it all

	fp = fopen (fpath, "wb");

	if (fp == NULL)
	{
		grg_free (CRC2, GRG_CRC_LEN);
		grg_free (toCRC2, compDim);
		return GRG_WRITE_FILE_ERR;
	}

	fwrite (GRG_FLAG, GRG_FLAG_LEN, 1, fp);
	fwrite (GRG_FILE_VERSION, GRG_FILE_VERSION_LEN, 1, fp);
	fwrite (CRC2, GRG_CRC_LEN, 1, fp);
	fwrite (toCRC2, compDim, 1, fp);

	err = fclose (fp);

	if (err != 0)
	{
		grg_free (CRC2, GRG_CRC_LEN);
		grg_free (toCRC2, compDim);
		return GRG_WRITE_FILE_ERR;
	}

	sync ();

	//closing
	grg_free (CRC2, GRG_CRC_LEN);
	grg_free (toCRC2, compDim);

	return GRG_OK;
}

/**
 * grg_load_crypted:
 * @origData: a pointer to a byte sequence to store the data in. It must be freed after use!
 * @pwd: the password to decode data
 * @fpath: path of the file to read data from
 * @just_validate: if 1, it only checks for file validity, without loading it.
 * 
 * Loads the encrypted data stored in a Gringotts file, or just check for validity
 *
 * Returns: 0 if OK; an error code otherwise (see grg_crypt.h)
 */
int
grg_load_crypted (unsigned char **origData, unsigned char *pwd,
		  const unsigned char *fpath, unsigned int just_validate)
{
	unsigned char *flag, *CRC32, *CRC32b, *compData, *bcompData, *IV,
		*key, *dimdata, *rawData, *salt;
	unsigned int err, dIV, salt_size;
	unsigned long flen, bflen, origLen, i;
	struct stat sbuf;
	FILE *fp;
	MCRYPT mod;

	salt_size = mhash_get_keygen_salt_size (KEYGEN_S2K_SALTED);

	//reads the file into a suitable buffer
	if (stat (fpath, &sbuf) != 0)
		return GRG_READ_FILE_ERR;

	flen = (unsigned long) sbuf.st_size;

	if (flen < GRG_OVERHEAD + salt_size)	//the total length of header part
		return GRG_READ_FILE_ERR;

	if (flen - GRG_OVERHEAD - salt_size > MAX_COMP_SIZE * 1048576l)
		return GRG_READ_TOO_BIG_ERR;

	fp = fopen (fpath, "rb");

	if (fp == NULL)
		return GRG_READ_FILE_ERR;

	//checks the GRG_FLAG
	flag = (unsigned char *) malloc (GRG_FLAG_LEN *
					 sizeof (unsigned char));

	fread (flag, 1, GRG_FLAG_LEN, fp);

	if (memcmp (GRG_FLAG, flag, GRG_FLAG_LEN))
		return GRG_READ_MAGIC_ERR;

	grg_free (flag, GRG_FLAG_LEN);

	//don't care GRG_VERSION, for now

	fseek (fp, GRG_FILE_VERSION_LEN, SEEK_CUR);

	//checks the 1st CRC

	flen -= GRG_FLAG_LEN + GRG_FILE_VERSION_LEN + GRG_CRC_LEN;

	CRC32 = (unsigned char *) malloc (GRG_CRC_LEN *
					  sizeof (unsigned char));
	compData = (unsigned char *) malloc (flen * sizeof (unsigned char));
	bcompData = compData;
	
bflen = flen;

	fread (CRC32, 1, GRG_CRC_LEN, fp);
	fread (compData, 1, flen, fp);

	fclose (fp);

	if (compare_CRC32 (CRC32, compData, flen) == GRG_FALSE)
	{
		grg_free (CRC32, GRG_CRC_LEN);
		grg_free (bcompData, bflen);
		return GRG_READ_CRC_ERR;
	}

	grg_free (CRC32, GRG_CRC_LEN);

	if (just_validate == 1)
		return GRG_OK;

	//don't care ALGO, for now
	compData += GRG_ALGO_LEN * sizeof (unsigned char);
	flen -= GRG_ALGO_LEN * sizeof (unsigned char);

	//decrypts the encrypted data
	mod = mcrypt_module_open (MCRYPT_SERPENT, NULL, MCRYPT_CFB, NULL);

	if (mod == MCRYPT_FAILED)
	{
		grg_free (bcompData, bflen);
		return GRG_READ_ENC_INIT_ERR;
	}

	key = get_key (pwd);
	dIV = mcrypt_enc_get_iv_size (mod);
	salt = grg_strdup (compData, salt_size);
	IV = get_IV (dIV, salt, salt_size, pwd);

	mcrypt_generic_init (mod, key, GRG_KEY_SIZE, IV);
	grg_free (key, GRG_KEY_SIZE);
	grg_free (IV, dIV);
	grg_free (salt, salt_size);
	if (err < 0)
	{
		grg_free (bcompData, bflen);
		return GRG_READ_ENC_INIT_ERR;
	}

	compData += salt_size * sizeof (unsigned char);
	flen -= salt_size;

	for (i = 0; i < flen; i++)
		mdecrypt_generic (mod, &compData[i], 1);

	//checks the 2nd CRC32

	CRC32b = grg_strdup (compData, GRG_CRC_LEN);

	compData += GRG_CRC_LEN * sizeof (unsigned char);
	flen -= GRG_CRC_LEN * sizeof (unsigned char);

	if (compare_CRC32 (CRC32b, compData, flen) == GRG_FALSE)
	{
		grg_free (bcompData, bflen);
		grg_free (CRC32b, GRG_CRC_LEN);
		return GRG_READ_PWD_ERR;
	}

	grg_free (CRC32b, GRG_CRC_LEN);

	//reads the uncompressed data length

	dimdata = grg_strdup (compData, GRG_DATA_DIM_LEN);

	compData += GRG_DATA_DIM_LEN * sizeof (unsigned char);
	flen -= GRG_DATA_DIM_LEN * sizeof (unsigned char);

	origLen = grg_char2long (dimdata);

	grg_free (dimdata, GRG_DATA_DIM_LEN);

	//uncompress the final data

	rawData =
		(unsigned char *) malloc ((origLen) * sizeof (unsigned char));

	err = uncompress (rawData, &origLen, compData, flen);

	grg_free (bcompData, bflen);

	//ensure it is NULL-terminated
	*origData = grg_strconcat (rawData, "", NULL, origLen, 1, 0);

	grg_free (rawData, origLen);

	return GRG_OK;
}
