/*
 * This file is a part of the gnetsentry project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "miscinet.h"
#include "netsentry.h"
#include "ns_exp.h"

/*#define NETSENTRY_CF	"/usr/local/etc/netsentry/etc/netsentry.cf"*/

#ifndef NETSENTRY_CF
# define NETSENTRY_CF	"./netsentry.cf"
#endif

char			*hard_filter = NULL;
char			*conf_file = NETSENTRY_CF;
char			*dev = NULL;
int			snaplen = 1024;
int			promisc = 1;
int			to_ms = 10;
int			optimize = 0;
t_boolean		first_match = TRUE;
t_boolean		resolve = TRUE;
t_boolean		verbose = FALSE;
#ifdef DEBUG_MALLOC
t_boolean		debug_malloc = FALSE;
#endif
char			*force_packet = NULL;
t_boolean		force_stdout = FALSE;
char			*force_rule = NULL;

t_opt			main_opts[] = 
{
  {"-hf",	(t_opt_proc)opt_str,		(off_t)&hard_filter,
   "hard_filter",NULL,0},
  {"-cf",	(t_opt_proc)opt_str,		(off_t)&conf_file,
   "conf_file",NULL,0},
  {"-i",	(t_opt_proc)opt_str,		(off_t)&dev,
   "dev",NULL,0},
  {"-s",	(t_opt_proc)opt_int,		(off_t)&snaplen,
   "snaplen",NULL,0},
  {"promisc",	(t_opt_proc)opt_true,		(off_t)&promisc,
   NULL,"promiscuous mode",0},
  {"-t",	(t_opt_proc)opt_int,		(off_t)&to_ms,
   "to_ms","timeout in milliseconds",0},
  {"-o",	(t_opt_proc)opt_int,		(off_t)&optimize,
   NULL,"optimize",0},
  {"-fm",	(t_opt_proc)opt_false,		(off_t)&first_match,
   NULL,"don't stop at first match",0},
  {"-n",	(t_opt_proc)opt_false,		(off_t)&resolve,
   NULL,"turn off resolve mode",0},
  {"-fp",	(t_opt_proc)opt_str,		(off_t)&force_packet,
   "force_packet","force evaluation of packet",0},
  {"-fr",	(t_opt_proc)opt_str,		(off_t)&force_rule,
   "force_rule","force evaluation of rule",0},
  {"-fout",	(t_opt_proc)opt_true,		(off_t)&force_stdout,
   NULL,"force a+ fcache openings to stdout",0},
  {"-v",	(t_opt_proc)opt_true,		(off_t)&verbose,
   NULL,"turn on verbose mode",0},
#ifdef DEBUG_MALLOC
  {"-dm",	(t_opt_proc)opt_true,		(off_t)&debug_malloc,
   NULL,"turn on debug_malloc",0},
#endif
};

FILE			*fopen_aplus(fname)
char			*fname;
{
  FILE			*f;

  if (force_stdout)
    return (stdout);
  if ((f = fopen(fname,"a+")) == NULL)
    return (NULL);
  return (f); 
}

FILE			*fopen_rplus(fname)
char			*fname;
{
  FILE			*f;

  if ((f = fopen(fname,"r+")) == NULL)
    return (NULL);
  return (f); 
}

FILE			*fopen_r(fname)
char			*fname;
{
  FILE			*f;

  if ((f = fopen(fname,"r")) == NULL)
    return (NULL);
  return (f); 
}

VOID_FUNC		fclose_fclose(f)
FILE			*f;
{
  fclose(f);
}

t_file_cache		file_cache_aplus = 
{
  NULL,
  fopen_aplus,
  fclose_fclose
};

t_file_cache		file_cache_rplus = 
{
  NULL,
  fopen_rplus,
  fclose_fclose
};

t_file_cache		file_cache_r = 
{
  NULL,
  fopen_r,
  fclose_fclose
};

VOID_FUNC		myexit(exitstatus)
int			exitstatus;
{
  if (verbose)
    fprintf(stderr,"exiting with code %d\n",exitstatus);
  exit(exitstatus);
}

VOID_FUNC		usage(VOID_DECL)
{
  opt_usage(stderr,main_opts,ARRAY_COUNT(main_opts),TRUE);
  myexit(1);
}

pcap_t			*open_pcap(VOID_DECL)
{
  char			errbuf[BUFSIZ];
  pcap_t		*pcap;

  if (!dev)
    if ((dev = pcap_lookupdev(errbuf)) == NULL)
      {
	fprintf(stderr,"%s",errbuf);
	return (NULL);
      }
  if ((pcap = pcap_open_live(dev,
			     snaplen,
			     promisc,
			     to_ms,
			     errbuf)) == NULL)
    {
      fprintf(stderr,"%s",errbuf);
      return (NULL);
    }
  return (pcap);
}

int			check_options(VOID_DECL)
{
  if (!conf_file)
    {
      err_print(ERR_CANTGOON,"need -cf");
      return (-1);
    }
  if (verbose)
    {
      fprintf(stderr,"options:\n");
      fprintf(stderr,"\t%s %s %s\n",PRODUCT,VERSION,VENDOR);
#ifdef DEBUG
      fprintf(stderr,"\tcompiled with DEBUG\n");
#endif
#ifdef DEBUG_MALLOC
      fprintf(stderr,"\tcompiled with DEBUG_MALLOC\n");
#endif
      fprintf(stderr,"\t-hf = %s\n",
	      hard_filter?hard_filter:"specified in conf_file");
      fprintf(stderr,"\t-cf = %s\n",conf_file);
      fprintf(stderr,"\t-i = %s\n",dev?dev:"default");
      fprintf(stderr,"\t-s = %d\n",snaplen);
      fprintf(stderr,"\t%s\n",promisc?"promisc":"-promisc");
      fprintf(stderr,"\t-t = %d\n",to_ms);
      fprintf(stderr,"\t-o = %s\n",optimize?"bpf optimized":"bpf not optimized");
      fprintf(stderr,"\t-fm = %s\n",first_match?"first_match":"all matches");
      fprintf(stderr,"\t-n = %s\n",resolve?"resolve":"don't resolve");
      fprintf(stderr,"\t-fp = %s\n",force_packet?"activated":"unused");
      fprintf(stderr,"\t-fr = %s\n",force_rule?force_rule:"unused");
      fprintf(stderr,"\t-fout = %s\n",force_stdout?"force stdout":
	      "don't force stdout");
      fprintf(stderr,"\t-v = %s\n",verbose?"verbose":"not verbose");
#ifdef DEBUG_MALLOC
      fprintf(stderr,"\t-dm = %s\n",debug_malloc?"debug_malloc":
	      "no debug_malloc");
#endif
    }
  return (0);
}

int				set_hard_filter(pcap,default_rule)
pcap_t				*pcap;
t_rule				*default_rule;
{
  struct bpf_program		bp;

  if (pcap_compile(pcap,
		   &bp,
		   hard_filter?hard_filter:default_rule->filter,
		   optimize,
		   0) < 0)
    {
      fprintf(stderr,"hard filter syntax error\n");
      return (-1);
    }
  if (pcap_setfilter(pcap,&bp) < 0)
    {
      fprintf(stderr,"can't set hard filter\n");
      return (-1);
    }
  return (0);
}

t_status			action(rule,
				       buf,
				       len,
				       caplen,
				       ts,
				       pcap)
t_rule				*rule;
char				*buf;
int				len;
int				caplen;
struct timeval			*ts;
pcap_t				*pcap;
{
  t_cmd_proc			cmd_proc;
  t_status			status;

  VEC_FOR(rule->cmds,t_vec *args)
    {
      t_compiled_cmd		*cc;

      cc = VEC_AT(rule->compiled_cmds,VEC_IDX);
      if ((status = cc->cmd_proc(args,
				 buf,
				 len,
				 caplen,
				 ts,
				 pcap,
				 rule,
				 cc->data)) < 0)
	  return (status);
    }
  VEC_ENDFOR;
  return (0);
}

t_boolean			eval_rule(gc,buf,pp,rule)
t_global_config			*gc;
char				*buf;
struct pcap_pkthdr		*pp;
t_rule				*rule;
{
  t_status			status;
  
  if (bpf_filter(rule->bp.bf_insns,
		 (unsigned char *)buf,
		 pp->len,
		 pp->caplen))
    {
      t_boolean			ok;
      
      rule->current_pkt_info.buf = buf;
      rule->current_pkt_info.len = pp->len;
      rule->current_pkt_info.caplen = pp->caplen;
      rule->current_pkt_info.ts = &(pp->ts);
      ok = TRUE;
      if (rule->node)
	{
	  t_object		*result;
	  
	  if ((status = eval_node(rule->node,
				  (VOID_PTR)(&result))) < 0)
	    {
	      err_print(-status,"eval_node %s",rule->name);
	      ok = FALSE;
	    }
	  else
	    {
	      if (verbose)
		{
		  object_show(rule->ns_exp.om,result);
		}
	      if (result->oid == rule->ns_exp.int_oid &&
		  result->value == NULL)
		ok = FALSE;
	    }
	  object_manager_destroy_objects(rule->ns_exp.om);
	}
      if (ok)
	{
	  rule->count++;
	  if ((status = action(rule,
			       buf,
			       pp->len,
			       pp->caplen,
			       &(pp->ts),
			       gc->pcap)) < 0)
	    {
	      if (rule->fails < rule->maxfails)
		err_print(-status,"%s: action",rule->name);
	      rule->fails++;
	    }
	}
      return (TRUE);
    }
  else
    return (FALSE);
}

t_status			treat_packet(gc,buf,pp)
t_global_config			*gc;
char				*buf;
struct pcap_pkthdr		*pp;
{
  t_boolean			match;
  t_status			status;

  match = FALSE;
  VEC_FOR(gc->rules,t_rule *rule)
    {
      if (eval_rule(gc,buf,pp,rule))
	{
	  match = TRUE;
	  if (first_match)
	    if (rule->cont_is_defined)
	      {
		if (!(rule->cont))
		  break ;
	      }
	    else
	      break ;
	}
    }
  VEC_ENDFOR;
  if (!match)
    {
      gc->default_rule->count++;
      if ((status = action(gc->default_rule,
			   buf,
			   pp->len,
			   pp->caplen,
			   &(pp->ts),
			   gc->pcap)) < 0)
	{
	  if (gc->default_rule->fails < gc->default_rule->maxfails)
	    err_print(-status,"default: action");
	  gc->default_rule->fails++;
	}
    }
  return (0);
}

int				do_packet_on_rule(gc,xdata,rule)
t_global_config			*gc;
char				*xdata;
t_rule				*rule;
{
  struct pcap_pkthdr		*pp;
  char				*buf;
  t_status			status;
  t_boolean			match;

  if ((status = pcap_pkthdr_from_xdata(xdata,
				       NS_ALLOC_PROC,
				       NS_FREE_PROC,
				       &buf,
				       &pp)) < 0)
    {
      err_print(-status,"converting xdata");
      return (status);
    }
  match = eval_rule(gc,buf,pp,rule);
  if (verbose)
    fprintf(stderr,"%s\n",match?"true":"false");
  return (0);
}

int				do_packet(gc,xdata)
t_global_config			*gc;
char				*xdata;
{
  struct pcap_pkthdr		*pp;
  char				*buf;
  t_status			status;

  if ((status = pcap_pkthdr_from_xdata(xdata,
				       NS_ALLOC_PROC,
				       NS_FREE_PROC,
				       &buf,
				       &pp)) < 0)
    {
      err_print(-status,"converting xdata");
      return (status);
    }
  return (treat_packet(gc,buf,pp));
}

int				loop_on_rule(gc,rule)
t_global_config			*gc;
t_rule				*rule;
{
  while (1)
    {
      struct pcap_pkthdr	pp;
      char			*buf;
      t_status			status;
      t_boolean			match;

      if ((buf = (char *)pcap_next(gc->pcap,&pp)) == NULL)
	{
/*
#ifndef HAVE_BPF_IN_KERNEL
		return (-1);
#endif
*/
	}
      match = eval_rule(gc,buf,&pp,rule);
      if (verbose)
	fprintf(stderr,"%s\n",match?"true":"false");
    }
}

int				loop(gc)
t_global_config			*gc;
{
  while (1)
    {
      struct pcap_pkthdr	pp;
      char			*buf;
      t_status			status;

      if ((buf = (char *)pcap_next(gc->pcap,&pp)) == NULL)
	{
/*
#ifndef HAVE_BPF_IN_KERNEL
		return (-1);
#endif
*/
	}
      if ((status = treat_packet(gc,buf,&pp)) < 0)
	return (status);
    }
}

t_global_config		gc; /* MUST NOT BE EXPORTED */

VOID_FUNC		pcap_statistics(VOID_DECL)
{
  struct pcap_stat	ps;

  if (pcap_stats(gc.pcap,&ps) < 0)
    {
      err_print(ERR_NOMETHOD,"packet filtering is not able to do statistics");
    }
  else
    {
      fprintf(stderr,"recv=%d\n",ps.ps_recv);
      fprintf(stderr,"drop=%d\n",ps.ps_drop);
      fprintf(stderr,"ifdrop=%d\n",ps.ps_ifdrop);
    }
}

RETSIGTYPE		sigint(ignore)
int			ignore;
{
  file_cache_close(&file_cache_aplus);
  file_cache_close(&file_cache_rplus);
  file_cache_close(&file_cache_r);
  cmd_keep_info();
  pcap_statistics();
  dict_delete(file_cache_aplus.dict);
  dict_delete(file_cache_rplus.dict);
  dict_delete(file_cache_r.dict);
  free_keepdict();
  delete_rules(gc.rules);
  rule_destroy(gc.default_rule);
  NS_FREE_PROC(gc.default_rule,"gc.default_rule");
  vec_ptr_delete(gc.prefixes);
  pcap_close(gc.pcap);
  ns_db_destroy();
#ifdef DEBUG_MALLOC
  if (debug_malloc)
    {
      debug_malloc_status(0);
      debug_malloc_destroy();
      alloc_algorithm_status();
    }
#endif
  myexit(1);
}

RETSIGTYPE		sighup(ignore)
int			ignore;
{
  file_cache_flush(&file_cache_aplus);
  file_cache_flush(&file_cache_rplus);
  cmd_keep_info();
  pcap_statistics();
}

VOID_FUNC		handle_signals(VOID_DECL)
{
  signal(SIGINT,sigint);
  signal(SIGHUP,sighup);
}

VOID_FUNC		tty_detach(VOID_DECL)
{
  /* TODO */
}

int			init_file_cache_dicts(VOID_DECL)
{
  t_status		status;
  
  if ((file_cache_aplus.dict = NS_DICT_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_DICT_NEW");
      return (-1);
    }
  if ((file_cache_rplus.dict = NS_DICT_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_DICT_NEW");
      return (-1);
    }
  if ((file_cache_r.dict = NS_DICT_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_DICT_NEW");
      return (-1);
    }
  return (0);
}

VOID_FUNC		doit(VOID_DECL)
{
  t_vec			*rules_names;
  t_vec			*prefixes_names;
  t_vec			*databases_names;
  t_status		status;
  char			*str;
  
  if (check_options() < 0)
    myexit(1);
  if (ns_db_init() < 0)
    myexit(1);
  if ((gc.vars = open_cf()) == NULL)
    myexit(1);
  if ((rules_names = get_rules_names(gc.vars)) == NULL)
    myexit(1);
  if ((prefixes_names = get_prefixes_names(gc.vars)) == NULL)
    myexit(1);
  if ((gc.pcap = open_pcap()) == NULL)
    myexit(1);
  if (init_file_cache_dicts() < 0)
    myexit(1);
  if ((gc.default_rule = compile_rule(gc.pcap,gc.vars,DEFAULT_STR)) == NULL)
    myexit(1);
  gc.default_rule->gc = &gc;
  if ((gc.prefixes = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      myexit(1);
    }
  VEC_FOR(prefixes_names,char *prefix)
    {
      t_subnet		subnet;
      t_subnet		*s;

      if ((status = subnet_from_str(prefix,&subnet)) < 0)
	{
	  err_print(-status,"bad subnet definition %s",prefix);
	  myexit(1);
	}
      if ((s = gc.prefixes->alloc_proc(sizeof (t_subnet),"s",&status)) == NULL)
	{
	  err_print(-status,"allocing s");
	  myexit(1);
	}
      bcopy((char *)(&subnet),(char *)s,sizeof (t_subnet));
      if ((status = vec_add(gc.prefixes,s)) < 0)
	{
	  err_print(-status,"vec_add");
	  myexit(1);
	}
    }
  VEC_ENDFOR;
  vec_str_delete(prefixes_names);
  if ((gc.rules = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      myexit(1);
    }
  VEC_FOR(rules_names,char *name)
    {
      t_rule		*rule;

      if ((rule = compile_rule(gc.pcap,gc.vars,name)) == NULL)
	myexit(1);
      if ((status = vec_add(gc.rules,rule)) < 0)
	{
	  err_print(-status,"vec_add");
	  myexit(1);
	}
      rule->gc = &gc;
    }
  VEC_ENDFOR;
  vec_str_delete(rules_names);
  if (verbose)
    show_rules(gc.rules);
  if (set_hard_filter(gc.pcap,gc.default_rule) < 0)
    myexit(1);
  if ((databases_names = get_databases_names(gc.vars)) == NULL)
    myexit(1);
  VEC_FOR(databases_names,char *fname)
    {
      if (ns_db_load(fname,&status) == NULL)
	myexit(1);
    }
  VEC_ENDFOR;
  vec_str_delete(databases_names);
  if (init_things_for_cmds() < 0)
    myexit(1);
  dict_str_delete(gc.vars);
  tty_detach();
  handle_signals();
  if (force_packet)
    {
      if (force_rule)
	{
	  t_rule	*rule;
	  
	  if ((rule = rule_get_from_name(gc.rules,force_rule)) == NULL)
	    {
	      err_print(ERR_NOENT,"%s",force_rule);
	      myexit(1);
	    }
	  if (do_packet_on_rule(&gc,force_packet,rule) < 0)
	    myexit(1);
	}
      else
	{
	  if (do_packet(&gc,force_packet) < 0)
	    myexit(1);
	}
    }
  else
    {
      if (force_rule)
	{
	  t_rule	*rule;
	  
	  if ((rule = rule_get_from_name(gc.rules,force_rule)) == NULL)
	    {
	      err_print(ERR_NOENT,"%s",force_rule);
	      myexit(1);
	    }
	  if (loop_on_rule(&gc,rule) < 0)
	    myexit(1);
	}
      else
	{
	  if (loop(&gc) < 0)
	    myexit(1);
	}
    }
  myexit(0);
}

int			main(argc,argv)
int			argc;
char			**argv;
{
  t_opt_context		oc;
  t_boolean		trargsbuf[256];
  t_boolean		troptsbuf[256];
  t_status              status;

#ifdef DEBUG
  a_debug_init();
  exp_debug_init();
  layer_debug_init();
#endif
  assert(opt_check(main_opts,ARRAY_COUNT(main_opts)) == 0);
  oc.tr_args = trargsbuf;
  oc.nb_tr_args = sizeof (trargsbuf);
  oc.tr_opts = troptsbuf;
  oc.nb_tr_opts = sizeof (troptsbuf);
  oc.argc = &argc;
  oc.argv = argv;
  oc.opts = main_opts;
  oc.nb_opts = ARRAY_COUNT(main_opts);
  oc.base_addr = NULL;
  if ((status = opt_get(&oc)) < 0)
    {
      err_print(-status,"couldn't get options");
      usage();
    }
  if (argc != 1)
    {
      usage();
    }
#ifdef DEBUG_MALLOC
  if (debug_malloc)
    {
      if ((status = debug_malloc_init()) < 0)
	{
	  err_print(-status,"debug_malloc_init");
	  exit(1);
	}
    }
#endif
  doit();
}
