/*
 * This file is a part of the gnetsentry project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <fcntl.h>
#include "netsentry.h"
#include "ns_exp.h"
#include "cmd_nop.h"
#include "cmd_count.h"
#include "cmd_format.h"
#include "cmd_exec.h"
#include "cmd_keep.h"

extern char		*conf_file;
extern int		optimize;

t_dict			*open_cf(VOID_DECL)
{
  int			fd;
  t_status		status;
  t_dict		*vars;

  if ((fd = open(conf_file,O_RDONLY)) < 0)
    {
      err_print(ERR_READ,"%s",conf_file);
      return (NULL);
    }
  if ((vars = NS_DICT_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_DICT_NEW");
      close(fd);
      return (NULL);
    }
  if ((status = parse_vars(fd,'=',vars,dict_str_add)) < 0)
    {
      err_print(-status,"while parsing");
      dict_str_delete(vars);
      return (NULL);
    }
  close(fd);
  return (vars);
}

#define GETVAR(str,dict,var) \
	if (((str) = dict_str_get((dict),(var))) == NULL)\
	{\
	   err_print(ERR_CANTGOON,"need %s in cf",(var));\
	   return (NULL);\
	}

t_vec			*get_rules_names(vars)
t_dict			*vars;
{
  char			*str;
  t_vec			*rules_names;
  t_status		status;

  if ((rules_names = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      return (NULL);
    }
  GETVAR(str,vars,RULES_STR);
  if ((status = vec_str_split(rules_names,str,',')) < 0)
    {
      err_print(-status,"parsing %s",str);
      return (NULL);
    }
  return (rules_names);
}

t_vec			*get_prefixes_names(vars)
t_dict			*vars;
{
  char			*str;
  t_vec			*prefixes_names;
  t_status		status;

  if ((prefixes_names = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      return (NULL);
    }
  if (str = dict_str_get(vars,PREFIXES_STR))
    if ((status = vec_str_split(prefixes_names,str,',')) < 0)
      {
	err_print(-status,"parsing %s",str);
	return (NULL);
      }
  return (prefixes_names);
}

t_vec			*get_databases_names(vars)
t_dict			*vars;
{
  char			*str;
  t_vec			*databases_names;
  t_status		status;

  if ((databases_names = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      return (NULL);
    }
  if (str = dict_str_get(vars,DATABASES_STR))
    if ((status = vec_str_split(databases_names,str,',')) < 0)
      {
	err_print(-status,"parsing %s",str);
	return (NULL);
      }
  return (databases_names);
}

#define STRCAT(buf,str)	\
{\
  t_status	status;\
\
  if ((status = str_cat_str((buf),sizeof (buf),(str))) < 0)\
    {\
      err_print(-status,"stop");\
      return (NULL);\
    }\
}

t_vec			*get_cmds(str)
char			*str;
{
  t_parse_lines_context	plc;
  t_status		status;
  char			wordbuf[STR_BUFSIZ];

  if ((plc.vec = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      return (NULL);
    }
  plc.wordbuf = wordbuf;
  plc.wordbuflen = sizeof (wordbuf);
  plc.wordseps = " \t";
  plc.lineseps = "\n;";
  parse_lines_init(&plc);
  if ((status = parse_lines(&plc,str,strlen(str))) < 0)
    {
      err_print(-status,"parsing cmd %s",str);
      return (NULL);
    }
  if ((status = parse_lines_finnish(&plc)) < 0)
    {
      err_print(-status,"parsing cmd %s",str);
      return (NULL);
    }
  return (plc.vec);
}

t_assoc			cmd_assocs[] = 
{
  {"nop",		cmd_nop_init},
  {"count",		cmd_count_init},
  {"format",		cmd_format_init},
  {"exec",		cmd_exec_init},
  {"keep",		cmd_keep_init},
  {NULL,		0},
};

t_vec			*compile_cmds(cmds)
t_vec			*cmds;
{
  t_vec			*compiled_cmds;
  t_status		status;

  if ((compiled_cmds = NS_VEC_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_VEC_NEW");
      return (NULL);
    }
  VEC_FOR(cmds,t_vec *args)
    {
      t_assoc		*assoc;
      t_cmd_init_proc	cmd_init_proc;
      t_cmd_proc	cmd_proc;
      t_compiled_cmd	*cc;

      if ((assoc = assoc_str_ptr_from_left(cmd_assocs,VEC_AT(args,0))) == NULL)
	{
	  err_print(ERR_NOENT,"cmd %s",VEC_AT(args,0));
	  return (NULL);
	}
      cmd_init_proc = (t_cmd_init_proc)(assoc->right);
      if ((cc = NS_ALLOC_PROC(sizeof (t_compiled_cmd),"cc",&status)) == NULL)
	{
	  err_print(-status,"stop");
	  return (NULL);
	}
      if ((status = cmd_init_proc(args,cc)) < 0)
	{
	  NS_FREE_PROC(cc,"cc");
	  return (NULL);
	}
      if ((status = vec_add(compiled_cmds,cc)) < 0)
	{
	  NS_FREE_PROC(cc,"cc");
	  err_print(-status,"stop");
	  return (NULL);
	}
    }
  VEC_ENDFOR;
  return (compiled_cmds);
}

t_rule			*compile_rule(pcap,vars,name)
pcap_t			*pcap;
t_dict			*vars;
char			*name;
{
  char			buf[STR_BUFSIZ];
  t_status		status;
  t_rule		*rule;
  char			*str;

  if ((rule = NS_ALLOC_PROC(sizeof (t_rule),"rule",&status)) == NULL)
    {
      err_print(-status,"stop");
      return (NULL);
    }
  if ((rule->name = strdup_alloc(name,
				 NS_ALLOC_PROC,
				 "rule->name",
				 &status)) == NULL)
    {
      err_print(-status,"strdup_alloc");
      return (NULL);
    }
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,FILTER_STR);
  GETVAR(str,vars,buf);
  if ((rule->filter = strdup_alloc(str,
				   NS_ALLOC_PROC,
				   "rule->filter",
				   &status)) == NULL)
    {
      err_print(-status,"strdup_alloc");
      return (NULL);
    }
  if (pcap_compile(pcap,&(rule->bp),str,optimize,0) < 0)
    {
      fprintf(stderr,"filter syntax error: %s\n",str);
      return (NULL);
    }
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,ACTION_STR);
  GETVAR(str,vars,buf);
  if ((rule->cmds = get_cmds(str)) == NULL)
    {
      return (NULL);
    }
  if ((rule->compiled_cmds = compile_cmds(rule->cmds)) == NULL)
    {
      return (NULL);
    } 
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,MAXFAILS_STR);
  if (str = dict_str_get(vars,buf))
    rule->maxfails = atoi(str);
  else
    rule->maxfails = MAXFAILS;
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,CONT_STR);
  if (str = dict_str_get(vars,buf))
    {
      rule->cont = atobooleanfalse(str);
      rule->cont_is_defined = TRUE;
    } 
  else
    rule->cont_is_defined = FALSE;
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,MAXKEEP_STR);
  if (str = dict_str_get(vars,buf))
    rule->maxkeep = atoi(str);
  else
    rule->maxkeep = MAXKEEP;
  buf[0] = 0;
  STRCAT(buf,name);
  STRCAT(buf,EXP_STR);
  if (str = dict_str_get(vars,buf))
    {
      if ((rule->exp = strdup_alloc(str,
				    NS_ALLOC_PROC,
				    "rule->exp",
				    &status)) == NULL)
	{
	  err_print(-status,"strdup_alloc");
	  return (NULL);
	}
      ns_exp_init(&(rule->ns_exp));
#ifdef DEBUG
      if (verbose)
	ns_exp_show(&(rule->ns_exp));
#endif
      compil_set_buf(rule->ns_exp.compil,rule->exp,strlen(rule->exp));
      if ((status = compile(rule->ns_exp.compil,&(rule->node))) < 0)
	{
	  err_print(-status,"compiling %s",rule->exp);
	  return (NULL);
	}
      if (rule->ns_exp.compil->paren != 0)
	{
	  err_print(ERR_PAREN,"%s",rule->exp);
	  return (NULL);
	}
      if (rule->ns_exp.compil->scan->text[0] != 0)
	{
	  err_print(ERR_PARSE,"%s",rule->exp);
	  return (NULL);
	}
      rule->ns_exp.rule = rule;
    }
  else
    {
      rule->exp = NULL;
      rule->node = NULL;
    }
  rule->fails = rule->count = 0;
  return (rule);
}

VOID_FUNC		show_rules(rules)
t_vec			*rules;
{
  VEC_FOR(rules,t_rule *rule)
    {
      fprintf(stderr,"%s:\n",rule->name);
      fprintf(stderr,"\tfilter: %s\n",rule->filter);
      VEC_FOR(rule->cmds,t_vec *vec)
	{
	  fprintf(stderr,"\tcmd: ");
	  VEC_FOR(vec,char *arg)
	    {
	      fprintf(stderr," `%s'",arg);
	    }
	  VEC_ENDFOR;
	  fprintf(stderr,"\n");
	}
      VEC_ENDFOR;
      if (rule->node)
	{
	  fprintf(stderr,"\tnode: ");
	  show_node(stderr,rule->node,TRUE);
	  fprintf(stderr,"\n");
	}
      fprintf(stderr,"\tmaxfails: %d\n",rule->maxfails);
    }
  VEC_ENDFOR;
}

t_rule			*rule_get_from_name(rules,name)
t_vec			*rules;
char			*name;
{
  VEC_FOR(rules,t_rule *rule)
    {
      if (!strcmp(rule->name,name))
	return (rule);
    }
  VEC_ENDFOR;
  return (NULL);
}

t_status		init_things_for_cmds(VOID_DECL)
{
  t_status		status;

  timeval_to_str_init();
  if ((keepdict = NS_DICT_NEW(&status)) == NULL)
    {
      err_print(-status,"NS_DICT_NEW");
      return (status);
    }
  return (0);
}

VOID_FUNC		rule_destroy(rule)
t_rule			*rule;
{
  NS_FREE_PROC(rule->name,"rule->name");
  NS_FREE_PROC(rule->filter,"rule->filter");
  VEC_FOR(rule->cmds,t_vec *vec_str)
    {
      vec_str_delete(vec_str);
    }
  VEC_ENDFOR;
  vec_delete(rule->cmds);
  vec_ptr_delete(rule->compiled_cmds);
  if (rule->exp)
    {
      NS_FREE_PROC(rule->exp,"rule->exp");
      ns_exp_destroy(&(rule->ns_exp));
      if (rule->node)
	{
	   delete_node(rule->node,TRUE);
	}
    }
}

VOID_FUNC		delete_rules(rules)
t_vec			*rules;
{
  VEC_FOR(rules,t_rule *rule)
    {
      rule_destroy(rule);
      NS_FREE_PROC(rule,"rule");
   }
  VEC_ENDFOR;
  vec_delete(rules);
}
