/*
 * This file is a part of the gnetsentry project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "exp_compil.h"

/* COMPILATION ALGORITM IS BASED UPON REVERSE POLISH */

t_node_item		*node_item_new(pool,token,status)
t_vec			*pool;
t_token			*token;
t_status		*status;
{
  t_node_item		*ni;

  if ((ni = EXP_ALLOC_PROC(sizeof (t_node_item),"ni",status)) == NULL)
    return (NULL);
  ni->type = NODE_ITEM;
  if ((ni->token = EXP_ALLOC_PROC(sizeof (t_token),
				  "ni->token",
				  status)) == NULL)
    {
      EXP_FREE_PROC(ni,"ni");
      return (NULL);
    }
  bcopy((char *)token,(char *)(ni->token),sizeof (t_token));
  if (((*status) = vec_add(pool,ni)) < 0)
    {
      EXP_FREE_PROC(ni->token,"ni->token");
      EXP_FREE_PROC(ni,"ni");
      return (NULL);
    }
  return (ni);
}

VOID_FUNC		node_item_delete(ni)
t_node_item		*ni;
{
  if (exp_keep_text)
    EXP_FREE_PROC(ni->token->token_show.text,"ni->token->token_show.text");
  EXP_FREE_PROC(ni->token,"ni->token");
  EXP_FREE_PROC(ni,"ni");
}

t_node_item		*node_item_dup(pool,ni,status)
t_vec			*pool;
t_node_item		*ni;
t_status		*status;
{
  return (node_item_new(pool,ni->token,status));
}

t_compil		*compil_new(status)
t_status		*status;
{
  t_compil		*compil;

  if ((compil = EXP_ALLOC_PROC(sizeof (t_compil),"compil",status)) == NULL)
    return (NULL);
  if ((compil->pool = EXP_VEC_NEW(status)) == NULL)
    {
      EXP_FREE_PROC(compil,"compil");
      return (NULL);
    }
  if ((compil->scan = scan_new(status)) == NULL)
    {
      vec_delete(compil->pool);
      EXP_FREE_PROC(compil,"compil");
      return (NULL);
    }
  compil->state = STATE_INIT;
  compil->paren = 0;
  compil->level = 0;
  return (compil);
}

t_subcompil		*subcompil_new(compil,status)
t_compil		*compil;
t_status		*status;
{
  t_subcompil		*subcompil;

  if ((subcompil = EXP_ALLOC_PROC(sizeof (t_subcompil),
				  "subcompil",
				  status)) == NULL)
    return (NULL);
  if ((subcompil->operand_stack = EXP_VEC_NEW(status)) == NULL)
    {
      EXP_FREE_PROC(subcompil,"subcompil");
      return (NULL);
    }
  if ((subcompil->operator_stack = EXP_VEC_NEW(status)) == NULL)
    {
      vec_delete(subcompil->operand_stack);
      EXP_FREE_PROC(subcompil,"subcompil");
      return (NULL);
    }
  subcompil->compil = compil;
  return (subcompil);
}

VOID_FUNC		compil_pool_destroy(compil)
t_compil		*compil;
{
#ifdef DEBUG
  if (EXP_VERB(VERB_COMPIL))
    fprintf(stderr,"destroying %d nodes\n",VEC_COUNT(compil->pool));
#endif
  VEC_FOR(compil->pool,t_node *node)
    {
      delete_node(node,FALSE);
    }
  VEC_ENDFOR;
  vec_destroy(compil->pool);
}

VOID_FUNC		compil_pool_empty(compil)
t_compil		*compil;
{
#ifdef DEBUG
  if (EXP_VERB(VERB_COMPIL))
    fprintf(stderr,"detaching %d nodes\n",VEC_COUNT(compil->pool));
#endif
  vec_destroy(compil->pool);
}

VOID_FUNC		compil_reset(compil)
t_compil		*compil;
{
#ifdef DEBUG
  if (EXP_VERB(VERB_COMPIL))
    fprintf(stderr,"compil_reset\n");
#endif
  scan_reset(compil->scan);
  compil->state = STATE_INIT;
  compil->paren = 0;
  compil->level = 0;
}

VOID_FUNC		compil_delete(compil)
t_compil		*compil;
{
  scan_delete(compil->scan);
  vec_delete(compil->pool);
  EXP_FREE_PROC(compil,"compil");
}

VOID_FUNC		subcompil_empty_stacks(subcompil)
t_subcompil		*subcompil;
{
  t_node		*node;

  while (POP_OPERATOR(subcompil,&node) == 0);
  while (POP_OPERAND(subcompil,&node) == 0);
}

VOID_FUNC		subcompil_delete(subcompil)
t_subcompil		*subcompil;
{
  subcompil_empty_stacks(subcompil);
  vec_delete(subcompil->operator_stack);
  vec_delete(subcompil->operand_stack);
  EXP_FREE_PROC(subcompil,"subcompil");
}

VOID_FUNC		compil_set_buf(compil,buf,len)
t_compil		*compil;
char			*buf;
int			len;
{
  scan_set_buf(compil->scan,buf,len);
}

t_status		push_result(subcompil)
t_subcompil		*subcompil;
{
  t_node_item		*operator;
  t_status		status;

  if (LOOKUP_OPERATOR(subcompil,&operator) == 0)
    {
      if (operator->token->token_type == TOKEN_UOP)
	{
	  t_node_uop	*nuop;

	  if ((nuop = node_uop_new(subcompil->compil->pool,&status)) == NULL)
	    return (status);
	  if ((status = POP_OPERAND(subcompil,&(nuop->operand))) < 0)
	    return (status);
	  if ((status = POP_OPERATOR(subcompil,&(nuop->uop))) < 0)
	    return (status);
	  return (PUSH_OPERAND(subcompil,nuop));
	}
      else
	{
	  t_node_bop	*nbop;

	  if ((nbop = node_bop_new(subcompil->compil->pool,&status)) == NULL)
	    return (status);
	  if ((status = POP_OPERAND(subcompil,&(nbop->right))) < 0)
	    return (status);
	  if ((status = POP_OPERAND(subcompil,&(nbop->left))) < 0)
	    return (status);
	  if ((status = POP_OPERATOR(subcompil,&(nbop->bop))) < 0)
	    return (status);
	  return (PUSH_OPERAND(subcompil,nbop));
	}
    }
}

t_status		check_prio(subcompil,operator)
t_subcompil		*subcompil;
t_node_item		*operator;
{
  t_node_item		*old_operator;
  t_status		status;

  if (LOOKUP_OPERATOR(subcompil,&old_operator) == 0)
    {
      if (old_operator->token->token_bop_prio >= 
	  operator->token->token_bop_prio)
	{
	  t_node_bop	*nbop;
	  
	  if ((nbop = node_bop_new(subcompil->compil->pool,&status)) == NULL)
	    return (status);
	  if ((status = POP_OPERAND(subcompil,&(nbop->right))) < 0)
	    return (status);
	  if ((status = POP_OPERAND(subcompil,&(nbop->left))) < 0)
	    return (status);
	  if ((status = POP_OPERATOR(subcompil,&(nbop->bop))) < 0)
	    return (status);
	  if ((status = PUSH_OPERAND(subcompil,nbop)) < 0)
	    return (status);
	  if ((status = check_prio(subcompil,operator)) < 0)
	    return (status);
	}
    }
  return (0);
}

t_status		check_uop_and_push(subcompil,node)
t_subcompil		*subcompil;
t_node			*node;
{
  t_node_item		*operator;
  t_status		status;

  if (LOOKUP_OPERATOR(subcompil,&operator) == 0)
    {
      if (operator->token->token_type == NODE_UOP)
	{
	  t_node_uop	*uop;

	  if ((uop = node_uop_new(subcompil->compil->pool,&status)) == NULL)
	    return (status);
	  uop->operand = node;
	  POP_OPERATOR(subcompil,&(uop->uop));
	  return (check_uop_and_push(subcompil,(t_node *)uop));
	}
      else
	return (PUSH_OPERAND(subcompil,node));
    }
  else
    return (PUSH_OPERAND(subcompil,node));
}

t_status		state_expr_token_operand(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  t_node_item		*operand;
  t_status		status;

  if ((operand = node_item_new(subcompil->compil->pool,token,&status)) == NULL)
    return (status);
  if ((status = check_uop_and_push(subcompil,(t_node *)operand)) < 0)
    return (status);
  subcompil->compil->state = STATE_OPERATOR;
  return (COMPIL_CONT);
}

t_status		state_expr_token_uop(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  t_node_item		*ni;
  t_status		status;
  
  if ((ni = node_item_new(subcompil->compil->pool,token,&status)) == NULL)
    return (status);
  if ((status = PUSH_OPERATOR(subcompil,ni)) < 0)
    return (status);
  subcompil->compil->state = STATE_EXPR;
  return (COMPIL_CONT);
}

t_status		state_expr_token_lp(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  t_node		*node;
  t_status		status;

  subcompil->compil->paren++;
  subcompil->compil->state = STATE_EXPR;
  if ((status = compile(subcompil->compil,&node)) < 0)
    return (status);
  if ((status = check_uop_and_push(subcompil,node)) < 0)
    return (status);
  subcompil->compil->state = STATE_OPERATOR;
  return (COMPIL_CONT);
}

t_status		state_expr_token_rp(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  subcompil->compil->paren--;
  subcompil->compil->state = STATE_OPERATOR;
  return (COMPIL_RET);
}

t_token_proc		state_expr_token_procs[] = 
{
  state_expr_token_operand,
  NULL,
  state_expr_token_uop,
  state_expr_token_lp,
  state_expr_token_rp
};

t_status		state_expr(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  assert(token->token_type < ARRAY_COUNT(state_expr_token_procs));
  if (state_expr_token_procs[token->token_type] == NULL)
    return (-ERR_SYNTAX);
  return (state_expr_token_procs[token->token_type](subcompil,token));
}

t_status		state_operator_token_bop(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  t_node_item		*ni;
  t_status		status;

  if ((ni = node_item_new(subcompil->compil->pool,token,&status)) == NULL)
    return (status);
  if ((status = check_prio(subcompil,ni)) < 0)
    return (status);
  if ((status = PUSH_OPERATOR(subcompil,ni)) < 0)
    return (status);
  subcompil->compil->state = STATE_EXPR;
  return (COMPIL_CONT);
}

t_status		state_operator_token_lp(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  t_node_item		*operand;
  t_node_func		*nf;
  t_node		*node;
  t_status		status;

  if ((status = POP_OPERAND(subcompil,&operand)) < 0)
    return (status);
  if ((nf = node_func_new(subcompil->compil->pool,&status)) == NULL)
    return (status);
  subcompil->compil->paren++;
  subcompil->compil->state = STATE_EXPR;
  if ((status = compile(subcompil->compil,&node)) < 0)
    return (status);
  nf->caller = operand;
  nf->params = node;
  if ((status = PUSH_OPERAND(subcompil,nf)) < 0)
    return (status);
  subcompil->compil->state = STATE_OPERATOR;
  return (COMPIL_CONT);
}

int			state_operator_token_rp(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  subcompil->compil->paren--;
  subcompil->compil->state = STATE_OPERATOR;
  return (COMPIL_RET);
}

t_token_proc		state_operator_token_procs[] = 
{
  NULL,
  state_operator_token_bop,
  NULL,
  state_operator_token_lp,
  state_operator_token_rp
};

t_status		state_operator(subcompil,token)
t_subcompil		*subcompil;
t_token			*token;
{
  assert(token->token_type < ARRAY_COUNT(state_operator_token_procs));
  if (state_operator_token_procs[token->token_type] == NULL)
    return (-ERR_SYNTAX);
  return (state_operator_token_procs[token->token_type](subcompil,token));
}

t_state_proc		state_procs[] = 
{
  state_expr,
  state_expr,
  state_operator
};

t_status		compile(compil,node)
t_compil		*compil;	
t_node			**node;
{
  t_subcompil		*subcompil;
  int			status;

  if ((subcompil = subcompil_new(compil,&status)) == NULL)
    return (status);
  (compil->level)++;
  while (1)
    {
      t_token		token;
      t_advice		advice;

      advice.state = compil->state;
      if ((status = scan_next(compil->scan,
			      (VOID_PTR)(&advice),
			      (VOID_PTR)(&token))) < 0)
	{
	  (compil->level)--;
	  subcompil_delete(subcompil);
	  return (status);
	}
      if (status == SCAN_CONT)
	continue ;
      if (status == SCAN_TOK)
	{
	  assert(compil->state < ARRAY_COUNT(state_procs) &&
		 state_procs[compil->state] != NULL);
#ifdef DEBUG
	  if (EXP_VERB(VERB_COMPIL))
	    compil_show_token(compil,&token);
#endif
	  if ((status = state_procs[compil->state](subcompil,&token)) < 0)
	    {
	      (compil->level)--;
	      subcompil_delete(subcompil);
	      return (status);
	    }
	}
      if (status == SCAN_EOB)
	{
	  if (compil->state == STATE_EXPR)
	    {
	      (compil->level)--;
	      subcompil_delete(subcompil);
	      return (-ERR_SYNTAX);
	    }
	  status == COMPIL_RET;
	} 
      if (compil->paren < 0)
	{
	  (compil->level)--;
	  subcompil_delete(subcompil);
	  return (-ERR_PAREN);
	}
      if (status == COMPIL_RET)
	{
	  t_node	*top;
	  t_node	*unused;
	  
	  while (LOOKUP_OPERATOR(subcompil,&unused) == 0)
	    if ((status = push_result(subcompil)) < 0)
	      {
		(compil->level)--;
		subcompil_delete(subcompil);
		return (status);
	      }
	  if (POP_OPERAND(subcompil,&top) == -ERR_NOENT)
	    top = NULL;
	  (*node) = top;
	  (compil->level)--;
	  subcompil_delete(subcompil);
	  return (0);
	}
    }
  (compil->level)--;
  subcompil_delete(subcompil);
  return (0);
}

#ifdef DEBUG
char			*state_strs[] = 
{
  "init",
  "expr",
  "operator"
};

char			*token_strs[] = 
{
  "operand",
  "bop",
  "uop",
  "lp",
  "rp",
};

VOID_FUNC		compil_show_token(compil,token)
t_compil		*compil;
t_token			*token;
{
  assert(compil->state < ARRAY_COUNT(state_strs));
  assert(token->token_type < ARRAY_COUNT(token_strs));
  fprintf(stderr,"token: %s: %s\n",
	  state_strs[compil->state],
	  token_strs[token->token_type]);
}
#endif
