/*
 * This file is a part of the gnetsentry project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * 
 */

#include "a.h"

#ifdef DEBUG
int		alloc_algorithm_count = 0;

VOID_FUNC	alloc_algorithm_status(VOID_DECL)
{
  fprintf(stderr,"alloc_algorithm_count=%d\n",alloc_algorithm_count);
}
#endif

t_status	alloc_algorithm_exact(eltsize,
				      base_count,
				      old_count,
				      new_count,
				      real_size)
size_t		eltsize;
int		base_count;
int		old_count;
int		new_count;
size_t		*real_size;
{
#ifdef DEBUG
  alloc_algorithm_count++;
#endif
  (*real_size) = new_count * eltsize;
  return (0);
}

t_status	alloc_algorithm_power2(eltsize,
				       base_count,
				       old_count,
				       new_count,
				       real_size)
size_t		eltsize;
int		base_count;
int		old_count;
int		new_count;
size_t		*real_size;
{
  int		n;
  int		y;
  size_t	new_size;
  size_t	old_size;

#ifdef DEBUG
  alloc_algorithm_count++;
#endif
  assert(new_count != 0 && base_count && eltsize != 0);
  if ((y = new_count / base_count) == 0)
    n = 0;
  else
    n = ilog(2,y) + 1;
  old_size = old_count * eltsize;
  new_size = new_count * eltsize;
  (*real_size) = 0;
  while ((*real_size) < new_size)
    if (((*real_size) = base_count * eltsize * ipow(2,n++)) == old_size)
      return (-ERR_EXIST);
  return (0);
}

VOID_PTR	alloc_malloc(size,comment,status)
size_t		size;
char		*comment;
t_status	*status;	
{
  VOID_PTR	ptr;

  if ((ptr = MALLOC(size,comment)) == NULL)
    (*status) = -ERR_NOMEM;
  return (ptr);
}

VOID_FUNC	free_free(ptr,comment)
VOID_PTR	ptr;
char		*comment;
{
  free_comment(ptr,comment);
}

#ifdef DEBUG_MALLOC
VOID_PTR	alloc_real_malloc(size,comment,status)
size_t		size;
char		*comment;
t_status	*status;
{
  VOID_PTR	ptr;

  if ((ptr = real_malloc(size)) == NULL)
    {
      (*status) = -ERR_NOMEM;
      return (NULL);
    }
  return (ptr);
}

VOID_FUNC	free_real_free(ptr,comment)
VOID_PTR	ptr;
char		*comment;
{
  real_free(ptr);
}

int		debug_malloc_total = 0;
int		debug_malloc_total2 = 0;
int		debug_malloc_calls = 0;
int		debug_malloc_calls_ok = 0;
int		debug_malloc_ident = 0;
t_id		*debug_malloc_id = NULL;

t_status	debug_malloc_init(VOID_DECL)
{
  t_status	status;

  if ((debug_malloc_id = id_new(HASH_BASE,
				VEC_BASE,
				alloc_algorithm_power2,
				alloc_real_malloc,
				free_real_free,
				&status)) == NULL)
    return (status);
  return (0);
}

VOID_FUNC	debug_malloc_destroy(VOID_DECL)
{
  id_ptr_delete(debug_malloc_id);
  debug_malloc_id = NULL;
}

VOID_FUNC		debug_malloc_elt_show(dme)
t_debug_malloc_elt	*dme;
{
  fprintf(stderr,"0x%p, %d bytes, id=%d, %s,%d, `%s'\n",
	  dme->ptr,
	  dme->size,
	  dme->id,
	  dme->file,
	  dme->line,
	  dme->comment);
}

t_status		debug_malloc_status_cb(he,after)
t_hash_elt		*he;
int			after;
{
  t_debug_malloc_elt	*dme;

  dme = (t_debug_malloc_elt *)(he->value);
  if (dme->id >= after)
    debug_malloc_elt_show(dme);
  return (0);
}

int			debug_malloc_status_cmp(p1,p2)
VOID_PTR		*p1;
VOID_PTR		*p2;
{
  t_hash_elt		*he1;
  t_hash_elt		*he2;
  t_debug_malloc_elt	*dme1;
  t_debug_malloc_elt	*dme2;

  he1 = (t_hash_elt *)(*p1);
  he2 = (t_hash_elt *)(*p2);
  dme1 = (t_debug_malloc_elt *)(he1->value);
  dme2 = (t_debug_malloc_elt *)(he2->value);
  return (dme1->id - dme2->id);
}

VOID_FUNC	debug_malloc_status(after)
int		after;
{
  fprintf(stderr,"%d bytes allocated\n",debug_malloc_total);
  fprintf(stderr,"%d bytes not freed (%d ptrs)\n",
	  debug_malloc_total2,
	  id_count(debug_malloc_id));
  fprintf(stderr,"%d/%d calls ok to malloc\n",
	  debug_malloc_calls_ok,
	  debug_malloc_calls);
  if (more_verbose)
    {
      id_walk_sorted_cmp(debug_malloc_id,
			 (t_id_walk_proc)debug_malloc_status_cb,
			 (t_vec_cmp_proc)debug_malloc_status_cmp,
			 (VOID_PTR)after);
    }
}

VOID_PTR			malloc_comment(size,file,line,comment)
size_t				size;
char				*file;
int				line;
char				*comment;
{
  VOID_PTR			ptr;
  t_status			status;

  debug_malloc_calls++;
  if (ptr = real_malloc(size))
    {
      if (debug_malloc_id)
	{
	  t_debug_malloc_elt	*dme;
	  
	  if ((dme = (VOID_PTR)real_malloc(sizeof (t_debug_malloc_elt))) 
	      == NULL)
	    {
	      fprintf(stderr,"internal debug_malloc error: real_malloc\n");
	      abort();
	    }
	  dme->ptr = ptr;
	  dme->size = size;
	  dme->id = debug_malloc_ident++;
	  dme->file = file;
	  dme->line = line;
	  dme->comment = comment;
	  debug_malloc_total += size;
	  debug_malloc_total2 += size;
	  debug_malloc_calls_ok++;
	  if ((status = id_add(debug_malloc_id,
			       ptr,
			       dme)) < 0)
	    {
	      fprintf(stderr,"internal debug_malloc error: id_add: %s\n",
		      err_msg(-status));
	      abort();
	    }
	}
    }
  return (ptr);
}
  
VOID_PTR		malloc(size)
size_t			size;
{
  return (malloc_comment(size,"?",0,"?"));
}

VOID_PTR		realloc(old_ptr,size)
VOID_PTR		old_ptr; 
size_t			size;
{
  abort();
}

VOID_FUNC		free_comment(ptr,comment)
VOID_PTR		ptr;
char			*comment;
{
  if (debug_malloc_id)
    {
      t_status		status;
      t_hash_elt	*he;
      
      if (he = id_get(debug_malloc_id,ptr))
	{
	  t_debug_malloc_elt	*dme;
	  
	  dme = (t_debug_malloc_elt *)(he->value);
	  debug_malloc_total2 -= dme->size;
	  real_free(dme);
	  if ((status = id_rm(debug_malloc_id,ptr)) < 0)
	    {
	      fprintf(stderr,"id_rm %p: %s\n",ptr,err_msg(-status));
	      abort();
	    }
	}
      else
	{
	  fprintf(stderr,"trying to free %p `%s'\n",ptr,comment);
	  /*abort();*/
	}
    }
  real_free(ptr);
}

VOID_FUNC		free(ptr)
VOID_PTR		ptr;
{
  free_comment(ptr,"?");
}
#else
VOID_PTR		malloc_comment(size,file,line,comment)
size_t			size;
char			*file;
int			line;
char			*comment;
{
  return (malloc(size));
}

VOID_FUNC		free_comment(ptr,comment)
VOID_PTR		ptr;
char			*comment;
{
  free(ptr);
}
#endif
