#include <config.h>
#include "basic.h"
#include "n_errno.h"
#include "jnetd.h"
#include "socketlb.h"
#include "parser.h"
#include "log.h"
#include "n_file.h"
#ifdef HAVE_PASSWD_H
# include <passwd.h>
#endif
#ifdef HAVE_PWD_H
# include <pwd.h>
#endif
#ifdef HAVE_GRP_H
# include <grp.h>
#endif
#ifdef HAVE_SHADOW_H
# include <shadow.h>
#endif
#ifdef HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif
#ifndef RLIM_INFINITY
# define RLIM_INFINITY (long int) (~0UL >> 1)
#endif
#include "tools.h"

/* Returns the name of a service according to a ServiceContext structure :
 * its description, the port name or the port number */

char *tools_servicenametxt(const ServiceContext * const servicecontext)
{
   static char buf[TOOLS_MAXPORTSTRINGLEN + 1];
   
   if (servicecontext->description != NULL) {
      return servicecontext->description;
   } else if (servicecontext->servicename.name != NULL) {
      return servicecontext->servicename.name;
   }
#ifdef HAVE_SNPRINTF
   snprintf(buf, TOOLS_MAXPORTSTRINGLEN, "%d", servicecontext->servicename.port);   
#else
   sprintf(buf, "%d", servicecontext->servicename.port);
#endif
   
   return buf;
}

struct env_table {
    const char *name;
    const int len;
};

static const struct env_table badenv_table[] = {
   { (const char *) "IFS=", 4 },
   { (const char *) "LD_", 3 },
   { (const char *) "_RLD", 5 },
   { (const char *) "SHLIB_PATH=", 11 },
   { (const char *) "LIBPATH=", 8 },
   { (const char *) "KRB_CONF", 8 },
   { (const char *) "ENV=", 4 },
   { (const char *) "BASH_ENV=", 9 },
   { NULL, 0 }
};

/* Cleans some nasty environment variables */

extern char **environ;   

void tools_cleanenv(void)
{
   char **envp = environ;
   const struct env_table *bad;
   char **cur;
   
   for (cur = envp; *cur; cur++) {
      for (bad = badenv_table; bad->name; bad++) {
	 if (strncmp(*cur, bad->name, (size_t) bad->len) == 0) {
	    char **move;
	    
	    for (move = cur; *move; move++) {
	       *move = *(move + 1);
	    }	    
	    cur--;	    
	    break;
	 }
      }
   }
}

/* Change the current uid/gid of the current process */

int tools_changeid(const char * const user, const char * const group)
{
   struct passwd *pwd;
   struct group *grp;

   if (group != NULL) {
      grp = getgrnam(group);
   }
   if (user != NULL && geteuid() != 0) {
      log_log(LOG_ERR, NULL, 
	      _("Need supervisor priviledges to change uid/gid"));
      
      return -1;
   } else if (user == NULL) {
      return 0;
   }
   if ((pwd = getpwnam(user)) == NULL) {
      log_log(LOG_ERR, NULL,
	      _("Can't find any user called [%s] - Process not launched"),
	      user);
      
      return -1;
   }
   if (group != NULL && (grp = getgrnam(group)) != NULL) {
      if (setgid(grp->gr_gid) < 0) {
	 log_log(LOG_ERR, NULL, _("Unable to switch to group [%s] <%s>"),
		 group, newstrerror(errno));
	 
	 return -1;
      }
   } else {
      if (setgid(pwd->pw_gid) < 0) {
	 log_log(LOG_ERR, NULL, _("Can't set default user group <%s>"),
		 newstrerror(errno));
	 
	 return -1;
      }
   }
#ifdef HAVE_INITGROUPS
   if (group == NULL && initgroups(user, pwd->pw_gid) < 0) {
      log_log(LOG_ERR, NULL,
	      _("Can't set user groups for [%s] <%s>"), 
	      user, newstrerror(errno));
      
      return -1;
   }
#endif
#ifdef HAVE_SETRESUID
   if (setresuid(pwd->pw_uid, pwd->pw_uid, 0) < 0) {
#endif
      if (setuid(pwd->pw_uid) < 0) {
	 log_log(LOG_ERR, NULL, _("Unable to switch to user [%s] <%s>"),
		 user, newstrerror(errno));
	 
	 return -1;
      }
#ifdef HAVE_SETEUID
      if (seteuid(pwd->pw_uid) < 0) {
	 log_log(LOG_ERR, NULL, _("Unable to switch to user [%s] <%s>"), 
		 user, newstrerror(errno));
	 
      return -1;
      }
#endif
#ifdef HAVE_SETRESUID
   }
#endif
   tools_cleanenv();
   
   return 0;
}

/* Return 1 if a name matches the regex */

int tools_namesmatchrx(const char **names, regex_t *rx)
{
   if (rx == NULL || names == NULL) {
      return -1;
   }
   while (*names != NULL) {
      if (regexec(rx, *names, 0, 0, 0) == 0) {
	 return 1;
      }
      names++;
   }
      
   return 0;
}

/* Needed for GNUregex */

void printchar(int c)
{
   putchar(c);
}

/* See, according to the IP options we found, if we should hangup or not */

const char *tools_seeipoptions(const SockEvent * const event)
{   
   SocketOptions options;
   
   if (socket == NULL) {
      return NULL;
   }
   options = event->options;
   
   if ((options & SOCKET_OPTION_SSRR)) {
      return _("strict source routing enabled");
   }
   if ((options & SOCKET_OPTION_LSRR)) {
      return _("loose source routing enabled");
   }
   
   return NULL;
}

/* The following function adapted from Stevens, "Advanced Programming in the
 * Unix Environment", p. 418,  initializes the the standalone daemon */

int tools_initdaemon(void)
{
   int i;
   
   pid_t pid, procgp;
   
   if ((pid = fork()) < 0) {
     return -1;
   } else if (pid != 0) {
      exit(EXIT_SUCCESS);
   }
   
#ifdef HAVE_SETSID
   if ((procgp = setsid()) == -1) {
      exit(EXIT_FAILURE);
   }
#else
# ifdef HAVE_SETPGRP
#  ifdef SETPGRP_VOID
   if ((procgp = setpgrp()) == -1) {
      exit(EXIT_FAILURE);
   }     
#  else
   if ((procgp = setpgid(getpid(), 0)) == -1) {
      exit(EXIT_FAILURE);
   }
#  endif
# endif
#endif    
   umask(0);   
   for (i = 3; i < SOCKMAX_FD; i++) {
      close(i);
   }
   return 0;
}

/* Set a resource limit */

int tools_setlimit(const int limitnb, long value)
{
#ifdef HAVE_SETRLIMIT
   static struct rlimit lim;
   int nb = -1;
   
   if (value < 0) {
      value = RLIM_INFINITY;
   }
   switch (limitnb) {
# ifdef RLIMIT_CPU
    case TOOLS_LIMIT_CPU :
      nb = RLIMIT_CPU;
      break;
# endif
# ifdef RLIMIT_FSIZE
    case TOOLS_LIMIT_FSIZE :
      nb = RLIMIT_FSIZE;
      break;
# endif
# ifdef RLIMIT_DATA
    case TOOLS_LIMIT_DATA :
      nb = RLIMIT_DATA;
      break;
# endif
# ifdef RLIMIT_STACK
    case TOOLS_LIMIT_STACK :
      nb = RLIMIT_STACK;
      break;
# endif
# ifdef RLIMIT_CORE
    case TOOLS_LIMIT_CORE :
      nb = RLIMIT_CORE;
      break;
# endif
# ifdef RLIMIT_RSS
    case TOOLS_LIMIT_RSS :
      nb = RLIMIT_RSS;
      break;
# endif
# ifdef RLIMIT_NPROC
    case TOOLS_LIMIT_NPROC :
      nb = RLIMIT_NPROC;
      break;
# endif
# ifdef RLIMIT_NOFILE
    case TOOLS_LIMIT_NOFILE :
      nb = RLIMIT_NOFILE;
      break;
# elif defined (RLIMIT_OFILE)
    case TOOLS_LIMIT_NOFILE :
      nb = RLIMIT_OFILE;
      break;      
# endif
# ifdef RLIMIT_MEMLOCK
    case TOOLS_LIMIT_MEMLOCK :
      nb = RLIMIT_MEMLOCK;
# endif      
   }
   if (nb == -1) {
      return 1;
   }
   if (value==-1) {
      lim.rlim_max = RLIM_INFINITY;
      lim.rlim_cur = RLIM_INFINITY;
   } else {
      lim.rlim_max = value;
      lim.rlim_cur = value;
   }
   if (setrlimit(nb, &lim) < 0) {
      return -1;
   } 
   return 0;
#else
   return 1;
#endif
}

/* Safely sets an environment variable - Needs to be rewritten with 'environ' */

int tools_putenv(const char * const envar, const char * const what)
{   
   static int first;
   
   if (envar == NULL || *envar == 0 || what == NULL) {
      return -1;
   }
#ifdef TOOLS_USE_SYSTEM_ENVFUNCS
# ifdef HAVE_SETENV
   
   return setenv(envar, what, 1);
# elif defined(HAVE_PUTENV)
     {
	char tmp[LINE_MAX + 1];
	size_t u;
	
	strncpy(tmp, envar, (size_t) LINE_MAX - 2);
	strcat(tmp, "=");
	u = strlen(tmp);
	strncpy(tmp + u, what, (size_t) LINE_MAX - u - 1);
	
	return putenv(tmp);
     }
# endif
#else
     {
	char tmp[LINE_MAX + 1];
	char **envp = environ;
	char **envtmp;
	char *dp;
	size_t u;	
	
	strncpy(tmp, envar, (size_t) LINE_MAX - 2);
	strcat(tmp, "=");
	u = strlen(tmp);
	strncpy(tmp + u, what, (size_t) LINE_MAX - u - 1);

	u = 0;
	while (*envp != NULL) {
	   u++;
	   envp++;
	}
	envp = malloc((u + TOOLS_ENVGAP) * (sizeof *environ));
	if (envp == NULL) {
	   return -1;
	}
	if (u != 0) {
	   memcpy(envp, environ, u * (sizeof *environ));
	}
	if ((dp = strdup(tmp)) == NULL) {
	   return -1;
	}
	envp[u] = dp;
	envp[u + 1] = NULL;
	envtmp = environ;
	environ = envp;
	
	if (first == 0) {
	   first = 1;
	} else if (envtmp != NULL) {
	   free(envtmp);
	} 
     }
#endif
   
   return 0;
}

/* Set TCP-related environment variables like UCSPI-TCP */

int tools_setucspienv(const SocketInfo * const socketinfo)
{
   tools_putenv("PROTO", "TCP");   
   if (*socketinfo->localport != 0) {
      tools_putenv("TCPLOCALPORT", socketinfo->localport);
   }
   if (*socketinfo->localip != 0) {
      tools_putenv("TCPLOCALIP", socketinfo->localip);
   }
   if (*socketinfo->localhost != 0) {
      tools_putenv("TCPLOCALHOST", socketinfo->localhost);
   }
   if (*socketinfo->remoteport != 0) {
      tools_putenv("TCPREMOTEPORT", socketinfo->remoteport);
   }
   if (*socketinfo->remoteip != 0) {
      tools_putenv("TCPREMOTEIP", socketinfo->remoteip);
   }
   if (*socketinfo->remotehost != 0) {
      tools_putenv("TCPREMOTEHOST", socketinfo->remotehost);
   }
   if (*socketinfo->remoteinfo != 0) {
      tools_putenv("TCPREMOTEINFO", socketinfo->remoteinfo);
   }   
   
   return 0;
}

/* shamelessly stolen from glibc documentation by uweo */
int tools_set_cloexec_flag(int desc, int value)
{
  int oldflags = fcntl(desc, F_GETFD, 0);
  /* If reading the flags failed, return error indication now. */
  if (oldflags < 0) {
    return oldflags;
  }
  /* Set just the flag we want to set. */
  if (value != 0) {
    oldflags |= FD_CLOEXEC;
  } else {
    oldflags &= ~FD_CLOEXEC;
  }
  /* Store modified flag word in the descriptor. */
  return fcntl(desc, F_SETFD, oldflags);
}
