#include <config.h>
#include "basic.h"
#include "n_errno.h"
#define DEFINE_PARSER_GLOBALS
#include "n_file.h"
#include "n_string.h"
#include "str.h"
#include "jnetd.h"
#include "log.h"
#include "parser.h"

static Lotus *findtoken(const char * const token)
{
   register Lotus *tokenindexpnt = tokenindex;
   size_t t = (sizeof tokenindex) / (sizeof tokenindex[0]);

   if (token != NULL) {
      do {
	 if (strcasecmp(tokenindexpnt->token, token) == 0) {
	    return tokenindexpnt;
	 }
	 tokenindexpnt++;
	 t--;
      } while (t != 0);
   }
   
   return NULL;
}

static long parser_atol(const char * const s)
{
   char *p;
   long r;
   errno = 0;
   r = strtol(s, &p, 0);
   if (errno != 0) {
     return 0;
   }
   switch(*p) {
    case 'k':
    case 'K': r *= 1024; break;
    case 'm':
    case 'M': r *= 1024 * 1024; break;
    case 'g': 
    case 'G': r *= 1024 * 1024 * 1024; break;
   }
   return r;
}

static int parser_checknumber(const char * const txt)
{
   return isdigit(*((const unsigned char *) txt));
}

static Port parser_parseport(char * const item)
{   
   char *svccut;
   Port port = { NULL, -1 };
   
   if ((svccut = strchr(item, PARSER_SERVICEEQUIV)) != NULL) {
      *svccut = 0;
   }
   if ((port.name = strdup(item)) == NULL) {
      log_log(LOG_ERR, NULL, _("No more memory to store a port number"));
      port.name = NULL;
      
      return port;
   }
   if (svccut != NULL) {
      svccut++;
      if (parser_checknumber(svccut) == 0) {
	 log_log(LOG_ERR, NULL, _("Missing port number"));
	 
	 return port;
      }
      port.port = atoi(svccut);
      if (port.port < 0) {
	 log_log(LOG_ERR, NULL, _("Invalid port number"));
	 port.port = -1;
      }
   }
   return port;
}
   
static int parser_defaultcontext(ServiceContext * const context)
{
   static const ServiceContext defaultcontext = {
      NULL,                              /* description */
      NULL,                              /* hostname */
      { NULL, -1 },                      /* servicename */
      0L,                                /* corelimit */
      PARSER_DEFAULTPROTOCOL,            /* proto */
      PARSER_DEFAULTENDPOINT,            /* endpoint */
      255,                               /* wait */
      0L,                                /* session_timelimit */
      0,                                 /* session_cpuplimit */
      0L,                                /* session_cpulimit */
      0L,                                /* session_filelimit */
      0L,                                /* session_datalimit */
      0L,                                /* session_stacklimit */
      0,                                 /* session_fdlimit */
      0,                                 /* service_cpuplimit */
      0L,                                /* service_cpulimit */
      0U,                                /* service_nblimit */
      0,                                 /* service_cur_nblimit */
      0,                                 /* cnx_timeout */
      60,                                /* abuse_detect_time */
      0,                                 /* abuse_detect_sameservice */
      100,                               /* abuse_detect_samehost */
      500,                               /* abuse_detect_samenet */
      600,                               /* abuse_denytime */
      NULL,                              /* abuse_denyservices */
      NULL,                              /* abuse_run */
      NULL,                              /* abuse_logfile */
      NULL,                              /* rx_allow */
      NULL,                              /* rx_allow_except */      
      NULL,                              /* rx_deny */
      NULL,                              /* rx_deny_except */
      NULL,                              /* rx_allow_ip */
      NULL,                              /* rx_allow_ip_except */      
      NULL,                              /* rx_deny_ip */
      NULL,                              /* rx_deny_ip_except */      
      NULL,                              /* user */
      NULL,                              /* group */
      NULL,                              /* chroot */
      NULL,                              /* logfile */
      NULL,                              /* run */
      NULL,                              /* env */
      NULL,                              /* filter */
      NULL,                              /* gateway_to_host */
      { NULL, -1 },                      /* gateway_to_port */
      NULL,                              /* gateway_from */
      0,                                 /* spawn_nblimit */
      0,                                 /* spawn_cpuplimit */
      0,                                 /* spawn_cpulimit */
      0,                                 /* spamfilter */
      BUILTIN_NONE,                      /* builtin */
      0,                                 /* hasaction */
      {                                  /* servicenetworkstatus */
	 NULL,                           /* . socket */
	 0                               /* . dgramfd */
      },
      NULL                               /* next */
   };
   
   if (context == NULL) {
      return -1;
   }
   *context = defaultcontext;
   
   return 0;
}

static ServiceContext *parser_createcontext(void)
{
   ServiceContext *servicecontext;
   
   if ((servicecontext =
	(ServiceContext *) malloc(sizeof *servicecontext)) == NULL) {
      log_log(LOG_ERR, NULL, _("Service allocation"));
      
      return NULL;
   }
   if (parser_defaultcontext(servicecontext) != 0) {
      return NULL;
   }
   
   return servicecontext;
}

ParserResult parser_parseconfigfile(const char *file)
{
   FILE *fp;
   char *tmp;
   Lotus *tokenlote;   
   char buf[LINE_MAX + 1];
   char *bufpnt = buf;   
   unsigned long linecounter = 0;
   ParserContext parsercontext;
   ServiceContext *firstservicecontext = NULL;
   ServiceContext *lastservicecontext = NULL;
   ServiceContext tmpservicecontext[PARSER_MAXDEPTH];
   ParserResult parserresult = { NULL, 0 };
      
   {
      register ServiceContext *tmpservicecontextpnt = tmpservicecontext;
      size_t t = PARSER_MAXDEPTH;      
      
      do {
	 parser_defaultcontext(tmpservicecontextpnt++);
	 t--;
      } while (t != 0);
   }      
   parsercontext.currentservicecontext = &tmpservicecontext[0];
   parsercontext.depth = 0;   
   if ((fp = fopen(file, "rt")) == NULL) {
      log_log(LOG_ERR, NULL, _("Can't read the configuration file [%s] <%s>"),
	      file, newstrerror(errno));
      parserresult.error = 1;
      
      return parserresult;
   }   
   while (str_fgetsn(buf, LINE_MAX, fp) != NULL) {
      linecounter++;
      if (*buf == 0 || *buf == PARSER_COMMENT) {
	 continue; 
      }
      bufpnt = buf;      
      while ((tmp = str_strtokspace(bufpnt)) != NULL) {
	 bufpnt = NULL;	 	 	 
	 if (*tmp == PARSER_STARTBLOCK) {
	    if (parsercontext.depth < PARSER_MAXDEPTH) {
	       parsercontext.depth++;
	       *(parsercontext.currentservicecontext =
		 &tmpservicecontext[parsercontext.depth]) =
		 tmpservicecontext[parsercontext.depth - 1];
	    } else {
	       log_log(LOG_ERR, NULL, 
		       _("Too much depth in recursion at line [%u]"), linecounter);
	    }	       
	    continue;
	 } else if (*tmp == PARSER_ENDBLOCK) {
	    if (parsercontext.currentservicecontext->hasaction != 0 &&
		(parsercontext.currentservicecontext->servicename.name != NULL ||
		 parsercontext.currentservicecontext->servicename.port > 0)) {
	       ServiceContext *newservicecontext;
	       
	       parsercontext.currentservicecontext->hasaction = 0;	       
	       if ((newservicecontext = parser_createcontext()) == NULL) {
		  log_log(LOG_ERR, NULL, 
			  _("Not enough memory to register a service at line [%u] <%s>"),
			  linecounter, newstrerror(errno));
	       } else {
		  *newservicecontext = *parsercontext.currentservicecontext;
		  if (firstservicecontext == NULL) {
		     firstservicecontext = newservicecontext;
		  } else {
		     lastservicecontext->next = newservicecontext;
		  }
		  lastservicecontext = newservicecontext;
	       }
	    }
	    if (parsercontext.depth > 0) {
	       parsercontext.depth--;
	       parsercontext.currentservicecontext =
		 &tmpservicecontext[parsercontext.depth];	       
	    } else {
	       log_log(LOG_ERR, NULL, 
		       _("End of block doesn't match any start of block at line [%u]"),
		       linecounter);
	    }
	    continue;
	 } else if ((tokenlote = findtoken(tmp)) == NULL) {
	    log_log(LOG_ERR, NULL, 
		    _("Unknown token : [%s] line [%u]"), tmp, linecounter);
	    continue;
	 }
	 if (tokenlote->fct(&parsercontext) != 0) {
	    log_log(LOG_ERR, NULL, _("Check line [%u]"), linecounter);
	 }
      }
   }
   fclose(fp);
   parserresult.firstservicecontext = firstservicecontext;
   if (firstservicecontext == NULL) {
      parserresult.error = 1;
   }
   
   return parserresult;
}
   
PRO(CORELIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number to specify the core limit."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->corelimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SESSION_TIMELIMIT)
{   
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number of secondes to specify a max time limit."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_timelimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;   
}

PRO(SESSION_CPULIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number of secondes to specify a CPU usage limit."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_cpulimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SESSION_FILELIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number of bytes to specify a maximum file size."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_filelimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SESSION_DATALIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number of bytes to specify the maximum data size."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_datalimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SESSION_STACKLIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number of bytes to specify the maximum stack size."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_stacklimit = (long) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SESSION_FDLIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a number : the maximum wanted number of descriptors."));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->session_fdlimit = (int) atoi(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SERVICE)
{
   char *item;
   
   if (GETNEXTOKEN) {
      Port * const port = &parsercontext->currentservicecontext->servicename;

      *port = parser_parseport(item);
   }
   return 0;
}

PRO(RUN)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   char **fifo;
   size_t argscounter = PARSER_MAXRUNARGS;
   
   if ((fifo = (char **) malloc(sizeof (char *) * (PARSER_MAXRUNARGS + 1)))
       == NULL) {
      log_log(LOG_ERR, NULL, _("Can't store the command-line definition"));
      
      return -1;
   }
   currentservicecontext->run = fifo;
   while (GETNEXTOKEN) {
      if (argscounter == 0) {
	 log_log(LOG_ERR, NULL, _("Too many command-line arguments"));
	 
	 return -1;
      }
      if ((*fifo++ = strdup(item)) == NULL) {
	 log_log(LOG_ERR, NULL, _("Missing memory for command-line arguments"));
	 
	 return -1;
      }
      argscounter--;
   }
   *fifo = NULL;
   currentservicecontext->hasaction = 1;
   
   return 0;
}

PRO(BUILTIN)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   register const ParserBuiltins *parserbuiltinspnt = parserbuiltins;
   size_t t = (sizeof parserbuiltins) / (sizeof parserbuiltins[0]);
   
   if (GETNEXTOKEN == 0) {
      log_log(LOG_ERR, NULL, _("Which builtin ?"));
      
      return -1;
   }
   do {
      if (strcasecmp(parserbuiltinspnt->keyword, item) == 0) {
	 currentservicecontext->builtin = parserbuiltinspnt->builtin;
	 currentservicecontext->hasaction = 1;
	 
	 return 0;	 
      }
      parserbuiltinspnt++;
      t--;
   } while (t != 0);
   log_log(LOG_ERR, NULL, _("Unknown builtin"));
   
   return 1;
}

PRO(GATEWAY_TO)
{
   char *item;
   ServiceContext *servicecontext = parsercontext->currentservicecontext;

   if (GETNEXTOKEN) {
      if ((servicecontext->gateway_to_host = strdup(item)) == NULL) {
	 log_log(LOG_ERR, NULL, _("Can't memorize the gateway target"));
	 
	 return -1;
      }
      if (GETNEXTOKEN) {
	 servicecontext->gateway_to_port = parser_parseport(item);
      } else {
	 log_log(LOG_ERR, NULL, 
		 _("Need to specify the target port for a gateway"));
	 
	 return -1;
      }
   } else {
      log_log(LOG_ERR, NULL, _("Where should be the gate to ?"));
      
      return -1;
   }
   servicecontext->hasaction = 1;
   servicecontext->builtin = BUILTIN_GATEWAY;
      
   return 0;
}

PRO(PROTO)
{
   char *item;
   ServiceContext *currentservicecontext;
   
   if (GETNEXTOKEN) {
      currentservicecontext = parsercontext->currentservicecontext;
      if ((currentservicecontext->proto = strdup(item)) == NULL) {
	 log_log(LOG_ERR, NULL, _("Protocol name storage"));
	 
	 return -1;	 
      }
   } else {
      log_log(LOG_ERR, NULL, 
	      _("ERROR : Need a protocol name (tcp, udp, ...)"));
      
      return -1;
   }
   if (currentservicecontext->endpoint != NULL &&
       currentservicecontext->wait != 255) {
      
      return 0;
   }
   {
      typedef struct DefaultAssociates_ {
	 const char *proto;
	 const char *endpoint;
	 const unsigned char wait;
      } DefaultAssociates;
      static const DefaultAssociates defaultassociates[] = {
	 { "ip", "raw", 1 },
	 { "icmp", "raw", 1 },
	 { "ggp", "stream", 0 },
	 { "tcp", "stream", 0 },
	 { "egp", "stream", 0 },
	 { "pup", "seqpacket", 0 },
	 { "udp", "dgram", 1 },
	 { "rdp", "dgram", 1 }
      };
      register const DefaultAssociates *defaultassociatespnt = defaultassociates;
      size_t t = (sizeof defaultassociates) / (sizeof defaultassociates[0]);
      
      do {
	 if (strcasecmp(defaultassociatespnt->proto, 
			currentservicecontext->proto) == 0) {
	    if (strcmp(currentservicecontext->endpoint, 
		       PARSER_DEFAULTENDPOINT) == 0) {
	       currentservicecontext->endpoint = defaultassociatespnt->endpoint;
	    }
	    if (currentservicecontext->wait == 255) {
	       currentservicecontext->wait = defaultassociatespnt->wait;
	    }
	    break;
	 }
	 defaultassociatespnt++;
	 t--;
      } while (t != 0);
   }
         
   return 0;
}
    
static int PIF(GENERICSTRINGET)(char ** const pnt,
				const char * const str1, 
				const char * const str2)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if ((*pnt = strdup(item)) == NULL) {
	 log_log(LOG_ERR, NULL, str1, newstrerror(errno));
	 
	 return -1;
      }
   } else {
      log_log(LOG_ERR, NULL, str2);
      
      return -1;
   }
   
   return 0;
}

PRO(GATEWAY_FROM)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->gateway_from,
      _("Can't memorize a gateway bind address <%s>"),
      _("What should the gateway bind ?"));
}

PRO(HOST)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->hostname,
      _("Can't memorize a bind address"),
      _("What should I bind ?"));
}

PRO(DESCRIPTION)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->description,
      _("Can't store a description"),
      _("Missing description"));
}

PRO(USER)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->user,
      _("Can't store a user name"),
      _("Missing user name"));
}

PRO(GROUP)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->user,
      _("Can't store a group name"),
      _("Missing group name"));
}

PRO(CHROOT)
{
   return PIF(GENERICSTRINGET)
     (&parsercontext->currentservicecontext->chroot,
      _("Can't store a chroot() path"),
      _("Missing path for a chroot() option"));
}

PRO(ALLOW)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the allowed host names"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_allow = comp;
   } else {
      currentservicecontext->rx_allow = NULL;
   }

   return 0;
}

PRO(ALLOW_EXCEPT)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the excepted allowed host names"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_allow_except = comp;
   } else {
      currentservicecontext->rx_allow_except = NULL;
   }

   return 0;
}

PRO(DENY)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the denied host names"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_deny = comp;
   } else {
      currentservicecontext->rx_deny = NULL;
   }

   return 0;
}

PRO(DENY_EXCEPT)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the excepted denied host names"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_deny_except = comp;
   } else {
      currentservicecontext->rx_deny_except = NULL;
   }

   return 0;
}

PRO(ALLOW_IP)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the allowed IP addresses"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_allow_ip = comp;
   } else {
      currentservicecontext->rx_allow_ip = NULL;
   }

   return 0;
}

PRO(ALLOW_IP_EXCEPT)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the excepted allowed IP addresses"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_allow_ip_except = comp;
   } else {
      currentservicecontext->rx_allow_ip_except = NULL;
   }

   return 0;
}

PRO(DENY_IP)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the denied IP addresses"));
	 
	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_deny_ip = comp;
   } else {
      currentservicecontext->rx_deny_ip = NULL;
   }

   return 0;
}

PRO(DENY_IP_EXCEPT)
{
   char *item;
   ServiceContext *currentservicecontext =
     parsercontext->currentservicecontext;
   regex_t *comp;
   int err;
   
   if (GETNEXTOKEN) {
      if ((comp = (regex_t *) malloc(sizeof *comp)) == NULL) {
	 log_log(LOG_ERR, NULL,
		 _("No more memory to store the excepted denied IP addresses"));

	 return -1;
      }
      if ((err = regcomp(comp, item, REG_EXTENDED | REG_ICASE)) != 0) {
	 char errbuf[LINE_MAX + 1];
	 
	 regerror(err, comp, errbuf, sizeof LINE_MAX);
	 log_log(LOG_ERR, NULL,
		 _("Suspicious regex : [%s] <%s>"), item, errbuf);
	 
	 return -1;
      }
      currentservicecontext->rx_deny_ip_except = comp;
   } else {
      currentservicecontext->rx_deny_ip_except = NULL;
   }

   return 0;
}

PRO(SERVICE_NBLIMIT)
{
   char *item;
   
   if (GETNEXTOKEN) {
      if (parser_checknumber(item) == 0) {
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : Need a maximum number of instances"));
	      
	 return -1;
      }
      parsercontext->currentservicecontext->service_nblimit =
	(size_t) parser_atol(item);
   } else {
      goto cr0;
   }   
   return 0;
}

PRO(SPAMFILTER)
{
   char *item;
   int t;
   
   if (GETNEXTOKEN) {      
      if (parser_checknumber(item) == 0) {	 
	 cr0:
	 log_log(LOG_ERR, NULL, 
		 _("ERROR : SPAMFILTER needs a number : 0 to disable, 1 or 2 to enable"));
	      
	 return -1;
      }
      t = atoi(item);
      if (t < 0 || t > 2) {
	 goto cr0;
      }
      parsercontext->currentservicecontext->spamfilter = t;
   } else {
      goto cr0;
   }   
   return 0;
}
