#include <config.h>
#include "basic.h"
#include "n_errno.h"
#include "socketlb.h"
#include "n_wait.h"
#include "signallb.h"
#include "jnetd.h"
#include "parser.h"
#include "version.h"
#include "log.h"
#include "tools.h"
#include "n_regex.h"
#ifdef HAVE_LOCALE_H
# include <locale.h>
#endif

typedef struct ChildTab_ {
   int fd;
   Socket *socket;
} ChildTab;

static MemBlock *m_childtab;

RETSIGTYPE sigterm(int dummy)
{
   int fd = 0;

   (void) dummy;
   log_log(LOG_WARNING, NULL, _("Shutting down the server"));
   while (fd <= SOCKMAX_FD) {
      socket_realclose(fd);
      fd++;
   }
   sleep((unsigned int) SOCKET_LINGER);
   log_log(LOG_WARNING, NULL, _("Server off"));   
   exit(EXIT_SUCCESS);
#if RETSIGTYPE != void
   return 0;
#endif
}

RETSIGTYPE sigchild(int dummy)
{
   int status;
   pid_t foundpid;

   (void) dummy;
#ifdef HAVE_WAIT3
   foundpid = wait3(&status, WNOHANG, NULL);
#elif defined (WNOHANG)	       
   foundpid = waitpid(-1, &status, WNOHANG);
#else
   foundpid = wait(&status);
#endif
   if (foundpid > 0) {
      ChildTab * const childtab = (ChildTab *) m_childtab->space;
      ServiceContext *servicecontext;
      ChildTab *child;
      
      if (m_childtab->size < (foundpid * sizeof (ChildTab))) {
	 PRINTWARNING(_("PID can't fit in childtab"));
      }
      child = &childtab[foundpid];
      servicecontext = (ServiceContext *) child->socket->geek;
      if (servicecontext == NULL) {
	 log_log(LOG_ERR, NULL, _("No service context ?!"));
	 signal_catch(SIGCHLD, sigchild);	 
#if RETSIGTYPE != void
	 return -1;
#else
	 return;
#endif
      }
      if (servicecontext->service_cur_nblimit > 0) {
	 servicecontext->service_cur_nblimit--;
      }
      if (socket_isstream(child->socket)) {
	 socket_hangup(child->socket, child->fd);
      } else {
	 if (servicecontext->servicenetworkstatus.dgramfd != 0) {
	    socket_fdlisten(child->socket->sockets,
			    servicecontext->servicenetworkstatus.dgramfd);
	    servicecontext->servicenetworkstatus.dgramfd = 0;
	 }
      }
      if (WIFEXITED(status) != 0) {
	 if (WEXITSTATUS(status) == 0) {
	    log_log(LOG_INFO, NULL, 
		    _("[%s] (PID=%d) terminated"),
		    tools_servicenametxt(servicecontext), (int) foundpid);
	 } else {
	    log_log(LOG_NOTICE, NULL,
		    _("[%s] (PID=%d) exited with return code [%d]"),
		    tools_servicenametxt(servicecontext), (int) foundpid,
		    WEXITSTATUS(status));
	 }
      } else {
	 log_log(LOG_WARNING, NULL,
		 _("Process [%d] from service [%s] exited abnormally (limits reached ?), please check [%s]"),
		 (int) foundpid,
		 tools_servicenametxt(servicecontext),
		 servicecontext->run[0]);
      }
   } else {
      log_log(LOG_DEBUG, NULL, _("Suspect SIGCHLD received"));
   }

   signal_catch(SIGCHLD, sigchild);
#if RETSIGTYPE != void
   return 0;
#endif
}

static int jnetd_builtin(SockEvent *event)
{
   ServiceContext *servicecontext;
   
   if (event == NULL) {
      return -1;
   }
   if ((servicecontext = (ServiceContext *) event->socket->geek) == NULL) {
      return -1;
   }
   switch (servicecontext->builtin) {
    case BUILTIN_ECHO :
      socket_print(event->fd, event->line);
      break;
    case BUILTIN_DISCARD :
      break;
    case BUILTIN_INFO :
      socket_print(event->fd, jnetd_version());
      socket_print(event->fd, "\n. \n");
      break;
    case BUILTIN_GATEWAY :
      {
	 MemBlock *memblock;
	 int targetfd;
	 
	 if ((memblock = (MemBlock *) event->socket->geek2) == NULL) {
	    LOG_ERRMARK(NULL, _("No gatewaying table for a gateway ?!"));
	    
	    return -1;
	 }	 
	 targetfd = ((int *) memblock->space)[event->fd];
	 write(targetfd, event->line, event->linelen);
      }
      break;
    default :
      return -1;
   }   
   return 0;
}

/* Check if a host is allowed to connect on a service, based upon regexes
 * JNETD_OLD_ACCESS_CHECK, if defined, matches both the host name and its
 * IP address against the same regex. This is yet deprecated. */

static int checkallowed(const char **names, const char * const ip,
			const ServiceContext *servicecontext)
{
   static const char *ips[2] = { NULL, NULL };
   
   if (servicecontext == NULL || servicecontext->rx_allow == NULL) {
      return -1;
   }
   ips[0] = ip;
#ifdef JNETD_OLD_ACCESS_CHECK            
   if ((tools_namesmatchrx(names, servicecontext->rx_allow) > 0 ||
	tools_namesmatchrx(ips, servicecontext->rx_allow) > 0) &&
       (tools_namesmatchrx(names, servicecontext->rx_allow_except) <= 0 ||
	tools_namesmatchrx(ips, servicecontext->rx_allow_except) <= 0)) {
      
      return 1;
   }
#else
   if ((tools_namesmatchrx(ips, servicecontext->rx_allow_ip) > 0 ||
	tools_namesmatchrx(names, servicecontext->rx_allow) > 0) &&
       (tools_namesmatchrx(ips, servicecontext->rx_allow_ip_except) <= 0 ||
	tools_namesmatchrx(names, servicecontext->rx_allow_except) <= 0)) {
      
      return 1;
   }
#endif
   
   return 0;
}

static int checkdenied(const char **names, const char * const ip,
		       const ServiceContext *servicecontext)
{
   static const char *ips[2] = { NULL, NULL };
   
   if (servicecontext == NULL || servicecontext->rx_deny == NULL) {
      return -1;
   }
   ips[0] = ip;
#ifdef JNETD_OLD_ACCESS_CHECK            
   if ((tools_namesmatchrx(names, servicecontext->rx_deny) > 0 ||
	tools_namesmatchrx(ips, servicecontext->rx_allow) > 0) &&
       (tools_namesmatchrx(names, servicecontext->rx_deny_except) <= 0 ||
	tools_namesmatchrx(ips, servicecontext->rx_deny_except) <= 0)) {
      
      return 1;
   }   
#else
   if ((tools_namesmatchrx(ips, servicecontext->rx_deny_ip) > 0 ||
	tools_namesmatchrx(names, servicecontext->rx_allow) > 0) &&
       (tools_namesmatchrx(ips, servicecontext->rx_deny_ip_except) <= 0 ||
	tools_namesmatchrx(names, servicecontext->rx_deny_except) <= 0)) {
      
      return 1;
   }      
#endif
   
   return 0;
}

int jnetd_setlimits(ServiceContext *servicecontext)
{
   if (servicecontext == NULL) {
      return -1;
   }
   if (tools_setlimit(TOOLS_LIMIT_CORE, servicecontext->corelimit) < 0) {
      return -1;
   }
   if (servicecontext->session_cpulimit != 0 && 
       tools_setlimit(TOOLS_LIMIT_CPU, servicecontext->session_cpulimit) < 0) {
      return -1;
   }
   if (servicecontext->session_filelimit != 0 &&
       tools_setlimit(TOOLS_LIMIT_FSIZE, servicecontext->session_filelimit) < 0) {
      return -1;
   }
   if (servicecontext->session_datalimit != 0 &&
       tools_setlimit(TOOLS_LIMIT_DATA, servicecontext->session_datalimit) < 0) {
      return -1;
   }
   if (servicecontext->session_stacklimit != 0 &&
       tools_setlimit(TOOLS_LIMIT_STACK, servicecontext->session_stacklimit) < 0) {
      return -1;
   }
   if (servicecontext->session_fdlimit != 0 &&
       tools_setlimit(TOOLS_LIMIT_NOFILE, servicecontext->session_fdlimit) < 0) {
      return -1;
   }

   if (servicecontext->session_timelimit > 0) {
      alarm((unsigned int) servicecontext->session_timelimit);
   }
   
   return 0;
}

int main(int argc, const char **argv)
{
   static const char *jnetd_defaultcffile = JNETD_DEFAULTCFFILE;
   Sockets sockets;
   SockEvent *event;
   const char *configfile;
   ParserResult parserresult;

   if (argc < 2) {
      puts(_("Usage : " JNETD_NAME " <configuration file>"));
      exit(EXIT_FAILURE);
   }
   tools_cleanenv();
#ifdef DEBUG
   puts(_("Debug mode enabled : not launching as a daemon."));
#else
   if (tools_initdaemon() != 0) {
      puts(_("Can't spawn the daemon"));
      exit(EXIT_FAILURE);
   }   
#endif
#ifdef LC_ALL
   setlocale(LC_ALL, "");
#endif
   log_opensyslog(JNETD_NAME, LOG_DAEMON);
   log_log(LOG_INFO, NULL, jnetd_version());   
   log_log(LOG_INFO, NULL, _("Starting our heavy job with PID=[%d]"),
	   (int) getpid());
   if ((configfile = argv[1]) == NULL || *configfile == 0) {
      configfile = jnetd_defaultcffile;
   }
   if ((parserresult =
	parser_parseconfigfile(configfile)).error != 0) {
      log_log(LOG_ERR, NULL, _("Unable to parse the configuration file <%s>"),
	      newstrerror(errno));
      
      return EXIT_FAILURE;
   }

   signal_catch(SIGALRM, SIG_IGN);
   signal_catch(SIGPIPE, SIG_IGN);
   signal_catch(SIGTERM, sigterm);
   signal_catch(SIGQUIT, sigterm);   
   signal_catch(SIGKILL, sigterm);      
   signal_catch(SIGCHLD, sigchild);
#ifdef SIGEMT
   signal_catch(SIGEMT, SIG_IGN);
#endif
#ifdef SIGXCPU
   signal_catch(SIGXCPU, SIG_IGN);
#endif
#ifdef SIGXFSZ
   signal_catch(SIGXFSZ, SIG_IGN);
#endif

   puts("");
   puts(jnetd_version());
   printf("Socketlib release %d.%d\n",
	  SOCKETLIB_MAJORVERSION, SOCKETLIB_MINORVERSION);
   puts("(C)oderite 1997-1998 by Jedi/Sector One <j@4u.net>\n");
         
   if (socket_initsockets(&sockets) != 0) {
      return EXIT_FAILURE;
   }
   {
      ServiceContext *servicecontextpnt = parserresult.firstservicecontext;
      
      while (servicecontextpnt != NULL) {
	 if ((servicecontextpnt->servicenetworkstatus.socket =
	      socket_servercreate(&sockets, 
				  servicecontextpnt->hostname, NULL,
				  servicecontextpnt->endpoint,
				  servicecontextpnt->proto,
				  servicecontextpnt->servicename.name,
				  servicecontextpnt->servicename.port))
	     == NULL) {
	    log_log(LOG_ERR, NULL, _("[%s] (%s/%s) not bound <%s>"),
		    tools_servicenametxt(servicecontextpnt),
		    servicecontextpnt->endpoint, servicecontextpnt->proto,
		    newstrerror(errno));
	    goto nextservice;
	 } else {	    
	    log_log(LOG_INFO, NULL, _("Activating service [%s] (%s/%s)"),
		    tools_servicenametxt(servicecontextpnt),
		    servicecontextpnt->endpoint, servicecontextpnt->proto);
	 }
	 socket_optionnodelay(servicecontextpnt->servicenetworkstatus.socket, 0);
	 servicecontextpnt->servicenetworkstatus.socket->geek =
	   (void *) servicecontextpnt;
	 nextservice:
	 servicecontextpnt = servicecontextpnt->next;
      }
   }
   if ((m_childtab = mem_create(JNETD_DEFAULTMAXPID * sizeof (ChildTab)))
	== NULL) {
      log_log(LOG_ERR, NULL, _("Can't create the children status table <%s>"),
	      newstrerror(errno));
	      
      return EXIT_FAILURE;
   }
   for (;;) {
      nextevent:
      signal_unblock(SIGCHLD); 
      event = socket_serverwaitevent(&sockets);

      switch (event->type) {
	 pid_t pid;	 
       case SOCKEVENT_NEWCNX :
	 {
	    const char **names;
	    const char *ipoptmsg;
	    const char *ip;
	    ServiceContext *servicecontext = 
	      (ServiceContext *) event->socket->geek;
	    int allowed, denied;

	    if ((ipoptmsg = tools_seeipoptions(event)) != NULL) {
	       socket_hangup(event->socket, event->fd);	       
	       log_log(LOG_WARNING, NULL, ipoptmsg);
	       goto nextevent;
	    }	    
	    if (servicecontext->spamfilter != 0) {
	       const int t = socket_ispammer(event->socket, event->fd, JNETD_RBLMAP);
	       
	       if (t > 0) {
		  log_log(LOG_WARNING, NULL,
			  _("[%s] is listed as a spammer : connexion to [%s] refused"),
			  socket_getpeernameoraddr(event->socket, event->fd),
			  tools_servicenametxt(servicecontext));
		  socket_print(event->fd, _("\n553 Access denied - Stop spamming, lamer\r\n"));
		  socket_hangup(event->socket, event->fd);
		  goto nextevent;
	       } else if (t < 0) {
		  if (servicecontext->spamfilter == 2) {
		     log_log(LOG_WARNING, NULL,
			     _("An RBL check caused a DNS error. [%s] was connected to [%s] anyway, but check [%s]"),
			     socket_getpeernameoraddr(event->socket, event->fd),
			     tools_servicenametxt(servicecontext),
			     JNETD_RBLMAP);
		  } else {
		     log_log(LOG_WARNING, NULL,
			     _("[%s] wanted to connect to [%s] but an RBL check caused a DNS error, check [%s]"),
			     socket_getpeernameoraddr(event->socket, event->fd),
			     tools_servicenametxt(servicecontext),
			     JNETD_RBLMAP);
		     socket_print(event->fd, _("\nUnable to prove your innocence yet, please try again later\n\n"));
		     socket_hangup(event->socket, event->fd);
		     goto nextevent;	
		  }
	       }
	    }
	    names = socket_getpeername(event->socket, event->fd);
	    ip = socket_getpeeraddr(event->socket, event->fd);
	    allowed = checkallowed(names, ip, servicecontext);
	    denied = checkdenied(names, ip, servicecontext);
	    
	    if (denied < 0) {
	       if (allowed == 0) {
		  refused:
		  log_log(LOG_WARNING, NULL,
			  _("[%s] was not authorized to connect to [%s]"),
			  socket_getpeernameoraddr(event->socket, event->fd),
			  tools_servicenametxt(servicecontext));
		  socket_print(event->fd, _("\nAccess denied - This will reported to our security managers\n\n"));
		  socket_hangup(event->socket, event->fd);
		  goto nextevent;
	       }
	    } else if (denied > 0) {
	       if (allowed <= 0) {
		  goto refused;
	       }
	    }
	    if (servicecontext->service_nblimit  != 0 &&
		servicecontext->service_cur_nblimit >= 
		servicecontext->service_nblimit) {
	       log_log(LOG_INFO, NULL,
		       _("[%s] couldn't join [%s] : [%u] users were already using that service"),
		       socket_getpeernameoraddr(event->socket, event->fd),
		       tools_servicenametxt(servicecontext),
		       servicecontext->service_nblimit);
	       socket_print(event->fd, _("\nAccess denied : too many users\n\n"));
	       socket_hangup(event->socket, event->fd);
	       goto nextevent;
	    }
	 }
	
	 if (((ServiceContext *) event->socket->geek)->builtin == 
	     BUILTIN_GATEWAY) {
	    ServiceContext *servicecontext = 
	      (ServiceContext *) event->socket->geek;
	    Socket *clientsocket;	 
	    char *gw_host = servicecontext->gateway_to_host;
	    Port gw_port = servicecontext->gateway_to_port;
	    
	    clientsocket = socket_clientcreate(&sockets,
					       gw_host, 
					       servicecontext->gateway_from,
					       servicecontext->endpoint,
					       servicecontext->proto,
					       gw_port.name,
					       gw_port.port);
	    if (clientsocket != NULL) {
	       log_log(LOG_INFO, NULL, _("Gatewaying [%s] to [%s] as [%s] for [%s]"),
		       tools_servicenametxt(servicecontext),
		       servicecontext->gateway_to_host,
		       servicecontext->gateway_from,
		       socket_getpeernameoraddr(event->socket, event->fd));
	       socket_optionnodelay(clientsocket, 0);
	       clientsocket->geek = (void *) servicecontext;
	       clientsocket->cookie = 1;
	       clientsocket->cookie2 = event->fd;
	       clientsocket->geek2 = (void *) event->socket;
	       {
		  MemBlock *memblock;
		  int *gwtab;
		  
		  if (event->socket->geek2 == NULL) {
		     if ((memblock = 
			  mem_create((event->fd + 1) * sizeof (int))) == NULL) {
			LOG_ERRMARK(NULL, _("Gateway table creation"));
			break;
		     }
		     event->socket->geek2 = (void *) memblock;
		  } else {
		     memblock = (MemBlock *) event->socket->geek2;
		     if (mem_resize(memblock, (event->fd + 1) * sizeof (int)) != 0) {
			LOG_ERRMARK(NULL, _("Gateway table expansion"));
			break;
		     }
		  }
		  gwtab = (int *) memblock->space;		  
		  gwtab[event->fd] = clientsocket->fd;
		  servicecontext->service_cur_nblimit++;
	       }
	    } else {
	       log_log(LOG_ERR, NULL, _("Unable to tunnel to [%s] (%s/%s) for [%s] <%s>"),
		       servicecontext->gateway_to_host,
		       servicecontext->endpoint, servicecontext->proto,
		       tools_servicenametxt(servicecontext), newstrerror(errno));
	       socket_hangup(event->socket, event->fd);
	    }
	    break;
	 } else if (((ServiceContext *) event->socket->geek)->builtin 
		    != BUILTIN_NONE) {
	    break;
	 }
	 
	 if (socket_isstream(event->socket) == 0) {
	    ServiceContext *servicecontext = 
	      (ServiceContext *) event->socket->geek;
	    ServiceNetworkStatus *servicenetworkstatus = 
	      &servicecontext->servicenetworkstatus;
	    
	    if (servicenetworkstatus->dgramfd != 0) {
	       socket_fdunlisten(&sockets, servicenetworkstatus->dgramfd);
	       break;
	    }	    
	 }
	 
	 signal_block(SIGCHLD);
#ifdef F_SETFD
	 fcntl(event->fd, F_SETFD, 0);
#endif
	   {
	      int fo_eventfd = event->fd;
	      SocketInfo fo_eventsocketinfo = event->socketinfo;
	      ServiceContext fo_servicecontext = *((ServiceContext *) event->socket->geek);
	      
	      switch (pid = fork()) {
	       case -1 :
		 LOG_ERRMARK(NULL, _("Fork() failed"));
		 log_log(LOG_ERR, NULL, _("[%s] not spawned"), 
			 tools_servicenametxt((ServiceContext *) event->socket->geek));
		 break;
	       case 0 :
		 close(0);
		 close(1);
		 fcntl(fo_eventfd, F_SETFD, 0);
		 if (socket_isstream(event->socket)) {
		    socket_serverclose(event->socket);
		    tools_setucspienv(&fo_eventsocketinfo); 
		 }
		 if (fd_move(0, fo_eventfd) < 0) {
		    log_log(LOG_DEBUG, NULL, _("File descriptors moving failure <%s>"),
			    newstrerror(errno));
		 }
		 if (fd_copy(1, 0) < 0) {
		    log_log(LOG_DEBUG, NULL, _("File descriptors copy failure <%s>"),
			    newstrerror(errno));
		 }
		 close(fo_eventfd);
		   {
		      ServiceContext *servicecontext = &fo_servicecontext;
		      
#ifdef HAVE_CHROOT
		      if (servicecontext->chroot != NULL) {
			 if (chroot(servicecontext->chroot) != 0 ||
			     chdir(PARSER_DIRECTORYSEP_STR) != 0) {
			    log_log(LOG_ERR, NULL,
				    _("Chroot() to [%s] failed <%s>"),
				    servicecontext->chroot, newstrerror(errno));
			    goto seize;
			 }
		      }
#endif
		      /* set limits bevor abandoning root rights */
		      if (jnetd_setlimits(servicecontext) != 0) {
			 log_log(LOG_ERR, NULL,
				 _("Can't set limits for [%s] : server not launched."),
				 tools_servicenametxt(servicecontext));
			 goto seize;
		      }
		      if (tools_changeid(servicecontext->user,
					 servicecontext->group) != 0) {
			 log_log(LOG_ERR, NULL,
				 _("Can't change user ID for service [%s]"),
				 tools_servicenametxt(servicecontext));
			 goto seize;
		      }
		      signal_catch(SIGCHLD, SIG_DFL);
		      signal_unblock(SIGCHLD); 	       	       
		      signal_catch(SIGTERM, SIG_DFL);
		      signal_catch(SIGPIPE, SIG_DFL);
		      signal_catch(SIGALRM, SIG_DFL);		      
		      execv(servicecontext->run[0], servicecontext->run);
		   }
		 seize:
		 if (socket_isstream(event->socket) == 0) {
		    socket_flush(0);
		 }
		 puts(_("Unable to execute"));
		 log_log(LOG_ERR, NULL, _("Unable to launch [%s]"),
			 ((ServiceContext *) event->socket->geek)->run[0]);
		 exit(EXIT_FAILURE);
	       default :
		 if (mem_resize(m_childtab, (pid + 1) * sizeof (ChildTab)) != 0) {
		    LOG_ERRMARK(NULL, _("Children table expansion"));
		    break;
		 }
		 ((ChildTab *) m_childtab->space)[pid].fd = event->fd;
		 ((ChildTab *) m_childtab->space)[pid].socket = event->socket;
		 ((ServiceContext *) event->socket->geek)
		   ->servicenetworkstatus.dgramfd = event->fd;
		 log_log(LOG_INFO, NULL, _("[%s] launched as PID [%u] for [%s]"),
			 tools_servicenametxt((ServiceContext *) event->socket->geek),
			 (unsigned int) pid,
			 socket_getpeernameoraddr(event->socket, event->fd));
		 ((ServiceContext *) event->socket->geek)->service_cur_nblimit++;
	      }
	   }
 	 socket_hangup(event->socket, event->fd); 
	 break;
       case SOCKEVENT_HANGUP :
	 if (event->socket->cookie2 == 0) {
	    if (event->socket->cookie != 0) {
	       socket_hangup(event->socket,
			     ((int *) event->socket->geek2)[event->fd]);
	    }
	 } else {
	    socket_hangup((Socket *) event->socket->geek2,
			  (int) event->socket->cookie2);
	 }
	 if (((ServiceContext *) event->socket->geek)->service_cur_nblimit > 0) {
	    ((ServiceContext *) event->socket->geek)->service_cur_nblimit--;
	 }
	 break;
       case SOCKEVENT_ANSWER :
#ifdef SYSTEMATIC_IPOPTIONS_CHECK
	 {
	    const char *ipoptmsg;
	    if ((ipoptmsg = tools_seeipoptions(event)) != NULL) {
	       if (((ServiceContext *) event->socket->geek)->service_cur_nblimit > 0) {
		  ((ServiceContext *) event->socket->geek)->service_cur_nblimit--;
	       }	 	    
	       socket_hangup(event->socket, event->fd);	       
	       log_log(LOG_WARNING, NULL, ipoptmsg);
	       goto nextevent;
	    }
	 }
#endif
	 if (event->socket->cookie != 0) {
	    write((int) event->socket->cookie2, event->line, event->linelen);
	 } else {
	    jnetd_builtin(event);
	 }
      	 break;
       case SOCKEVENT_INTERRUPTED :
	 break;
       default :
	 if (((ServiceContext *) event->socket->geek)->service_cur_nblimit > 0) {
	    ((ServiceContext *) event->socket->geek)->service_cur_nblimit--;
	 }	 
	 socket_hangup(event->socket, event->fd);
      }      
   }
   socket_closeall(&sockets);
   log_closesyslog();
   mem_free(m_childtab);
   
   return EXIT_SUCCESS;
}
