/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>

#include "utmp.hpp"
#include "logfile.hpp"

CUtmpFile::CUtmpFile(CConfigurationFile *pConfig, char **pszError)
{
	*pszError = NULL;	
	
	pConf = pConfig;
	
	iRecords = 0;
	pRecords = NULL;
	
	char szLine[UT_LINESIZE + 1];
	char szName[UT_NAMESIZE + 1];

	struct utmp *pEntry;
	setutent();
	while (NULL != (pEntry = getutent())) {
// It is very important that a string in the utmp structure is NOT null
// terminated if the length of the string is equal to the given space for
// it in the utmp structure.
		strncpy(szLine, pEntry->ut_line, UT_LINESIZE);
		strncpy(szName, pEntry->ut_name, UT_NAMESIZE);
		szLine[UT_LINESIZE] = '\0';
		szName[UT_NAMESIZE] = '\0';
// If this a stale utmp entry		
		if (0 != kill(pEntry->ut_pid, 0))
			continue;
// Count only user records		
		if (USER_PROCESS != pEntry->ut_type)
			continue;
// Count only lines in configuration file
		if (NULL == pConfig->GetString("Devices", szLine, (char *)NULL))
			continue;
// If user is not allowed to be checked ... continue
		char *pszMessage;
		if (false == IsUserAllowed(szName, &pszMessage))
			continue;
		if (NULL != pszMessage) {
			*pszError = pszMessage;
			return;
		}
// If there is a newer utmp entry on the same terminal than this then continue
		bool fgNewer = false;
		
		for (int i = 0; i < iRecords; i++)
			if ((pEntry->ut_time < pRecords[i].ut_time) &&
				 (0 == strncmp(pEntry->ut_line, pRecords[i].ut_line, UT_LINESIZE)))
				fgNewer = true;
// At last ... add entry
		if (false == fgNewer) {
			struct utmp *pNewRecords = (struct utmp *)realloc(pRecords, (iRecords + 1) * sizeof(struct utmp));
			if (NULL == pNewRecords) {
				*pszError = "Error allocating memory...";
				return;
			}
			pRecords = pNewRecords;
			memcpy(&pRecords[iRecords], pEntry, sizeof(struct utmp));
			iRecords += 1;
		}
	}
	endutent();
}

CUtmpFile::~CUtmpFile()
{
	if (0 != iRecords)
		free(pRecords);
}

void CUtmpFile::Dump()
{
	char szLine[UT_LINESIZE + 1];
	char szName[UT_NAMESIZE + 1];	
	
	printf("Total records: %d\n", iRecords);
	for (int i = 0; i < iRecords; i++) {
		strncpy(szLine, pRecords[i].ut_line, UT_LINESIZE);
		strncpy(szName, pRecords[i].ut_name, UT_NAMESIZE);
		szLine[UT_LINESIZE] = '\0';
		szName[UT_NAMESIZE] = '\0';
		
		printf("%8s%12s\n", szLine, szName);
	}
}

bool CUtmpFile::IsUserAllowed(char *pszLogin, char **pszError)
{
	static char szBuf[128];
	static char szResult[128];
	char *pszAllowed;
	char *pszAllowedUsers;
	struct passwd *pwLogin;
	struct group *grLogin;
	
	*pszError = NULL;

	pszAllowedUsers = pConf->GetString("Permissions", "users", "@users");
	
	if (NULL == (pwLogin = getpwnam(pszLogin))) {
		snprintf(szResult, sizeof(szResult), "Invalid passwd entry for user %s...", pszLogin);
		*pszError = szResult;
		return false;
	}
	
	int i, j;
	for (i = j = 0; (pszAllowedUsers[i] != '\0') && (j < (int)sizeof(szBuf)); i++)
		if (NULL == strchr(" \t\x0a\x0d", pszAllowedUsers[i]))
			szBuf[j++] = pszAllowedUsers[i];
	szBuf[j] = '\0';
	
	if (NULL == (grLogin = getgrgid(pwLogin->pw_gid))) {
		snprintf(szResult, sizeof(szResult), "Invalid group entry for user %s...", pszLogin);
		*pszError = szResult;
		return false;
	}
	
	if (NULL != (pszAllowed = strtok(szBuf, ",;")))
		do {
			if ('@' == pszAllowed[0]) {
				if (0 == strcmp(pszAllowed + 1, grLogin->gr_name))
					return true;
			} else {
				if (0 == strcmp(pszLogin, pszAllowed))
					return true;
			}
		} while (NULL != (pszAllowed = strtok(NULL, ",;")));
	
	return false;
}

bool CUtmpFile::ExistsDevice(char *pszDevice)
{
	for (int i = 0; i < iRecords; i++)
		if (0 == strncmp(pRecords[i].ut_line, pszDevice, UT_LINESIZE))
			return true;
	
	return false;
}

void CUtmpFile::KillUser(char *pszLogin, char *pszDevice)
{
	for (int i = 0; i < iRecords; i++)
		if (0 == strncmp(pRecords[i].ut_line, pszDevice, UT_LINESIZE) &&
			 0 == strncmp(pRecords[i].ut_name, pszLogin, UT_NAMESIZE)) {
			lprintf("Killing process %d with signal SIGHUP...\n", pRecords[i].ut_pid);
			kill(pRecords[i].ut_pid, SIGHUP);
		}
}

void CUtmpFile::EnumUtmp(void (*pEnumerator)(char *, char *, void *), void *pArgs)
{
	char szLine[UT_LINESIZE + 1];
	char szName[UT_NAMESIZE + 1];
	for (int i = 0; i < iRecords; i++) {
		strncpy(szLine, pRecords[i].ut_line, UT_LINESIZE);
		strncpy(szName, pRecords[i].ut_name, UT_NAMESIZE);
		szLine[UT_LINESIZE] = '\0';
		szName[UT_NAMESIZE] = '\0';
		(*pEnumerator)(szName, szLine, pArgs);
	}
}

bool CUtmpFile::GetFirstRecord(SUtmpEntry *pEntry)
{
	if (0 == iRecords)
		return false;
	iCurrent = 0;
	
	strcpy(pEntry->szLogin, pRecords[0].ut_name);
	strcpy(pEntry->szDevice, pRecords[0].ut_line);
	
	char *pszHost = pConf->GetString("Users", pEntry->szLogin, NULL);
	
	if (NULL == pszHost)
		pszHost = pConf->GetString("Devices", pEntry->szDevice, "HOSTERR");
	
	strncpy(pEntry->szHost, pszHost, CE_HOSTSIZE);
	
	return true;
}

bool CUtmpFile::GetNextRecord(SUtmpEntry *pEntry)
{
	iCurrent += 1;
	if (iCurrent == iRecords)
		return false;

	strcpy(pEntry->szLogin, pRecords[iCurrent].ut_name);
	strcpy(pEntry->szDevice, pRecords[iCurrent].ut_line);
	
	char *pszHost = pConf->GetString("Users", pEntry->szLogin, NULL);
	
	if (NULL == pszHost)
		pszHost = pConf->GetString("Devices", pEntry->szDevice, "HOSTERR");
	
	strncpy(pEntry->szHost, pszHost, CE_HOSTSIZE);
	
	return true;
}