/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>

#include "tokenized_string.hpp"
#include "logfile.hpp"
#include "conf.hpp"
#include "trim.hpp"
#include "shell.hpp"
#include "poll.hpp"
#include "logfile.hpp"

extern CConfigurationFile *pConfigFile;
extern uid_t iOldUser;

char *strchgn(char *pszStorage, char *pszWhat, char *pszNew, unsigned nMax)
{
	char *pszTemp = strstr(pszStorage, pszWhat);
	if ((NULL != pszTemp) && (strlen(pszStorage) - strlen(pszWhat) + strlen(pszNew) < nMax)) {
		memmove(pszTemp + strlen(pszNew), pszTemp + strlen(pszWhat), strlen(pszTemp + strlen(pszWhat)) + 1);
		memcpy(pszTemp, pszNew, strlen(pszNew));
	}
	
	return pszStorage;
}

void shell()
{
//	char *pszLogin = getlogin();				// This returns NULL in some glibc
	char *pszLogin = getenv("LOGNAME");
	char *pszDevice = ttyname(0);

	if (NULL == pszLogin) {
		lprintf("Error getting login from the environment...\n");
		return;
	}
	if (NULL == pszDevice) {
		lprintf("Error getting tty name...\n");
		return;
	}

	char *pszTemp;
	if (NULL != (pszTemp = strrchr(pszDevice, '/')))
		pszDevice = pszTemp + 1;

	char *pszHost = pConfigFile->GetString("Users", pszLogin, NULL);
	if (NULL == pszHost)
		pszHost = pConfigFile->GetString("Devices", pszDevice, NULL);
	if (NULL == pszHost) {
		lprintf("Error getting host IP...\n");
		return;
	}
	
	char *pszAllowed = pConfigFile->GetString("Permissions", pszDevice, "");
	char *pszNext = pConfigFile->GetString("Login", pszDevice, "/bin/false");
	
	if (false == AllowUser(pszLogin, pszAllowed)) {
		printf("Access denied.\n");
		return;
	}
	
	if (pConfigFile->GetString("Devices", pszDevice, NULL) != NULL) {
		int iResult = UserUp(pszLogin, pszDevice, pszHost, false);
		if (3 == iResult) {
			printf("Invalid account name specified...\n");
			return;
		} else if (4 == iResult) {
			printf("Account is expired...\n");
			return;
		} else if (0 != iResult) {
			printf("Extended error %d. Contact sysadmin...\n", iResult);
			return;
		}
		printf("Remote IP: %s\n", pszHost);
	}
	
	char szBuf[256];
	
	strncpy(szBuf, pszNext, sizeof(szBuf));
	strchgn(szBuf, "${IP}", pszHost, sizeof(szBuf));
	strchgn(szBuf, "${DEVICE}", pszDevice, sizeof(szBuf));
	strchgn(szBuf, "${LOGIN}", pszLogin, sizeof(szBuf));
	
	char **ppszArgs = new (char *)[128];
	if (NULL == ppszArgs) {
		lprintf("Memory allocation error...\n");
		return;
	}
	int iArgs = 0;
	
	char *pszArg = strtok(szBuf, " \t");
	do {
		if (iArgs >= 128)
			ppszArgs = (char **)realloc(ppszArgs, sizeof(char *) * (iArgs + 1));
		ppszArgs[iArgs++] = pszArg;
	} while ((pszArg = strtok(NULL, " \t")));
	ppszArgs[iArgs] = pszArg;
	
	setuid(iOldUser);
	
	execv(ppszArgs[0], ppszArgs);
}

bool AllowUser(char *pszLogin, char *pszList)
{
	CTokenizedString cAllowedUsers(pszList, ";,");
	
	struct passwd *pstrPasswd = getpwnam(pszLogin);
	if (NULL == pstrPasswd) {
		lprintf("Invalid login: %s...\n", pszLogin);
		return false;
	}
	struct group *pstrGroup = getgrgid(pstrPasswd->pw_gid);
	if (NULL == pstrGroup) {
		lprintf("Invalid group for User: %s...\n", pszLogin);
		return false;
	}
	char *pszGroup = pstrGroup->gr_name;
	char *pszAllowed;
	if ((pszAllowed = cAllowedUsers.GetFirstString()))
		do {
			pszAllowed = TrimBoth(pszAllowed, " \t\x0a\x0d");
			if ('@' == pszAllowed[0]) {
			  	if (0 == strcmp(pszGroup, pszAllowed + 1))
					return true;
			} else {
			   if (0 == strcmp(pszLogin, pszAllowed))
					return true;
			}
		} while ((pszAllowed = cAllowedUsers.GetNextString()));
	
	lprintf("Login denied for user %s: %s...\n");
	return false;
}