/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "poll.hpp"
#include "conf.hpp"
#include "logfile.hpp"
#include "utmp.hpp"
#include "logged.hpp"
#include "sockio.hpp"
#include "server.hpp"
#include "string_array.hpp"

extern CConfigurationFile *pConfigFile;

void ScanUsers()
{
	char *pszMessage;
	
	CUtmpFile cUtmp(pConfigFile, &pszMessage);
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		return;			// Error reading utmp file...
	}
	
	CStringArray cUsersUp;
	CStringArray cUsersDown;
	int iUsersUp = 0;
	int iUsersDown = 0;

	CLoggedDatabase *pLoggedDatabase = new CLoggedDatabase(pConfigFile, &pszMessage, true);
	if (NULL == pLoggedDatabase) {
		lprintf("Memory allocation error...\n");
		return;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		delete pLoggedDatabase;
		return;			// Error opeinig current database...
	}
	SUtmpEntry strUtmp;
	if (true == cUtmp.GetFirstRecord(&strUtmp))
		do {
			if (false == pLoggedDatabase->ExistsRecord(strUtmp.szHost)) {
				cUsersUp.AddString(strUtmp.szLogin);
				cUsersUp.AddString(strUtmp.szDevice);
				cUsersUp.AddString(strUtmp.szHost);
				iUsersUp += 1;
			}
//				UserUp(strUtmp.szLogin, strUtmp.szDevice, strUtmp.szHost, true);
		} while (true == cUtmp.GetNextRecord(&strUtmp));
	
	SLoggedEntry strLogged;
	if (true == pLoggedDatabase->GetFirstRecord(&strLogged))
		do {
			if (false == cUtmp.ExistsDevice(strLogged.szDevice)) {
				cUsersDown.AddString(strLogged.szLogin);
				cUsersDown.AddString(strLogged.szDevice);
				cUsersDown.AddString(strLogged.szHost);
				iUsersDown += 1;
			}
//		   	UserDown(strLogged.szLogin, strLogged.szDevice, strLogged.szHost);
		} while (true == pLoggedDatabase->GetNextRecord(&strLogged));
	delete pLoggedDatabase;
// I need a better error handling for memory allocation in the above code

	for (int i = 0; i < iUsersUp; i++)
		UserUp(cUsersUp.GetString(i * 3),
				 cUsersUp.GetString(i * 3 + 1),
				 cUsersUp.GetString(i * 3 + 2),
				 true);
	for (int i = 0; i < iUsersDown; i++)
   	UserDown(cUsersDown.GetString(i * 3),
					cUsersDown.GetString(i * 3 + 1),
					cUsersDown.GetString(i * 3 + 2));
}

int UserUp(char *pszLogin, char *pszDevice, char *pszHost, bool fgKill)
{
	char *pszServer = pConfigFile->GetString("Alligator", "host", "localhost");
	int iServerPort = pConfigFile->GetInteger("Alligator", "port", 1098);
	
	int newsockfd = 0;
	
	if ((newsockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		lprintf("socket error: %s\n", strerror(errno));
		return 1;
	}

	struct hostent *pHost = gethostbyname(pszServer);
	if (NULL == pHost) {
		lprintf("resolving error: %s\n", pszServer);
		close(newsockfd);
		return 1;
	}
	
	struct sockaddr_in client;
	
	memset(&client, 0, sizeof(client));
	client.sin_family = AF_INET;
	client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
	client.sin_port = htons(iServerPort);
	
	if (connect(newsockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
		lprintf("connect error: %s\n", strerror(errno));
		close(newsockfd);
		return 1;
	}
	
	unsigned char c;
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 1;
	}
	if (WELCOME != c) {
		lprintf("Invalid response from server...\n");
		close(newsockfd);
		return 1;
	}
	write_string(newsockfd, pszLogin);
	write_string(newsockfd, "@login@");
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 1;
	}
	if (INVALID_LOGIN == c)  {
		lprintf("Invalid login to server. Must be trusted..\n");
		close(newsockfd);
		return 1;
	}
	if (GO_AHEAD != c) {
		lprintf("Invalid response from server...\n");
		close(newsockfd);
		return 1;
	}
	write_byte(newsockfd, ACTION_TRUSTEDLOGIN);
	write_string(newsockfd, pszDevice);
	write_string(newsockfd, pszHost);
	write_string(newsockfd, GetHostNumber());
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 1;
	}
	close(newsockfd);
	if (c != OK) {
		char *pszMessage;
		CUtmpFile cUtmp(pConfigFile, &pszMessage);
		if (NULL != pszMessage) {
			lprintf("%s\n", pszMessage);
			return 2;			// Error reading utmp file...
		}
		if (true == fgKill)
			cUtmp.KillUser(pszLogin, pszDevice);
		
		if (c == ERROR) {
			lprintf("Error logging in alligator: %s, %s, %s\n", pszLogin, pszDevice, pszHost);
			return 2;
		}
		if (c == INVALID_USER) {
			lprintf("Invalid user logging in alligator: %s, %s, %s\n", pszLogin, pszDevice, pszHost);
			return 3;
		}
		if (c == NO_RIGHTS) {
			lprintf("Account expired: %s, %s, %s\n", pszLogin, pszDevice, pszHost);
			return 4;
		}
		lprintf("Invalid response from alligator.\n");
		return 2;
	}
	
	char *pszMessage;
	CLoggedDatabase cLoggedDatabase(pConfigFile, &pszMessage);
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		return 5;			// Error opeinig current database...
	}
	
	if (false == cLoggedDatabase.AddRecord(pszLogin, pszDevice, pszHost)) {
		lprintf("Error while adding a record to the database...\n");
		return 5;
	}

	lprintf("Log in: %s, %s, %s\n", pszLogin, pszDevice, pszHost);
	
	return 0;
}

int UserDown(char *pszLogin, char *pszDevice, char *pszHost)
{
	char *pszServer = pConfigFile->GetString("Alligator", "host", "localhost");
	int iServerPort = pConfigFile->GetInteger("Alligator", "port", 1098);
	
	int newsockfd = 0;
	
	if ((newsockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		lprintf("socket error: %s\n", strerror(errno));
		return 1;
	}

	struct hostent *pHost = gethostbyname(pszServer);
	if (NULL == pHost) {
		lprintf("resolving error: %s\n", pszServer);
		close(newsockfd);
		return 1;
	}
	
	struct sockaddr_in client;
	
	memset(&client, 0, sizeof(client));
	client.sin_family = AF_INET;
	client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
	client.sin_port = htons(iServerPort);
	
	if (connect(newsockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
		lprintf("connect error: %s\n", strerror(errno));
		close(newsockfd);
		return 1;
	}
	
	unsigned char c;
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 1;
	}
	if (WELCOME != c) {
		lprintf("Invalid response from server...\n");
		close(newsockfd);
		return 1;
	}
	write_string(newsockfd, pszLogin);
	write_string(newsockfd, "@logout@");
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 1;
	}
	if (INVALID_LOGIN == c)  {
		lprintf("Invalid login to server. Must be trusted..\n");
		close(newsockfd);
		return 2;
	}
	if (GO_AHEAD != c) {
		lprintf("Invalid response from server...\n");
		close(newsockfd);
		return 2;
	}
	write_byte(newsockfd, ACTION_TRUSTEDLOGOUT);
	write_string(newsockfd, pszDevice);
	write_string(newsockfd, pszHost);
	write_string(newsockfd, GetHostNumber());
	if (true != read_byte(newsockfd, &c)) {
		lprintf("Error in alligator communication...\n");
		close(newsockfd);
		return 2;
	}
	close(newsockfd);
	
	if (OK != c) {
		lprintf("Error in alligator communication...\n");
		return 2;
	}
	
	char *pszMessage;
	CLoggedDatabase cLoggedDatabase(pConfigFile, &pszMessage);
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		return 3;			// Error opeinig current database...
	}

	cLoggedDatabase.DeleteRecord(pszHost);

	lprintf("Log out: %s, %s, %s\n", pszLogin, pszDevice, pszHost);
	
	return 0;
}

char *GetHostNumber()
{
	static char szResult[256];
	char szName[256];
	gethostname(szName, sizeof(szName));
	
	struct hostent *p = gethostbyname(szName);

	snprintf(szResult,
				sizeof(szResult),
				"%d.%d.%d.%d\n", 
				(unsigned char)p->h_addr[0],
				(unsigned char)p->h_addr[1],
				(unsigned char)p->h_addr[2],
				(unsigned char)p->h_addr[3]);
	return szResult;
}
