/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "pidfile.hpp"
#include "errors.h"

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <paths.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/types.h>

#ifndef _PATH_VARRUN
	#define _PATH_VARRUN "/var/run/"
#endif // _PATH_VARRUN

CPidFile::CPidFile(char *pszPidFile)
{
	strcpy(szPidFile, _PATH_VARRUN);
	strcat(szPidFile, pszPidFile);
	
	iErrorCode = NO_ERROR;
}

CPidFile::~CPidFile()
{
	unlink(szPidFile);
}

bool CPidFile::WritePid()
{
	int iFile = open(szPidFile, O_RDWR | O_CREAT, 0644);
	if (-1 == iFile) {
		iErrorCode = OPEN_ERROR;
		return false;
	}
	
	FILE *fpFile = fdopen(iFile, "r+");
	if (NULL == fpFile) {
		iErrorCode = OPEN_ERROR;		
		close(iFile);
		return false;
	}
	
	if (flock(iFile, LOCK_EX | LOCK_NB) == -1) {
		iErrorCode = LOCK_ERROR;		
		return false;
	}
	if (0 == fprintf(fpFile, "%d\n", getpid())) {
		iErrorCode = WRITE_ERROR;		
		fclose(fpFile);
		return false;
	}
	fflush(fpFile);
	if (-1 == flock(iFile, LOCK_UN)) {
		iErrorCode = UNLOCK_ERROR;		
		fclose(fpFile);
		return false;
	}
	fclose(fpFile);
	
	return true;
}

bool CPidFile::CheckPid()
{
	FILE *fpFile = fopen(szPidFile, "r");
	if (NULL == fpFile)
		return true;
	int iPid = 0;
	fscanf(fpFile, "%d", &iPid);
	fclose(fpFile);
	
	if (getpid() == iPid)
		return true;
	
	if (kill(iPid, 0) && (ESRCH == errno))
		return true;
	
	return false;
}
