/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <time.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>

#include "config.h"
#include "conf.hpp"
#include "pidfile.hpp"
#include "server.hpp"
#include "resolv.hpp"
#include "logfile.hpp"
#include "poll.hpp"
#include "shell.hpp"
#include "logged.hpp"

#ifdef _SOCKLEN_T_UNDEFINED
	typedef int socklen_t;
#endif

#define PID_FILE 					"mugger.pid"
#define LOG_FILE					"mugger.log"

#define DEFAULT_CONFFILE		"/etc/mugger.conf"
#define DEFAULT_SERVERPORT		1097

// Function prototypes
void SignalProc(int);
bool AllowHost(struct in_addr);

CConfigurationFile *pConfigFile = NULL;
CPidFile *pPidFile = NULL;
struct sockaddr_in serv;
struct sockaddr_in child;
char szConfigFile[256] = DEFAULT_CONFFILE;
int iServerPort;
int sockfd;
int listenfd;
int newlistenfd;

void usage()
{
	printf("Usage: mugger [-c <config file>] [-l]\n"
	       "Options:\n"
			 "  -c <file>   Use alternate config file instead the default\n"
	       "              '/etc/mugger.conf' system wide config file.\n"
			 "  -l          Use mugger as a login shell for dial-up users.\n"
	       "Written by: Alexander Feldman <alex@varna.net>\n\n");
}

uid_t iOldUser;

int main(int argc, char **argv)
{
	int pidstatus;
	int child_process;
	bool fgShell = false;

	int w;
	while (-1 != (w = getopt(argc, argv, "c:l")))
		switch (w) {
			case 'c':
				strncpy(szConfigFile, optarg, sizeof(szConfigFile));
			break;
			case 'l':
				fgShell = true;
			break;
			default:
				usage();
				return 0;
			break;
		}
	
	if (0 != argc - optind) {
		usage();
		return 0;
	}
	
	tzset();

	char *pszMessage;
	pConfigFile = new CConfigurationFile(szConfigFile, &pszMessage);
	if (NULL == pConfigFile) {
		fprintf(stderr, "Error allocating memory...\n");
		return 1;
	}
	if (NULL != pszMessage) {
		fprintf(stderr, "%s\n", pszMessage);
		delete pConfigFile;
		return 1;
	}
	
	iOldUser = getuid();
	if (0 != setuid(0)) {
		fprintf(stderr, "Mugger needs root privileges: %s\n", strerror(errno));
		return 1;
	}
	
	CLoggedDatabase *pLoggedDatabase = new CLoggedDatabase(pConfigFile, &pszMessage, false);
	if (NULL == pLoggedDatabase) {
		fprintf(stderr, "Error allocating memory...\n");
		delete pConfigFile;
		return 1;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		delete pConfigFile;
		delete pLoggedDatabase;
		return 1;		// Error opeinig current database...
	}
	delete pLoggedDatabase;

	char *pszLogFile = new char[strlen(argv[0]) + 32];
	if (NULL == pszLogFile) {
		fprintf(stderr, "Error allocating memory...\n");
		delete pPidFile;
		delete pConfigFile;
		return 1;
	}
	strcpy(pszLogFile, argv[0]);
	strcat(pszLogFile, ".log");

	if (true == fgShell) {
		logopen(pszLogFile);
		delete pszLogFile;
		shell();
		return 0;
	}
	
	pPidFile = new CPidFile(PID_FILE);
	if (NULL == pPidFile) {
		fprintf(stderr, "Error allocating memory...\n");
		delete pszLogFile;
		return 1;
	}
	if (false == pPidFile->CheckPid()) {
		fprintf(stderr, "Daemon is already started...\n");
		delete pPidFile;
		delete pConfigFile;
		delete pszLogFile;
		return 1;
	}

#ifndef DEBUG
	int i = getdtablesize();
	for (int j = 0; j < i; j++)
		close(j);
	setsid();
#endif // DEBUG
	
	if (false == logopen(pszLogFile)) {
		delete pszLogFile;
		delete pPidFile;
		delete pConfigFile;
		return 2;
	}
	delete pszLogFile;

#ifndef DEBUG	
	int iPid;
	
	if ((iPid = fork()) > 0)
		return 0;				// Parrent process...
   
	if (iPid < 0) {
		lprintf("Initial fork error...\n");
		delete pPidFile;
		delete pConfigFile;
		return 1;
	}
#endif // DEBUG
	
	pPidFile->WritePid();

	signal(SIGTERM, SignalProc);
	signal(SIGHUP, SignalProc);
	signal(SIGINT, SignalProc);
	signal(SIGQUIT, SignalProc);
	signal(SIGUSR1, SignalProc);
	signal(SIGALRM, SignalProc);
	signal(SIGCHLD, SignalProc);
	
	lprintf("Starting...\n");
	
	int iUpdateTime = pConfigFile->GetInteger("Global", "poll_time", 60);
	if (iUpdateTime < 5 || iUpdateTime > 3600) {
		lprintf("Invalid poll time %d. Setting default value of 60 seconds...\n", iUpdateTime);
		iUpdateTime = 60;		
	}
	alarm(iUpdateTime);
	
	if ((listenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		lprintf("socket error: %s\n", strerror(errno));
		return 1;
	}
	
	iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
	
	memset(&serv, 0, sizeof(serv));
	serv.sin_family = AF_INET;
	serv.sin_addr.s_addr = htonl(INADDR_ANY);
	serv.sin_port = htons(iServerPort);
	
	if (bind(listenfd, (struct sockaddr *)&serv, sizeof(serv)) < 0) {
		lprintf("bind error: %s\n", strerror(errno));
		return 1;
	}
	
	if (listen(listenfd, SOMAXCONN) < 0) {
		lprintf("listen error: %s\n", strerror(errno));
		return 1;
	}

	ScanUsers();
	
	child_process = 0;
	while (0 == child_process) {			// Loop begins here
		memset(&child, 0, sizeof(child));
		socklen_t childlen = (socklen_t)sizeof(child);
		if ((sockfd = accept(listenfd, (struct sockaddr *)&child, &childlen)) < 0) {
			if (EINTR != errno)
				lprintf("accept error (%s): %s\n", naddr2str(&child), strerror(errno));
			continue;
		}
		lprintf("opened client connection from %s\n", naddr2str(&child));
		if (false == AllowHost(child.sin_addr)) {
			lprintf("unauthorized client connection from %s\n", naddr2str(&child));
			close(sockfd);
			continue;
		}
		pid_t pid;
		if ((pid = fork()) == 0) {			// This is the child
			close(listenfd);
			server(sockfd);
			close(sockfd);
			child_process = 1;				// Exit from main loop for the child process
		} else {									// We are parent, go on listening and error checking
			if (pid < 0)
				lprintf("fork error: %s\n", strerror(errno));
			while(waitpid(-1, &pidstatus, WNOHANG) > 0);
			close(sockfd);
			child_process = 0;
		}
	}

	return 0;
}

void SignalProc(int iSignal)
{
	CConfigurationFile *pNewConfigFile = NULL;
	char *pszResult;
	int iUpdateTime;
	int iPidStatus;
	
	switch (iSignal) {
		case SIGALRM:
			iUpdateTime = pConfigFile->GetInteger("Global", "poll_time", 60);
			if (iUpdateTime < 5 || iUpdateTime > 3600) {
				lprintf("Invalid poll time %d. Setting default value of 60 seconds...\n", iUpdateTime);
				iUpdateTime = 60;		
			}
			alarm(iUpdateTime);
			ScanUsers();
		break;
		case SIGUSR1:
			ScanUsers();
		break;
		case SIGHUP:
			lprintf("Reinitializing...\n");
			pNewConfigFile = new CConfigurationFile(szConfigFile, &pszResult);
			if (NULL == pNewConfigFile) {
				lprintf("Error allocating memory...\n");
				return;
			}
			if (NULL != pszResult) {
				lprintf("Conf error: %s\n", pszResult);
				delete pNewConfigFile;
				return;
			}
			pConfigFile = pNewConfigFile;
		
			if (iServerPort != pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT)) {
				iServerPort = pConfigFile->GetInteger("Global", "port", DEFAULT_SERVERPORT);
				if ((newlistenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
					lprintf("Socket error: %s\n", strerror(errno));
					return;
				}
				memset(&serv, 0, sizeof(serv));
				serv.sin_family = AF_INET;
				serv.sin_addr.s_addr = htonl(INADDR_ANY);
				serv.sin_port = htons(iServerPort);
	
				if (bind(newlistenfd, (struct sockaddr *)&serv, sizeof(serv)) < 0) {
					lprintf("Bind error: %s\n", strerror(errno));
					return;
				}
	
				if (listen(newlistenfd, SOMAXCONN) < 0) {
					lprintf("Listen error: %s\n", strerror(errno));
					return;
				}
				close(listenfd);
				listenfd = newlistenfd;
			}
		break;
		case SIGINT:
		case SIGQUIT:
		case SIGTERM:
			lprintf("Exiting on signal %d...\n", iSignal);
			logclose();
			delete pConfigFile;
			delete pPidFile;
			exit(1);
		break;
		case SIGCHLD:
			while(waitpid(-1, &iPidStatus, WNOHANG) > 0);
		break;
	}
	signal(iSignal, SignalProc);
}

bool AllowHost(struct in_addr lSource)
{
	register int i, j;
	static char szBuf[256];
// We will compare the source address, the target address and target netmask
// all in host byte order. There is no matter what ordering we will use
// but is logically more correct to perform it in native ordering.	
	unsigned long int ulSourceHost = ntohl(lSource.s_addr);
	unsigned long int ulTargetHost;
	unsigned long int ulTargetMask;
	char *pszHost;
	bool fgInvert;
	
	char *pszAllowedHosts = pConfigFile->GetString("Permissions", "hosts_allow", "localhost");
	for (i = j = 0; pszAllowedHosts[i] != '\0'; i++)
	   if (NULL == strchr(" \t\x0a\x0d", pszAllowedHosts[i]))
		   szBuf[j++] = pszAllowedHosts[i];
	szBuf[j] = '\0';

	if (NULL == (pszHost = strtok(szBuf, ",;")))
		return false;
	do {
		if (false == naddr2h(pszHost, &ulTargetHost, &ulTargetMask, &fgInvert)) {
			lprintf("configuration file error, skipping host\n");
			continue;
		}
		if (fgInvert && ((ulSourceHost & ulTargetMask) == ulTargetHost))
			return false;
		if ((ulSourceHost & ulTargetMask) == ulTargetHost)
			return true;
	} while (NULL != (pszHost = strtok(NULL, ",;")));
	
	return false;
}
