/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "config.h"
#include "logged.hpp"

CLoggedDatabase::CLoggedDatabase(CConfigurationFile *pConfig, char **pszError, bool fgReadOnlyMode)
{
	fgReadOnly = fgReadOnlyMode;
	
	strLoggedKey.dptr = (char *)NULL;
	
	*pszError = NULL;
	
	strncpy(szLoggedDatabase, 
			  pConfig->GetString("Global", "data_dir", "/tmp"), 
			  sizeof(szLoggedDatabase));
	strcat(szLoggedDatabase, "/logged.db");
	if (true == fgReadOnly)
		dbfLogged = gdbm_open(szLoggedDatabase, 0, GDBM_READER, 0, NULL);
	else
		dbfLogged = gdbm_open(szLoggedDatabase, 0, GDBM_WRCREAT, 0644, NULL);
	if (NULL == dbfLogged) {
		*pszError = "Error opening current database...";
		return;
	}
}

CLoggedDatabase::~CLoggedDatabase()
{
	if (NULL != strLoggedKey.dptr)
		free(strLoggedKey.dptr);
	
	if (NULL != dbfLogged)
		gdbm_close(dbfLogged);
}

bool CLoggedDatabase::AddRecord(char *pszLogin, char *pszDevice, char *pszHost)
{
	SLoggedEntry sLogged;
	
	strcpy(sLogged.szLogin, pszLogin);
	strcpy(sLogged.szDevice, pszDevice);
	strcpy(sLogged.szHost, pszHost);
	
	datum strLoggedKey = { pszHost, strlen(pszHost) };
	datum strLoggedData = { (char *)&sLogged, sizeof(SLoggedEntry) };
	
	if (0 != gdbm_store(dbfLogged, strLoggedKey, strLoggedData, GDBM_REPLACE))
   	return false;
	
	return true;	
}

void CLoggedDatabase::DeleteRecord(char *pszHost)
{
	datum strLoggedKey = { pszHost, strlen(pszHost) };
	gdbm_delete(dbfLogged, strLoggedKey);
}

bool CLoggedDatabase::ExistsRecord(char *pszHost)
{
	datum strLoggedKey = { pszHost, strlen(pszHost) };
	return gdbm_exists(dbfLogged, strLoggedKey);
}

void CLoggedDatabase::EnumCurrent(void (*pEnumerator)(char *, char *, char *))
{
	datum strKey;
	datum strNextKey;
	char szLogin[UT_NAMESIZE + 1];
	char szDevice[UT_LINESIZE + 1];	
	char szHost[CE_HOSTSIZE];
	
	strKey = gdbm_firstkey(dbfLogged);
	while (strKey.dptr) {
		strNextKey = gdbm_nextkey(dbfLogged, strKey);

		datum strData = gdbm_fetch(dbfLogged, strKey);
	
		if (NULL == strData.dptr) {
			free(strKey.dptr);			
			continue;
		}
	
		strcpy(szDevice, ((SLoggedEntry *)strData.dptr)->szDevice);
		strcpy(szLogin, ((SLoggedEntry *)strData.dptr)->szLogin);
		strcpy(szHost, ((SLoggedEntry *)strData.dptr)->szHost);
		(*pEnumerator)(szLogin, szDevice, szHost);
		free(strData.dptr);		
		free(strKey.dptr);
		strKey = strNextKey;
	}
}

bool CLoggedDatabase::CheckRecord(char *pszLogin, char *pszDevice, char *pszHost)
{
	SLoggedEntry sLogged;
	
	datum strLoggedKey = { pszHost, strlen(pszHost) };
	datum strLoggedData = gdbm_fetch(dbfLogged, strLoggedKey);
	
	if (NULL == strLoggedData.dptr)
		return false;
	
	memcpy(&sLogged,  strLoggedData.dptr, sizeof(SLoggedEntry));
			 
	free(strLoggedData.dptr);
	
	if (0 == strcmp(sLogged.szLogin, pszLogin) &&
		 0 == strcmp(sLogged.szDevice, pszDevice) &&
		 0 == strcmp(sLogged.szHost, pszHost))
		return true;
	
	return false;
}

bool CLoggedDatabase::GetFirstRecord(SLoggedEntry *pEntry)
{
	free(strLoggedKey.dptr);
	
	strLoggedKey = gdbm_firstkey(dbfLogged);
	if (NULL == strLoggedKey.dptr)
		return false;
	
	datum strLoggedData = gdbm_fetch(dbfLogged, strLoggedKey);
	if (NULL == strLoggedData.dptr)
		return false;
	
	memcpy(pEntry,  strLoggedData.dptr, sizeof(SLoggedEntry));
			 
	free(strLoggedData.dptr);

	return true;
}

bool CLoggedDatabase::GetNextRecord(SLoggedEntry *pEntry)
{
	datum strNextLoggedKey = gdbm_nextkey(dbfLogged, strLoggedKey);
	if (NULL == strNextLoggedKey.dptr)
		return false;
	
	datum strLoggedData = gdbm_fetch(dbfLogged, strNextLoggedKey);
	if (NULL == strLoggedData.dptr)
		return false;
	
	memcpy(pEntry,  strLoggedData.dptr, sizeof(SLoggedEntry));
			 
	free(strLoggedData.dptr);
	free(strLoggedKey.dptr);
	
	strLoggedKey = strNextLoggedKey;

	return true;
}