/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef WIN32
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#else
#include <winsock.h>
#include <io.h>
#endif

#include "sockio.hpp"
#include "key.hpp"

#define WELCOME							'+'
#define GO_AHEAD							'!'
#define ACTION_GETKEY					'k'
#define OK									'+'
#define ERR									'*'

char *get_key(char *pszServer, int iServerPort, char *pszKeyDirectory)
{
	static char szBuf[256];
	struct sockaddr_in client;
	int sockfd;
	
	if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		snprintf(szBuf, sizeof(szBuf), "socket error: %s", strerror(errno));
		return szBuf;
	}

	struct hostent *pHost = gethostbyname(pszServer);
	if (NULL == pHost) {
		snprintf(szBuf, sizeof(szBuf), "resolving error: %s", pszServer);
		close(sockfd);
		return szBuf;
	}
	memset(&client, 0, sizeof(client));
	client.sin_family = AF_INET;
	client.sin_addr.s_addr = *((unsigned long int *)(pHost->h_addr));
	client.sin_port = htons((unsigned short)iServerPort);
	if (connect(sockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
		snprintf(szBuf, sizeof(szBuf), "connect error: %s", strerror(errno));
		close(sockfd);
		return szBuf;
	}

	unsigned char c = '\0';
	read_byte(sockfd, &c);
	if (WELCOME != c) {
		snprintf(szBuf, sizeof(szBuf), "invalid response from server");
		close(sockfd);
		return szBuf;
	}
	if ((int)strlen("@key@") + 1 != write_string(sockfd, "@key@")) {
		snprintf(szBuf, sizeof(szBuf), "write error");
		close(sockfd);
		return szBuf;
	}
	read_byte(sockfd, &c);			
	if (c != GO_AHEAD) {
		snprintf(szBuf, sizeof(szBuf), "invalid response from server");
		close(sockfd);
		return szBuf;
	}
	write_byte(sockfd, ACTION_GETKEY);
	read_byte(sockfd, &c);
	if (ERR == c) {
		snprintf(szBuf, sizeof(szBuf), "error reading public key from server");
		close(sockfd);
		return szBuf;
	} else if (OK == c) {
		int iKeyLength;
		void *pvData = read_data(sockfd, &iKeyLength);
		if (NULL == pvData) {
			snprintf(szBuf, sizeof(szBuf), "error reading data");
			close(sockfd);
			return szBuf;
		}
		
		char szFileName[256];			// Buffer for the file name for the public key
		
		strncpy(szFileName, pszKeyDirectory, sizeof(szFileName));
#ifndef WIN32
		strncat(szFileName, "/", sizeof(szFileName));
#else
		strncat(szFileName, "\\", sizeof(szFileName));
#endif
		strncat(szFileName, pszServer, sizeof(szFileName));
		FILE *fpKey = fopen(szFileName, "w");
		if (NULL == fpKey) {
			snprintf(szBuf, sizeof(szBuf), "error opening file %s", szFileName);
			free(pvData);
			close(sockfd);
			return szBuf;
		}
		if (iKeyLength != (int)fwrite(pvData, sizeof(char), iKeyLength, fpKey)) {
			snprintf(szBuf, sizeof(szBuf), "error writing to file %s", szFileName);
			fclose(fpKey);
			free(pvData);
			close(sockfd);
			return szBuf;
		}
		fclose(fpKey);
		free(pvData);
	} else {
		snprintf(szBuf, sizeof(szBuf), "invalid response from server");
		close(sockfd);
		return szBuf;
	}
	
	close(sockfd);
	return NULL;
}