#include <stdio.h>
#include <utmp.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <sys/types.h>
#include <gdbm.h>
#include <time.h>
#include <stdlib.h>
#include <signal.h>

#include "conf.hpp"

// Database functions
bool InitDatabases();
void DestroyDatabases();
char *GetTime(time_t);
char *GetLocalTime(time_t);
char *GetLocalDate(time_t);
char *datum2str(datum);
int TimeSort(const void *, const void *);

char szCurrentDatabase[256];
char szHronoDatabase[256];
char szUsersDatabase[256];
char szLoggedDatabase[256];

GDBM_FILE dbfCurrent;
GDBM_FILE dbfHrono;
GDBM_FILE dbfUsers;
GDBM_FILE dbfLogged;

typedef struct {
	char szLogin[UT_NAMESIZE + 1];
	char szDevice[UT_LINESIZE + 1];
	char szHost[48];
	char szGateway[48];	
	time_t tStartTime;
	time_t tFinalTime;
	int iStartInputTraffic;
	int iStartOutputTraffic;
	int iFinalInputTraffic;
	int iFinalOutputTraffic;
} CURRENT_ENTRY;		// The key of this database is the login device

typedef struct {
	char szLogin[UT_NAMESIZE + 1];
	char szDevice[UT_LINESIZE + 1];
	char szHost[48];
} LOGGED_ENTRY;		// The key of this database is the login device

typedef struct {
	time_t tTime;
	int iTraffic;
	int iDisconnect;
} USER_ENTRY;			// The key of this database is the login name

typedef struct {
	char szLogin[UT_NAMESIZE + 1];
	char szDevice[UT_LINESIZE + 1];
	char szHost[48];	
	time_t tStartTime;
	time_t tFinalTime;
	time_t tLoginTime;		// tFinalTime - tStartTime
	int iInputTraffic;
	int iOutputTraffic;
	int iTotalTraffic;		// iInputTraffic + iOutputTraffic
} HRONO_ENTRY;			// The key of this database is the login name

CConfigurationFile *pConfig = NULL;

int main(int argc, char **argv)
{
	char *pszMessage;
	
	static CURRENT_ENTRY strCurrent;
	static USER_ENTRY strUser;
	static HRONO_ENTRY strHrono;
	static LOGGED_ENTRY strLogged;
	
	datum strCurrentKey;
	datum strNextCurrentKey;
	datum strCurrentData;

	datum strUserKey;
	datum strNextUserKey;
	datum strUserData;

	datum strHronoKey;
	datum strNextHronoKey;
	datum strHronoData;
	
	datum strLoggedKey;
	datum strNextLoggedKey;
	datum strLoggedData;
	
	register int i, j;
	
	datum *pHronoKeys = NULL;
	
	if (argc != 2 && argc != 3)
		return 1;

  	pConfig = new CConfigurationFile("/etc/alligator.conf", &pszMessage);
	if (NULL != pszMessage) {
		printf("%s\n", pszMessage);
		return 1;
	}
	
	InitDatabases();
	
	if ((0 == strcmp(argv[1], "-logged")) && (NULL != dbfLogged)) {
		printf("+--------+-------+---------------+\n");
		printf("|%-8s|%-7s|%-15s|\n", "login", "device", "host");
		printf("+--------+-------+---------------+\n");
		strLoggedKey = gdbm_firstkey(dbfLogged);
		while (strLoggedKey.dptr) {
			strNextLoggedKey = gdbm_nextkey(dbfLogged, strLoggedKey);
			strLoggedData = gdbm_fetch(dbfLogged, strLoggedKey);
			memcpy(&strLogged, strLoggedData.dptr, sizeof(LOGGED_ENTRY));		

			printf("|%-8s|%-7s|%-15s|\n",
					 strLogged.szLogin,
			       strLogged.szDevice,
					 strLogged.szHost);
			
			free(strLoggedKey.dptr);
			free(strLoggedData.dptr);
			
			strLoggedKey = strNextLoggedKey;
		}
		printf("+--------+-------+---------------+\n");
	}
	
	if ((0 == strcmp(argv[1], "-current")) && (NULL != dbfCurrent)) {	
		printf("+---------------+---------------+---------------------------+---------------+\n");		
		printf("|%-15s|%-15s|%-27s|%-15s|\n",
				 "login",
				 "device",
				 "host",
				 "gateway");
		printf("|%-15s|%-15s|%-27s|%-15s|\n",
				 "start time",
				 "update time",
				 "received bytes",
				 "sent bytes");
		printf("+---------------+---------------+---------------------------+---------------+\n");		
		strCurrentKey = gdbm_firstkey(dbfCurrent);
		while (strCurrentKey.dptr) {
			strNextCurrentKey = gdbm_nextkey(dbfCurrent, strCurrentKey);
			strCurrentData = gdbm_fetch(dbfCurrent, strCurrentKey);
			memcpy(&strCurrent, strCurrentData.dptr, sizeof(CURRENT_ENTRY));		

			printf("|%-15s|%-15s|%-27s|%-15s|\n",
					 strCurrent.szLogin,
			       strCurrent.szDevice,
					 strCurrent.szHost,
			       strCurrent.szGateway);
			printf("|%15s|%15s|%27d|%15d|\n",
					 GetLocalTime(strCurrent.tStartTime),
			       GetLocalTime(strCurrent.tFinalTime),
				    strCurrent.iFinalInputTraffic - strCurrent.iStartInputTraffic,
				    strCurrent.iFinalOutputTraffic - strCurrent.iStartOutputTraffic);
			printf("+---------------+---------------+---------------------------+---------------+\n");			
			free(strCurrentKey.dptr);
			free(strCurrentData.dptr);
			
			strCurrentKey = strNextCurrentKey;
		}
	}
	
	if ((0 == strcmp(argv[1], "-users")) && (NULL != dbfUsers)) {	
		printf("+---------+---------+----------+----------+\n");		
		printf("|%-9s|%9s|%10s|%10s|\n",
				 "login",
				 "time",
				 "bytes",
				 "leave");
		printf("+---------+---------+----------+----------+\n");
		strUserKey = gdbm_firstkey(dbfUsers);
		while (strUserKey.dptr) {
			strNextUserKey = gdbm_nextkey(dbfUsers, strUserKey);
			strUserData = gdbm_fetch(dbfUsers, strUserKey);
			memcpy(&strUser, strUserData.dptr, sizeof(USER_ENTRY));		

			printf("|%-9s|%c%8s|%10d|%10x|\n",
					 datum2str(strUserKey),
					 strUser.tTime < 0 ? '-' : ' ',
       			 GetTime(strUser.tTime),
					 strUser.iTraffic,
					 strUser.iDisconnect);
			
			free(strUserKey.dptr);
			free(strUserData.dptr);
			
			strUserKey = strNextUserKey;
		}
		printf("+---------+---------+----------+----------+\n");		
	}
	
	if ((0 == strcmp(argv[1], "-hrono")) && (NULL != dbfHrono)) {
		printf("+--------+------+-------------------+--------+--------+--------+--------+\n");		
		printf("|%-8s|%-6s|%-19s|%-8s|%-8s|%-8s|%-8s|\n",
				 "login",
				 "device",
				 "login time",
				 "time",
				 "in",
				 "out",
				 "both");
		printf("+--------+------+-------------------+--------+--------+--------+--------+\n");
		
		i = 0;
		strHronoKey = gdbm_firstkey(dbfHrono);
		while (strHronoKey.dptr) {
			strNextHronoKey = gdbm_nextkey(dbfHrono, strHronoKey);
			pHronoKeys = (datum *)realloc(pHronoKeys, sizeof(datum) * (i + 1));
			pHronoKeys[i] = strHronoKey;
			i += 1;
			strHronoKey = strNextHronoKey;
		}
		
		qsort(pHronoKeys, i, sizeof(datum), TimeSort);
		
		for (j = 0; j < i; j++) {
			strHronoKey = pHronoKeys[j];
			strHronoData = gdbm_fetch(dbfHrono, strHronoKey);
			memcpy(&strHrono, strHronoData.dptr, sizeof(HRONO_ENTRY));
			if (3 != argc || 0 == strcmp(argv[2], strHrono.szLogin)) {
				printf("|%-8s|%-6s|%10s %8s",
						 strHrono.szLogin,
						 strHrono.szDevice,
						 GetLocalDate(strHrono.tStartTime),					 
						 GetLocalTime(strHrono.tStartTime));
				printf("|%8s|%8d|%8d|%8d|\n",
						 GetTime(strHrono.tLoginTime),
						 strHrono.iInputTraffic,
						 strHrono.iOutputTraffic,
						 strHrono.iTotalTraffic);
			}
			free(strHronoData.dptr);
			free(strHronoKey.dptr);
		}
		printf("+--------+------+-------------------+--------+--------+--------+--------+\n");		
		free(pHronoKeys);
	}
	
	DestroyDatabases();
	
	delete pConfig;
	
	return 0;
}

int TimeSort(const void *pKey1, const void *pKey2)
{
	time_t tKey1 = *((time_t *)((datum *)pKey1)->dptr);
	time_t tKey2 = *((time_t *)((datum *)pKey2)->dptr);
	if (tKey1 == tKey2)
		return 0;
	if (tKey1 > tKey2)
		return 1;
	return -1;
}

bool InitDatabases()
{
	strncpy(szCurrentDatabase, pConfig->GetString("Global", "data_dir", "/tmp"), sizeof(szCurrentDatabase));
	strncpy(szHronoDatabase, pConfig->GetString("Global", "data_dir", "/tmp"), sizeof(szHronoDatabase));
	strncpy(szUsersDatabase, pConfig->GetString("Global", "data_dir", "/tmp"), sizeof(szUsersDatabase));
	strncpy(szLoggedDatabase, pConfig->GetString("Global", "data_dir", "/tmp"), sizeof(szLoggedDatabase));
	strcat(szCurrentDatabase, "/current.db");
	strcat(szHronoDatabase, "/hrono.db");
	strcat(szUsersDatabase, "/users.db");
	strcat(szLoggedDatabase, "/logged.db");
	
	dbfCurrent = gdbm_open(szCurrentDatabase, 0, GDBM_READER, 0, NULL);
	dbfHrono = gdbm_open(szHronoDatabase, 0, GDBM_READER, 0, NULL);
	dbfUsers = gdbm_open(szUsersDatabase, 0, GDBM_READER, 0, NULL);
	dbfLogged = gdbm_open(szLoggedDatabase, 0, GDBM_READER, 0, NULL);
	
	if (NULL == dbfCurrent || NULL == dbfUsers || NULL == dbfUsers || NULL == dbfLogged)
		return false;
	
	return true;
}

void DestroyDatabases()
{
	if (NULL != dbfCurrent)
		gdbm_close(dbfCurrent);
	if (NULL != dbfHrono)
		gdbm_close(dbfHrono);
	if (NULL != dbfUsers)
		gdbm_close(dbfUsers);	
	if (NULL != dbfLogged)
		gdbm_close(dbfLogged);	
}

char *GetTime(time_t tTime)
{
	time_t tt = abs(tTime);
	static char szResult[128];
	struct tm *t = gmtime(&tt);
	
	sprintf(szResult, "%02d:%02d:%02d", t->tm_hour, t->tm_min, t->tm_sec);
	
	return szResult;
}

char *GetLocalTime(time_t tTime)
{
	static char szResult[128];
	struct tm *t = localtime(&tTime);
	
	sprintf(szResult, "%02d:%02d:%02d", t->tm_hour, t->tm_min, t->tm_sec);
	
	return szResult;
}

char *GetLocalDate(time_t tTime)
{
	static char szResult[128];
	struct tm *t = localtime(&tTime);
	
	sprintf(szResult, "%02d.%02d.%04d", t->tm_mday, t->tm_mon + 1, t->tm_year + 1900);
	
	return szResult;
}

char *datum2str(datum d)
{
	static char szResult[128];
	
	strncpy(szResult, d.dptr, d.dsize);
	szResult[d.dsize] = '\0';
	
	return szResult;
}