/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>

#ifndef WIN32
#include "config.h"
#endif
#include "obn.hpp"
#include "rsa.hpp"

CBigNumber p;
CBigNumber s;
CBigNumber n;
CBigNumber u;
CBigNumber p1;
CBigNumber p2;
CBigNumber dp1;
CBigNumber dp2;
CBigNumber phi;

CBigNumber pp;
CBigNumber nn;

int RSALoadPrivateKey(char *pszFile)
{
	char szBuf[128];

	FILE *fpPrivate = fopen(pszFile, "r");
	if (NULL == fpPrivate)
		return false;
	fgets(szBuf, sizeof(szBuf), fpPrivate);
	s.Rad64Read(fpPrivate);
	n.Rad64Read(fpPrivate);
	p1.Rad64Read(fpPrivate);
	p2.Rad64Read(fpPrivate);
	phi.Rad64Read(fpPrivate);
	u = CBigNumber::ModInv(p1, p2);
	dp1 = s % (p1 - 1);
	dp2 = s % (p2 - 1);
	fclose(fpPrivate);

	return true;
}

int RSALoadPublicKey(char *pszFile)
{
	char szBuf[128];

	FILE *fpPublic = fopen(pszFile, "r");
	if (NULL == fpPublic)
		return false;
	fgets(szBuf, sizeof(szBuf), fpPublic);
	p.Rad64Read(fpPublic);
	n.Rad64Read(fpPublic);	
	fclose(fpPublic);

	return true;
}

void RSAStorePublicKey()
{
	pp = p;
	nn = n;
}

void RSARetreivePublicKey()
{
	p = pp;
	n = nn;
}

void *RSAEncryptString(char *pszString)
{
	if (n.IsZero() || p.IsZero())
		return pszString;
	
	void *pvResult = calloc(1, sizeof(WORD));

	WORD wBytesPerBlock = n.GetWords() * sizeof(WORD) - 1 - sizeof(WORD);
	WORD wTotalWords = 1;
	WORD wBlockSize;
	WORD wWords;

	do {
// Byte count to form the BigNumber		
		wBlockSize = min(wBytesPerBlock, strlen(pszString) + 1);
// Create BigNumber		
		CBigNumber a(pszString, wBlockSize);
// Find the encrypted data
		CBigNumber b = CBigNumber::ModExp(a, p, n);
// Size in words of the encrypted data		
		wWords = b.GetWords();
// Alloc space, then write the words count and the data		
		pvResult = realloc(pvResult, (wTotalWords + wWords + 1) * sizeof(WORD));
		memcpy((char *)pvResult + wTotalWords * sizeof(WORD), &wWords, sizeof(WORD));
		memcpy((char *)pvResult + (wTotalWords + 1) * sizeof(WORD), b.GetData(), wWords * sizeof(WORD));
		wTotalWords += (wWords + 1);
		
		pszString += wBlockSize;
	} while (wBlockSize == wBytesPerBlock);
	((WORD *)pvResult)[0] = wTotalWords * sizeof(WORD);
	
	return pvResult;
}

char *RSADecryptString(void *pvEncryptedData)
{
	if (n.IsZero() || s.IsZero())
		return (char *)pvEncryptedData;
	
	char *pszResult = NULL;
	WORD wBytes;
	WORD wWords;
	WORD wWordsNow = 0;
	WORD wBytesNow = 0;
	WORD wBytesPerBlock = n.GetWords() * sizeof(WORD) - 1 - sizeof(WORD);
	WORD wTotalWords = ((WORD *)pvEncryptedData)[0];
	
	do {
		CBigNumber a((WORD *)pvEncryptedData + wWordsNow + 1);
		wWords = ((WORD *)pvEncryptedData + wWordsNow + 1)[0];
		
// The chinese remainder theorem
		CBigNumber x = CBigNumber::ModExp(a % p1, dp1, p1);
		CBigNumber y = CBigNumber::ModExp(a % p2, dp2, p2);
		if (y < x)
			y += p2;
		CBigNumber b = x + p1 * (((y - x) * u) % p2);
//		CBigNumber b = CBigNumber::ModExp(a, s, n);
		
		wBytes = ((WORD *)b.GetData())[0];
		if (wBytes <= wBytesPerBlock) {
			pszResult = (char *)realloc(pszResult, wBytesNow + wBytes);
		
			memcpy(pszResult + wBytesNow, ((WORD *)b.GetData() + 1), wBytes);
			wBytesNow += wBytes;
		}
		wWordsNow += (wWords + 1);
	} while (wWordsNow < (wTotalWords / sizeof(WORD)) - 1);
	
	return pszResult;
}
