/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "config.h"
#include "obn.hpp"
#include "rad64.hpp"

// Default constructor. Allocates zero initialized memory for CBigNumber.
CBigNumber::CBigNumber() : wWords(0), fgSign(false)
{
	wAllocated = BLOCK;
	pwBigNumber = (WORD *)calloc(BLOCK, sizeof(WORD));
}

CBigNumber::CBigNumber(WORD wValue)
{
	CBigNumber::wWords = 1;
	CBigNumber::wAllocated = BLOCK;
	CBigNumber::fgSign = false;
	
	pwBigNumber = (WORD *)calloc(BLOCK, sizeof(WORD));
	pwBigNumber[0] = wValue;
}

void CBigNumber::SetWords(WORD wWords)
{
	WORD wOld = wAllocated;
	
	CBigNumber::wWords = wWords;
	if (wWords > wAllocated) {
		wAllocated = (wWords / BLOCK + 1) * BLOCK;
		pwBigNumber = (WORD *)realloc(pwBigNumber, wAllocated * sizeof(WORD));
		for (register WORD w = wOld; w < wAllocated; w++)
			pwBigNumber[w] = 0;
	}
}

CBigNumber::CBigNumber(char *pszHexString) : pwBigNumber(NULL), wWords(0), wAllocated(0), fgSign(false)
{
	char szHex[] = "0123456789ABCDEF";
	
	if ('-' == pszHexString[0]) {
		fgSign = true;
		pszHexString += 1;
	}
	
	int iStringLength = strlen(pszHexString);
	
	wWords = (iStringLength * 4 + BITSINWORD - 1) / BITSINWORD;
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (WORD *)calloc(wAllocated, sizeof(WORD));
	
	for (int i = 0; i < iStringLength; i++)
		pwBigNumber[i / 8] |= (WORD)(strchr(szHex, toupper(pszHexString[iStringLength - i - 1])) - szHex) << 4 * (i % 8);
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(void *pvData, WORD wDataLength) : pwBigNumber(NULL), wWords(0), wAllocated(0), fgSign(false)
{
	wWords = (wDataLength * 8 + BITSINWORD - 1) / BITSINWORD + 1;
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (WORD *)calloc(wAllocated, sizeof(WORD));
	
	pwBigNumber[0] = wDataLength;
	for (WORD i = 0; i < wDataLength; i++)
#ifdef WORDS_BIGENDIAN
		pwBigNumber[i / 4 + 1] |= (WORD)(((char *)pvData)[i] << 8 * (3 - i % 4));
#else
		pwBigNumber[i / 4 + 1] |= (WORD)(((char *)pvData)[i] << 8 * (i % 4));
#endif // WORDS_BIGENDIAN
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(WORD *pwData) : pwBigNumber(NULL), wWords(0), wAllocated(0), fgSign(false)
{
	wWords = *pwData;
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (WORD *)calloc(wAllocated, sizeof(WORD));
	
	memcpy(pwBigNumber, pwData + 1, wWords * sizeof(WORD));
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(const CBigNumber &cNumber)
{
	fgSign = cNumber.fgSign;
	wAllocated = cNumber.wAllocated;
	wWords = cNumber.wWords;
	if (NULL != cNumber.pwBigNumber) {
		pwBigNumber = (WORD *)calloc(wAllocated, sizeof(WORD));
		memcpy(pwBigNumber, cNumber.pwBigNumber, wWords * sizeof(WORD));
	}
}

CBigNumber::~CBigNumber()
{
	if (NULL != pwBigNumber)
		free(pwBigNumber);
}

CBigNumber CBigNumber::operator = (const CBigNumber &cNumber)
{
	if (this != &cNumber) {
		if (NULL != pwBigNumber)
			free(pwBigNumber);
		fgSign = cNumber.fgSign;
		wAllocated = cNumber.wAllocated;
		wWords = cNumber.wWords;
		if (NULL != cNumber.pwBigNumber) {
			pwBigNumber = (WORD *)calloc(wAllocated, sizeof(WORD));
			memcpy(pwBigNumber, cNumber.pwBigNumber, wWords * sizeof(WORD));
		}
	}
	return *this;
}

CBigNumber CBigNumber::operator + (const CBigNumber &cAddend) const
{
	CBigNumber cSum;
	if (fgSign == cAddend.fgSign) {
		cSum = *this;
		cSum.Add(cAddend);
	} else {
		cSum = *this;
		cSum.Sub(cAddend);
	}
	return cSum;
}

CBigNumber & CBigNumber::operator += (const CBigNumber &cAddend)
{
	if (fgSign == cAddend.fgSign)
		Add(cAddend);
	else
		Sub(cAddend);
	return *this;
}

CBigNumber & CBigNumber::operator -= (const CBigNumber &cSubtrahend)
{
	if (fgSign == cSubtrahend.fgSign)
		Sub(cSubtrahend);
	else
		Add(cSubtrahend);
	return *this;
}

void CBigNumber::Add(const CBigNumber &cAddend)
{
	SetWords(max(wWords, cAddend.wWords));

	register WORD i;
	register WORD j;
	register WORD c = 0;
	
	WORD *q = cAddend.pwBigNumber;
	WORD *r = pwBigNumber;
	
	for (i = 0; i < cAddend.wWords; i++) {
		j = HIWORD((DWORD)q[i] + (DWORD)r[i] + (DWORD)c);
		r[i] += (q[i] + c);
		c = j;
	}
	for (; i < wWords; i++) {
		j = HIWORD((DWORD)r[i] + (DWORD)c);
		r[i] += c;
		c = j;
	}
	if (c != 0) {
		SetWords(wWords + 1);
		pwBigNumber[i] = c;
	}
	
	StripLeadingZeroes();	
}

CBigNumber CBigNumber::operator - (const CBigNumber &cSubtrahend) const
{
	CBigNumber cDifference;
	if (fgSign == cSubtrahend.fgSign) {
		cDifference = *this;
		cDifference.Sub(cSubtrahend);
	} else {
		cDifference = *this;
		cDifference.Add(cSubtrahend);
	}
	return cDifference;
}

// Calculated the result of abs(cMinuend) /* this */ - abs(cSubtrahend)
// cMinuend is expected to be >= cMinuend
void CBigNumber::NormalSub(const CBigNumber &cSubtrahend)
{
	register WORD d;
	register WORD i;
	register WORD b = 0;						// borrow

	register WORD *q = cSubtrahend.pwBigNumber;
	register WORD *r = pwBigNumber;	
	for (i = 0; i < cSubtrahend.wWords; i++) {
		d = r[i] < q[i] + b;
		r[i] -= (q[i] + b);
		b = d;
	}
	for (; i < wWords && b; i++) {
		d = r[i] < b;
		r[i] -= b;
		b = d;
	}
	
	StripLeadingZeroes();
}

void CBigNumber::Sub(const CBigNumber &cSubtrahend)
{
	register WORD d;
	register WORD i;
	register WORD b = 0;						// borrow

	WORD *q;
	WORD *r;
	
	if (Cmp(*this, cSubtrahend) == -1) {
		SetWords(cSubtrahend.wWords);
		  
		q = cSubtrahend.pwBigNumber;
		r = pwBigNumber;	
		for (i = 0; i < wWords; i++) {
			d = q[i] < r[i] + b;
			r[i] = q[i] - r[i] - b;
			b = d;
		}
		for (; i < cSubtrahend.wWords; i++) {
			d = q[i] < b;
			r[i] = q[i] - b;
			b = d;
		}
		
		fgSign = !fgSign;
	} else {
		q = cSubtrahend.pwBigNumber;
		r = pwBigNumber;	
		for (i = 0; i < cSubtrahend.wWords; i++) {
			d = r[i] < q[i] + b;
			r[i] -= (q[i] + b);
			b = d;
		}
		for (; i < wWords; i++) {
			d = r[i] < b;
			r[i] -= b;
			b = d;
		}
	}
	
	StripLeadingZeroes();
}

CBigNumber CBigNumber::operator * (const CBigNumber &cMultiplier) const
{
	CBigNumber cProduct;
	if (IsZero() || cMultiplier.IsZero())
		return cProduct;
	cProduct.SetWords(wWords + cMultiplier.wWords);
	
	WORD *p = pwBigNumber;
	WORD *q = cMultiplier.pwBigNumber;
	WORD *r = cProduct.pwBigNumber;
	
	WORD i, j, k;
	register DWORD t;
	register DWORD m;
	for (i = 0; i < wWords; i++) {
		m = p[i];
// This is the critical loop		
		for (j = i, k = 0; j < i + cMultiplier.wWords; j++) {
			t = m * q[j - i] + r[j] + k;
			r[j] = LOWORD(t);
			k = HIWORD(t);
		}
		r[j] = k;
	}
	
	if (fgSign != cMultiplier.fgSign)
		cProduct.fgSign = true;
	
	cProduct.StripLeadingZeroes();
	return cProduct;
}

CBigNumber CBigNumber::operator / (const CBigNumber &cDivisor) const
{
	CBigNumber cQuotient;
	CBigNumber cRemainder = *this;
	
	CBigNumber m = cDivisor;
	CBigNumber s(1);
	
	while (cRemainder > m) {
		m.Shl(); // m <<= 1;
		s.Shl(); // s <<= 1;
	}
	
	while (cRemainder >= cDivisor) {
		while (cRemainder < m) {
			m.Shr(); // m >>= 1;
			s.Shr(); // s >>= 1;
		}
		cRemainder -= m;
		cQuotient += s;
	}
	
	if (fgSign != cDivisor.fgSign)
		cQuotient.fgSign = true;
	
	cQuotient.StripLeadingZeroes();
	return cQuotient;
}

CBigNumber CBigNumber::operator / (WORD wDivisor) const
{
	CBigNumber cQuotient;
	if (IsZero())
		return cQuotient;
	
	cQuotient.SetWords(wWords);
	
	DWORD b = 0;
	
	for (WORD i = wWords - 1; i < wWords; i--) {
		b = b << 32 | pwBigNumber[i];
		cQuotient.pwBigNumber[i] = (WORD)(b / (DWORD)wDivisor);
		b -= (b / (DWORD)wDivisor) * (DWORD)wDivisor;
	}
	
	cQuotient.StripLeadingZeroes();
	
	return cQuotient;
}

WORD CBigNumber::operator % (WORD wDivisor) const
{
	if (IsZero())
		return 0;
	
	DWORD b = 0;
	for (WORD i = wWords - 1; i < wWords; i--) {
		b = b << 32 | pwBigNumber[i];
		b %= wDivisor;
	}
	
	return (WORD)b;
}

CBigNumber CBigNumber::operator % (const CBigNumber &cDivisor) const
{
	CBigNumber cRemainder = *this;
	
	CBigNumber m = cDivisor;
	
/*	while (cRemainder > m)
		m.Shl(); // m <<= 1;
	
	while (cRemainder >= cDivisor) {
		while (cRemainder < m)
			m.Shr(); // m >>= 1;
		
		cRemainder -= m;
	} */

	while (cRemainder > m)
		m <<= 16;
	
	while (cRemainder >= cDivisor) {
		while (cRemainder < m)
			m.Shr();
	
		cRemainder.NormalSub(m);
	}
	
	cRemainder.fgSign = fgSign;
	
	cRemainder.StripLeadingZeroes();
	return cRemainder;
}

CBigNumber CBigNumber::operator << (WORD wBits) const
{
	CBigNumber cResult;
	if (0 != wWords) {
		cResult.SetWords(wWords + 1);
		register WORD i;
		for (i = 0; i < wWords; i++)
			cResult.pwBigNumber[i] = pwBigNumber[i] << wBits;
		for (i = 1; i < wWords + 1; i++)			
			cResult.pwBigNumber[i] |= (pwBigNumber[i - 1] >> (32 - wBits));
	}
	return cResult;
}

CBigNumber & CBigNumber::operator <<= (WORD wBits)
{
	if (0 == wWords)
		return *this;
	WORD q = 32 - wBits;
	if ((pwBigNumber[wWords - 1] >> q) != 0)
		SetWords(wWords + 1);
	WORD *p = pwBigNumber;	
	for (register WORD i = wWords - 1; i > 0; i--) {
		p[i] <<= wBits;
		p[i] |= (p[i - 1] >> q);
	}
	p[0] <<= wBits;
	return *this;
}

void CBigNumber::Shl()
{
	if (0 == wWords)
		return;
	
	if ((pwBigNumber[wWords - 1] & 0x80000000L) != 0)
		SetWords(wWords + 1);
	WORD *p = pwBigNumber;	
	for (register WORD i = wWords - 1; i > 0; i--) {
		p[i] <<= 1;
		p[i] |= (p[i - 1] >> 31);
	}
	p[0] <<= 1;
}

CBigNumber & CBigNumber::operator += (WORD wValue)
{
	WORD c = pwBigNumber[0] + (DWORD)wValue > MAXWORD ? 1 : 0;
	pwBigNumber[0] += wValue;
	
	WORD i = 1;
	while (0 != c) {
		if (i == wWords)
			SetWords(wWords + 1);
		pwBigNumber[i] += c;
		c = pwBigNumber[i] + (DWORD)c > MAXWORD ? 1 : 0;
		i += 1;
	}
	return *this;
}

CBigNumber & CBigNumber::operator -= (WORD wValue)
{
	if (pwBigNumber[0] >= wValue) {
		pwBigNumber[0] -= wValue;
	} else {
		pwBigNumber[0] = (WORD)(pwBigNumber[0] + BASE - wValue);
		WORD i = 1;
		while (0 == pwBigNumber[i])
			pwBigNumber[i++] -= 1;
		pwBigNumber[i] -= 1;
	}
	StripLeadingZeroes();
	return *this;
}

CBigNumber CBigNumber::operator >> (WORD wBits) const
{
	CBigNumber cResult;
	if (0 != wWords) {
		cResult.SetWords(wWords);
		register WORD i;
		for (i = 0; i < wWords; i++)
			cResult.pwBigNumber[i] = pwBigNumber[i] >> wBits;
		for (i = 0; i < wWords - 1; i++)			
			cResult.pwBigNumber[i] |= (pwBigNumber[i + 1] << (32 - wBits));
	}
	cResult.StripLeadingZeroes();
	return cResult;
}

CBigNumber & CBigNumber::operator >>= (WORD wBits)
{
	register WORD *p = pwBigNumber;
	register WORD t = 32 - wBits;
	if (0 != wWords) {
		for (register WORD i = 0; i < wWords - 1; i++)
			p[i] = (p[i] >> wBits) | (p[i + 1] << t);
		p[wWords - 1] >>= wBits;
	}
	StripLeadingZeroes();
	return *this;
}

void CBigNumber::Shr()
{
/*	asm (
		  "jcxz quit\n\t"
		  "clc\n\t"
		  "movl (%%esi, %%ecx, 4), %%ebx\n\t"
		  "next:\n\t"
		  "rcrl (%%esi, %%ecx, 4)\n\t"
		  "decl %%ecx\n\t"
		  "jnz next\n\t"
		  "decl %%ebx\n\t"
		  "testl %%ebx, %%ebx\n\t"
		  "jnz quit\n\t"
		  "decl %0\n\t"
		  "quit:\n\t"
		  : "=g"(wWords)
		  : "c"(wWords), "S"(pwBigNumber - 1)
		  : "ebx", "ecx", "esi");
*/
	if (0 == wWords)
		return;

	WORD *p = pwBigNumber;

	for (WORD i = 0; i < wWords - 1; i++)
		p[i] = (p[i] >> 1) | (p[i + 1] << 31);
	p[wWords - 1] >>= 1;

	if (0 == pwBigNumber[wWords - 1])
		wWords -= 1;
}

bool CBigNumber::operator < (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return true;
	if (!fgSign && cRightValue.fgSign)
		return false;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return iCmp == 1;
	
	return iCmp == -1;
}

bool CBigNumber::operator <= (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return true;
	if (!fgSign && cRightValue.fgSign)
		return false;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return (iCmp == 1) || (iCmp == 0);
	
	return (iCmp == -1) || (iCmp == 0);
}

bool CBigNumber::operator > (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return false;
	if (!fgSign && cRightValue.fgSign)
		return true;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return iCmp == -1;
	
	return iCmp == 1;
}

bool CBigNumber::operator >= (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return false;
	if (!fgSign && cRightValue.fgSign)
		return true;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return (iCmp == -1) || (iCmp == 0);
	
	return (iCmp == 1) || (iCmp == 0);
}

bool CBigNumber::operator == (const CBigNumber &cRightValue) const
{
	if (fgSign != cRightValue.fgSign)
		return false;
	
	return 0 == Cmp(*this, cRightValue);
}

// Returns 0 if abs(cLeftValue) == abs(cRightValue)
// Returns 1 if abs(cLeftValue) > abs(cRightValue)
// Returns -1 if abs(cLeftValue) < abs(cRightValue)
int CBigNumber::Cmp(const CBigNumber &cLeftValue, const CBigNumber &cRightValue) const
{
	if (cLeftValue.wWords > cRightValue.wWords)
		return 1;
	if (cLeftValue.wWords < cRightValue.wWords)
		return -1;
	if (0 == cLeftValue.wWords)
		return 0;
	
	WORD *p = cLeftValue.pwBigNumber;
	WORD *q = cRightValue.pwBigNumber;
	
	register WORD i;
	register WORD j = cLeftValue.wWords;
	for (i = j - 1; i < j; i--) {
		if (p[i] > q[i])
			return 1;
		if (p[i] < q[i])
			return -1;
	}
	return 0;
}

bool CBigNumber::operator != (const CBigNumber &cRightValue) const
{
	return !(*this == cRightValue);
}

bool CBigNumber::operator < (WORD wRightValue)
{
	if (IsZero() && (0 != wRightValue))
		return true;
	if (fgSign)									// Negative number is smaller than
		return true;							// any nonnegative word
	if (wWords != 1)
		return false;
	return pwBigNumber[0] < wRightValue;
}

bool CBigNumber::operator <= (WORD wRightValue)
{
	if (IsZero())
		return true;
	if (fgSign)									// Negative number is smaller than
		return true;							// any nonnegative word
	if (wWords != 1)
		return false;
	return pwBigNumber[0] <= wRightValue;
}

bool CBigNumber::operator > (WORD wRightValue)
{
	if (IsZero() && (0 != wRightValue))
		return false;
	if (fgSign)
		return false;
	if (wWords != 1)
		return false;
	return pwBigNumber[0] > wRightValue;
}

bool CBigNumber::operator >= (WORD wRightValue)
{
	if (IsZero() && (0 == wRightValue))
		return true;
	if (fgSign)
		return false;
	if (wWords != 1)
		return false;
	return pwBigNumber[0] >= wRightValue;
}

bool CBigNumber::operator == (WORD wRightValue)
{
	if (IsZero() && (0 == wRightValue))
		return true;
	if (wWords != 1)
		return false;
	if (fgSign)
		return false;
	return pwBigNumber[0] == wRightValue;
}

bool CBigNumber::operator != (WORD wRightValue)
{
	return !(*this == wRightValue);
}

bool CBigNumber::IsZero() const
{
	register WORD i;
	
	for (i = 0; i < wWords; i++)
		if (pwBigNumber[i] != 0)
			return false;
	return true;
}

CBigNumber CBigNumber::Abs(const CBigNumber &a)
{
	CBigNumber b = a;
	b.fgSign = false;
	return b;
}
/*
CBigNumber CBigNumber::ModExp(const CBigNumber &b, const CBigNumber &c, const CBigNumber &d)
{
	CBigNumber cResult(1);
	
	CBigNumber cCache[4];
	bool fgCache[] = { false, false, false, false };
	
	for (WORD z = c.wWords - 1; z < c.wWords; z--) {
		WORD w = 32;
		WORD u = c.pwBigNumber[z];
		if (z == c.wWords - 1) {
			for (WORD i = 0; (i < 32) && (0 == ((u >> (31 - i)) & 1)); i++, w--);
			w = ((w + 1) / 2) * 2;
			u <<= (32 - w);
		}
		for (WORD y = 0; y < w; y += 2) {
			WORD g = (u >> 30 & 3);
			if (!fgCache[g] && (0 != g)) {
				cCache[g] = ModExpWord(b, g, d);
				fgCache[g] = true;
			}
			
			cResult = ModExpWord(cResult, 4, d);
			if (0 != g)
				cResult = cResult * cCache[g] % d;
			u <<= 2;
		}
	}

	return cResult;
}
*/

CBigNumber CBigNumber::ModExp(const CBigNumber &b, const CBigNumber &c, const CBigNumber &d)
{
	CBigNumber cResult(1);
	CBigNumber cTemp = b % d;
	
	for (WORD z = c.wWords - 1; z < c.wWords; z--) {
		WORD w = 32;
		WORD u = c.pwBigNumber[z];
		if (z == c.wWords - 1) {
			for (WORD i = 0; (i < 32) && (0 == ((u >> (31 - i)) & 1)); i++, w--);
			u <<= (32 - w);
		}
		for (WORD y = 0; y < w; y++) {
			WORD g = (u >> 31 & 1);
			cResult = cResult % d;
			cResult = cResult * cResult;
			if (0 != g)
				cResult = (cResult % d) * cTemp;
			u <<= 1;
		}
	}

	return cResult % d;
}

// This function is designed for c <= 4 && c >= 1
CBigNumber CBigNumber::ModExpWord(const CBigNumber &b, WORD c, const CBigNumber &d)
{
	CBigNumber cResult = b % d;
	CBigNumber cCopy = cResult;

	for (WORD w = 0; w < c - 1; w++)
		cResult = cResult * cCopy % d;

	return cResult % d;
}

CBigNumber CBigNumber::GCD(const CBigNumber &a, const CBigNumber &b)
{
	CBigNumber c = a;
	CBigNumber d = b;
	while (true) {
		if (d.IsZero())
			return c;
		c = c % d;
		if (c.IsZero())
			return d;
		d = d % c;
	}
}

CBigNumber CBigNumber::ModInv(const CBigNumber &a, const CBigNumber &m)
// modular inverse
// returns i in range 1..m-1 such that i * a = 1 mod m
// a must be in range 1..m-1
{
	CBigNumber j(1), i, b = m, c = a, x, y;
	
	while (!c.IsZero()) {
		x = b / c;
		y = b - x * c;
		b = c;
		c = y;
		y = j;
		j = i - j * x;
		i = y;
	} 
	if (i.IsNegative())
		i += m;
	return i;
}

void CBigNumber::StripLeadingZeroes()
{
	register WORD i;
	for (i = wWords - 1; i < wWords && 0L == pwBigNumber[i]; i--);
	wWords = i + 1;
}

void CBigNumber::Dump() const
{
	if (fgSign)
		printf("-");
	for (WORD i = 0; i < wWords; i++)
		printf("%08x", pwBigNumber[wWords - i - 1]);
	printf("\n");
}

int CBigNumber::Rad64Print(FILE *fpOut, int iPosition) const
{
	char *pszString = _l64a(wWords);
	for (WORD j = 0; '\0' != pszString[j]; j++) {
		putc(pszString[j], fpOut);
		iPosition += 1;
		if (iPosition % 64 == 0)
			putc('\n', fpOut);
	}
	for (WORD i = 0; i < wWords; i++) {
		char *pszString = _l64a(pwBigNumber[wWords - i - 1]);
		for (WORD j = 0; '\0' != pszString[j]; j++) {
			putc(pszString[j], fpOut);
			iPosition += 1;
			if (iPosition % 64 == 0)
				putc('\n', fpOut);
		}
	}
	return iPosition;
}

bool CBigNumber::Rad64Read(FILE *fpIn)
{
	char szBuf[8];
	
	WORD i = 0;
	while (i < 6) {
		int c = getc(fpIn);
		if (c == EOF)
			return false;
		if (c != 10 && c != 13)
			szBuf[i++] = c;
	}
	szBuf[i] = '\0';
	WORD w = _a64l(szBuf);
	if (w > 32768)
		return false;
	SetWords(w);
	for (WORD x = 0; x < w; x++) {
		WORD i = 0;
		while (i < 6) {
			int c = getc(fpIn);
			if (c == EOF)
				return false;
			if (c != 10 && c != 13)
				szBuf[i++] = c;
		}
		szBuf[i] = '\0';
		pwBigNumber[wWords - x - 1] = _a64l(szBuf);
	}
	return true;
}