/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <stdio.h>

#include "haval.hpp"

CHavalDigest::CHavalDigest()
{
	Initialize();
}

CHavalDigest::CHavalDigest(void *pvData, WORD wDataLength)
{
	Initialize();
	HashData(pvData, wDataLength);
}

CHavalDigest::CHavalDigest(char *pszString)
{
	Initialize();
	HashData(pszString, strlen(pszString));
}

BYTE *CHavalDigest::GetFingerPrint()
{
	return (BYTE *)pwFingerPrint;
}

void CHavalDigest::Initialize()
{
	dwBits = 0;
	
	pwFingerPrint[0] = 0x243F6A88;
	pwFingerPrint[1] = 0x85A308D3;
	pwFingerPrint[2] = 0x13198A2E;
	pwFingerPrint[3] = 0x03707344;
	pwFingerPrint[4] = 0xA4093822;
	pwFingerPrint[5] = 0x299F31D0;
	pwFingerPrint[6] = 0x082EFA98;
	pwFingerPrint[7] = 0xEC4E6C89;
}

void CHavalDigest::HashData(void *pvData, WORD wDataLength)
{
	dwBits = wDataLength << 3;				// Number of bits in the original text
	
	WORD wFullBlocks = wDataLength / 128;
	WORD wLastBlock = wDataLength % 128;
	WORD i;
	
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(pwBlock, (BYTE *)pvData + i * 128, 128);
		HashBlock();
	}
	memcpy(pwBlock, (BYTE *)pvData + i * 128, wLastBlock);
	((BYTE *)pwBlock)[wLastBlock] = 0x80;
	memset((BYTE *)pwBlock + wLastBlock + 1, 0, 128 - wLastBlock - 1);
	
// If there are not 10 free bytes at the end of pwBlock then hash it and clear
//	pwBlock after that
	if (wLastBlock >= 118) {
		HashBlock();
		memset(pwBlock, 0, 118);
	}
	
	*((BYTE *)pwBlock + 118) = ((FPTLEN & 3) << 6 | (PASSES & 7) << 3 | VERSION & 7);
	*((BYTE *)pwBlock + 119) = (FPTLEN >> 2 & 0x0FF);
	*(WORD *)((BYTE *)pwBlock + 120) = (WORD)(dwBits);
	*(WORD *)((BYTE *)pwBlock + 124) = (WORD)(dwBits >> 32);
	
	HashBlock();
	
	HavalFold();
}

void CHavalDigest::HashBlock()
{
	register WORD t0 = pwFingerPrint[0],
					  t1 = pwFingerPrint[1],
					  t2 = pwFingerPrint[2],
					  t3 = pwFingerPrint[3],
					  t4 = pwFingerPrint[4],
					  t5 = pwFingerPrint[5],
					  t6 = pwFingerPrint[6],
					  t7 = pwFingerPrint[7],
					  *w = pwBlock;

// Pass 1	
	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w     ));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 1 ));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 2 ));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 3 ));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 4 ));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 5 ));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 6 ));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 7 ));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 8 ));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 9 ));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 10));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 11));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 12));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 13));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 14));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 15));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 16));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 17));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 18));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 19));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 20));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 21));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 22));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 23));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 24));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 25));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 26));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 27));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 28));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 29));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 30));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 31));

// Pass 2
	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 5 ), 0x452821E6);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 14), 0x38D01377);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 26), 0xBE5466CF);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 18), 0x34E90C6C);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 11), 0xC0AC29B7);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 28), 0xC97C50DD);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 7 ), 0x3F84D5B5);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 16), 0xB5470917);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w     ), 0x9216D5D9);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 23), 0x8979FB1B);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 20), 0xD1310BA6);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 22), 0x98DFB5AC);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 1 ), 0x2FFD72DB);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 10), 0xD01ADFB7);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 4 ), 0xB8E1AFED);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 8 ), 0x6A267E96);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 30), 0xBA7C9045);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  3), 0xF12C7F99);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0x24A19947);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  9), 0xB3916CF7);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x0801F2E2);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 24), 0x858EFC16);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 29), 0x636920D8);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  6), 0x71574E69);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0xA458FEA3);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 12), 0xF4933D7E);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 15), 0x0D95748F);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 13), 0x728EB658);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  2), 0x718BCD58);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 25), 0x82154AEE);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 31), 0x7B54A41D);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 27), 0xC25A59B5);

// Pass 3
	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0x9C30D539);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  9), 0x2AF26013);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  4), 0xC5D1B023);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 20), 0x286085F0);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 28), 0xCA417918);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 17), 0xB8DB38EF);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w +  8), 0x8E79DCB0);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 22), 0x603A180E);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 29), 0x6C9E0E8B);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 14), 0xB01E8A3E);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 25), 0xD71577C1);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 12), 0xBD314B27);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 24), 0x78AF2FDA);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 30), 0x55605C60);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 16), 0xE65525F3);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 26), 0xAA55AB94);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 31), 0x57489862);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 15), 0x63E81440);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  7), 0x55CA396A);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  3), 0x2AAB10B6);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  1), 0xB4CC5C34);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w     ), 0x1141E8CE);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 18), 0xA15486AF);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 27), 0x7C72E993);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 13), 0xB3EE1411);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  6), 0x636FBC2A);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0x2BA9C55D);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 10), 0x741831F6);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 23), 0xCE5C3E16);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 11), 0x9B87931E);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w +  5), 0xAFD6BA33);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  2), 0x6C24CF5C);

#if PASSES >= 4
// Pass 4
	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 24), 0x7A325381);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  4), 0x28958677);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w     ), 0x3B8F4898);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 14), 0x6B4BB9AF);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  2), 0xC4BFE81B);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  7), 0x66282193);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 28), 0x61D809CC);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 23), 0xFB21A991);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 26), 0x487CAC60);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  6), 0x5DEC8032);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 30), 0xEF845D5D);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 20), 0xE98575B1);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 18), 0xDC262302);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 25), 0xEB651B88);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 19), 0x23893E81);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  3), 0xD396ACC5);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 22), 0x0F6D6FF3);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 11), 0x83F44239);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 31), 0x2E0B4482);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 21), 0xA4842004);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  8), 0x69C8F04A);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 27), 0x9E1F9B5E);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 12), 0x21C66842);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  9), 0xF6E96C9A);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  1), 0x670C9C61);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 29), 0xABD388F0);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  5), 0x6A51A0D2);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 15), 0xD8542F68);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x960FA728);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 10), 0xAB5133A3);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 16), 0x6EEF0B6C);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 13), 0x137A3BE4);
#endif
	
#if PASSES == 5
// Pass 5	
	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 27), 0xBA3BF050);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  3), 0x7EFB2A98);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0xA1F1651D);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 26), 0x39AF0176);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x66CA593E);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 11), 0x82430E88);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 20), 0x8CEE8619);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 29), 0x456F9FB4);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0x7D84A5C3);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w     ), 0x3B8B5EBE);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 12), 0xE06F75D8);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  7), 0x85C12073);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 13), 0x401A449F);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  8), 0x56C16AA6);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 31), 0x4ED3AA62);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 10), 0x363F7706);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  5), 0x1BFEDF72);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  9), 0x429B023D);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 14), 0x37D0D724);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 30), 0xD00A1248);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 18), 0xDB0FEAD3);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  6), 0x49F1C09B);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 28), 0x075372C9);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 24), 0x80991B7B);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  2), 0x25D479D8);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 23), 0xF6E8DEF7);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 16), 0xE3FE501A);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 22), 0xB6794C3B);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  4), 0x976CE0BD);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  1), 0x04C006BA);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 25), 0xC1A94FB6);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 15), 0x409F60C4);
#endif	
	
	pwFingerPrint[0] += t0;
	pwFingerPrint[1] += t1;
	pwFingerPrint[2] += t2;
	pwFingerPrint[3] += t3;
	pwFingerPrint[4] += t4;
	pwFingerPrint[5] += t5;
	pwFingerPrint[6] += t6;
	pwFingerPrint[7] += t7;
}

void CHavalDigest::HavalFold()
{
#if FPTLEN == 128
	wTemp = (pwFingerPrint[7] & 0x000000FF) |
			  (pwFingerPrint[6] & 0xFF000000) |
	  		  (pwFingerPrint[5] & 0x00FF0000) |
	        (pwFingerPrint[4] & 0x0000FF00);
	pwFingerPrint[0] += ror(wTemp, 8);

	wTemp = (pwFingerPrint[7] & 0x0000FF00) |
           (pwFingerPrint[6] & 0x000000FF) |
           (pwFingerPrint[5] & 0xFF000000) |
           (pwFingerPrint[4] & 0x00FF0000);
	pwFingerPrint[1] += ror(wTemp, 16);

	wTemp = (pwFingerPrint[7] & 0x00FF0000) |
	  		  (pwFingerPrint[6] & 0x0000FF00) |
	  		  (pwFingerPrint[5] & 0x000000FF) |
	  		  (pwFingerPrint[4] & 0xFF000000);
	pwFingerPrint[2] += ror(wTemp, 24);

	wTemp = (pwFingerPrint[7] & 0xFF000000) |
	  		  (pwFingerPrint[6] & 0x00FF0000) |
	  		  (pwFingerPrint[5] & 0x0000FF00) |
	  		  (pwFingerPrint[4] & 0x000000FF);
	pwFingerPrint[3] += wTemp;
#elif FPTLEN == 160
	wTemp = (pwFingerPrint[7] & (WORD)0x3F) |
	  		  (pwFingerPrint[6] & ((WORD)0x7F << 25)) |
	  		  (pwFingerPrint[5] & ((WORD)0x3F << 19));
	pwFingerPrint[0] += ror(wTemp, 19);

	wTemp = (pwFingerPrint[7] & ((WORD)0x3F <<  6)) |
	  		  (pwFingerPrint[6] &  (WORD)0x3F) |
	  		  (pwFingerPrint[5] & ((WORD)0x7F << 25));
	pwFingerPrint[1] += ror(wTemp, 25);

	wTemp = (pwFingerPrint[7] & ((WORD)0x7F << 12)) |
	  		  (pwFingerPrint[6] & ((WORD)0x3F <<  6)) |
	  		  (pwFingerPrint[5] &  (WORD)0x3F);
	pwFingerPrint[2] += wTemp;

	wTemp = (pwFingerPrint[7] & ((WORD)0x3F << 19)) |
	  		  (pwFingerPrint[6] & ((WORD)0x7F << 12)) |
	  		  (pwFingerPrint[5] & ((WORD)0x3F <<  6));
	pwFingerPrint[3] += wTemp >> 6;

	wTemp = (pwFingerPrint[7] & ((WORD)0x7F << 25)) |
	  		  (pwFingerPrint[6] & ((WORD)0x3F << 19)) |
	 		  (pwFingerPrint[5] & ((WORD)0x7F << 12));
	pwFingerPrint[4] += wTemp >> 12;
#elif FPTLEN == 192
	wTemp = (pwFingerPrint[7] & (WORD)0x1F) |
	  		  (pwFingerPrint[6] & ((WORD)0x3F << 26));
	pwFingerPrint[0] += ror(wTemp, 26);

	wTemp = (pwFingerPrint[7] & ((WORD)0x1F <<  5)) |
	  		  (pwFingerPrint[6] &  (WORD)0x1F);
	pwFingerPrint[1] += wTemp;

	wTemp = (pwFingerPrint[7] & ((WORD)0x3F << 10)) |
	  		  (pwFingerPrint[6] & ((WORD)0x1F <<  5));
	pwFingerPrint[2] += wTemp >> 5;

	wTemp = (pwFingerPrint[7] & ((WORD)0x1F << 16)) |
	  		  (pwFingerPrint[6] & ((WORD)0x3F << 10));
	pwFingerPrint[3] += wTemp >> 10;

	wTemp = (pwFingerPrint[7] & ((WORD)0x1F << 21)) |
	  		  (pwFingerPrint[6] & ((WORD)0x1F << 16));
	pwFingerPrint[4] += wTemp >> 16;

	wTemp = (pwFingerPrint[7] & ((WORD)0x3F << 26)) |
	  		  (pwFingerPrint[6] & ((WORD)0x1F << 21));
	pwFingerPrint[5] += wTemp >> 21;
#elif FPTLEN == 224
	pwFingerPrint[0] += (pwFingerPrint[7] >> 27) & 0x1F;
	pwFingerPrint[1] += (pwFingerPrint[7] >> 22) & 0x1F;
	pwFingerPrint[2] += (pwFingerPrint[7] >> 18) & 0x0F;
	pwFingerPrint[3] += (pwFingerPrint[7] >> 13) & 0x1F;
	pwFingerPrint[4] += (pwFingerPrint[7] >>  9) & 0x0F;
	pwFingerPrint[5] += (pwFingerPrint[7] >>  4) & 0x1F;
	pwFingerPrint[6] +=  pwFingerPrint[7]        & 0x0F;
#endif
}