#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "sockio.hpp"
#include "server.hpp"
#include "conf.hpp"
#include "logfile.hpp"
#include "users.hpp"
#include "current.hpp"
#include "hrono.hpp"
#include "firewall.hpp"
#include "account.hpp"
#include "tokenized_string.hpp"
#include "trim.hpp"
#include "rsa.hpp"
#include "key.hpp"

extern CConfigurationFile *pConfig;

bool action_login(int, char *, bool);
bool action_logout(int, char *, bool);
bool action_add_user(int, int, int, char *, char *, CTokenizedString &);
bool action_delete_user(int, int, int, char *, char *, CTokenizedString &);
bool action_get_user(int, char *);
bool action_set_user(int, char *);
bool action_dump_users(int, char *);
bool action_dump_current(int, char *);
bool action_dump_hrono(int, char *);
void action_getkey(int);

void server(int sockfd, bool fgTrustedHost)
{
	char szLogin[32];
	char *pszPassword = NULL;
	bool fgTrusted = false;
	int newsockfd = -1;
	
	write_byte(sockfd, WELCOME);
	
	if (-1 == read_string(sockfd, szLogin, sizeof(szLogin)))
		write_byte(sockfd, TIMEOUT);
	
	CTokenizedString cPasswdServers(pConfig->GetString("Passwdd", "host", "localhost"), ";,");
	int iServerPort = pConfig->GetInteger("Passwdd", "port", 1099);

	if (0 != strcmp(szLogin, "@key@")) {
		if (NULL == (pszPassword = read_password(sockfd)))
			write_byte(sockfd, TIMEOUT);

		char *pszFirstServer = cPasswdServers.GetFirstString();	
		if ((0 == strcmp(pszPassword, "@login@") ||
			  0 == strcmp(pszPassword, "@logout@")) &&
			 (true == fgTrustedHost)) {
			fgTrusted = true;
		} else {
			if (-1 == (newsockfd = connect_passwdd_server(pszFirstServer, iServerPort))) {
				write_byte(sockfd, LOGIN_FAILED);
				free(pszPassword);
				return;
			}
			if (false == passwdd_check_login(pszFirstServer, iServerPort, newsockfd, szLogin, pszPassword)) {
				write_byte(sockfd, LOGIN_FAILED);
				close(newsockfd);
				free(pszPassword);
				return;
			}
		}
	}
	write_byte(sockfd, GO_AHEAD);
	unsigned char cAction = '\0';
	if (true != read_byte(sockfd, &cAction)) {
		write_byte(sockfd, TIMEOUT);
		if (NULL != pszPassword)
			free(pszPassword);
		return;
	}
	if ((ACTION_ADDUSER != cAction) &&
		 (ACTION_DELETEUSER != cAction)) {
		write_byte(newsockfd, ACTION_NOTHING);
		close(newsockfd);				// We have nothing to do with passwdd.
											// Authentication is successfull, so
											// bye for now.
	}
	if ((true == fgTrusted) && (ACTION_TRUSTEDLOGIN != cAction) && (ACTION_TRUSTEDLOGOUT != cAction)) {
		write_byte(sockfd, ERROR);
		if (NULL != pszPassword)
			free(pszPassword);
		return;
	}
	if (0 == strcmp("@key@", szLogin) && (ACTION_GETKEY != cAction)) {
		write_byte(sockfd, ERROR);
		return;
	}
	switch (cAction) {
		case ACTION_TRUSTEDLOGIN:
		case ACTION_LOGIN:
			action_login(sockfd, szLogin, fgTrusted);
		break;
		case ACTION_TRUSTEDLOGOUT:
		case ACTION_LOGOUT:
			action_logout(sockfd, szLogin, fgTrusted);
		break;
		case ACTION_ADDUSER:
			action_add_user(sockfd, newsockfd, iServerPort, szLogin, pszPassword, cPasswdServers);
		break;
		case ACTION_DELETEUSER:
			action_delete_user(sockfd, newsockfd, iServerPort, szLogin, pszPassword, cPasswdServers);
		break;
		case ACTION_GETUSER:
			action_get_user(sockfd, szLogin);
		break;
		case ACTION_SETUSER:
			action_set_user(sockfd, szLogin);
		break;
		case ACTION_DUMPUSERS:
			action_dump_users(sockfd, szLogin);
	   break;
		case ACTION_DUMPCURRENT:
			action_dump_current(sockfd, szLogin);
	   break;
		case ACTION_DUMPHRONO:
			action_dump_hrono(sockfd, szLogin);
	   break;
		case ACTION_GETKEY:
			action_getkey(sockfd);
		break;
		default:
			write_byte(sockfd, INVALID_ACTION);
			free(pszPassword);
			return;
		break;
	}
	write_byte(sockfd, GOOD_BYE);
	free(pszPassword);
}

void action_getkey(int sockfd)
{
	syslog(LOG_INFO, "request for public key");

	char *pszPublicKey = pConfig->GetString("Security", "public_key", "/etc/passwdd.pbk");
	FILE *fpPublicKey = fopen(pszPublicKey, "rb");
	if (NULL == fpPublicKey) {
		write_byte(sockfd, ERROR);
		return;
	}
	if (0 != fseek(fpPublicKey, 0, SEEK_END)) {
		write_byte(sockfd, ERROR);
		fclose(fpPublicKey);
		return;
	}
	long lFileSize = ftell(fpPublicKey);
	if (-1 == lFileSize)	{
		write_byte(sockfd, ERROR);
		fclose(fpPublicKey);
		return;
	}
	rewind(fpPublicKey);
	void *pvPublicKey = new char[lFileSize];
	if (NULL == pvPublicKey) {
		write_byte(sockfd, ERROR);
		fclose(fpPublicKey);
		return;
	}
	if (lFileSize != (long)fread(pvPublicKey, sizeof(char), lFileSize, fpPublicKey)) {
		write_byte(sockfd, ERROR);
		delete pvPublicKey;
		fclose(fpPublicKey);
		return;
	}
	write_byte(sockfd, OK);
	if (lFileSize != write_data(sockfd, pvPublicKey, lFileSize)) {
		delete pvPublicKey;
		fclose(fpPublicKey);
		return;
	}
	delete pvPublicKey;
	fclose(fpPublicKey);
}

bool action_dump_hrono(int sockfd, char *pszLogin)
{
	char szTarget[32];
	time_t tStartTime;
	time_t tFinalTime;

	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		 (false == read_integer(sockfd, (int *)&tStartTime)) ||
		 (false == read_integer(sockfd, (int *)&tFinalTime))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
// All dates are in GMT		
	tStartTime += timezone + localtime(&tStartTime)->tm_isdst * -3600;
	tFinalTime += timezone + localtime(&tStartTime)->tm_isdst * -3600;
// You need to be in the staff group to be able to see other users accounts
	if (0 != strcmp(szTarget, pszLogin) &&
		 false == check_staff(pszLogin)) {		
		lprintf("Unauthorized dump hrono request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszMessage;
	CHronoDatabase *pHronoDatabase;
	if (NULL == (pHronoDatabase = new CHronoDatabase(pConfig, &pszMessage, true))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pHronoDatabase;
		return false;
	}
	int iRecords = pHronoDatabase->GetCount(szTarget, tStartTime, tFinalTime) + 1;
	int iRecordsPerPage = pConfig->GetInteger("Global", "records_per_page", 10);
	if (iRecordsPerPage < 1 || iRecordsPerPage > 50)
		iRecordsPerPage = 10;
	int iPages = iRecords / iRecordsPerPage;
	iPages += (iRecords % iRecordsPerPage ? 1 : 0);
	write_integer(sockfd, iRecords == 1 ? 0 : iPages);
	int iPage;
	if (1 != iRecords) {
		if (false == read_integer(sockfd, &iPage)) {
			write_string(sockfd, "");
			delete pHronoDatabase;
			return false;
		}
		if (iPage < 1 || iPage > iPages) {
			write_string(sockfd, "");
			delete pHronoDatabase;
			return false;
		}
		SHronoEntry strEntry;
		for (int i = 0; i < iRecordsPerPage; i++) {
			bool fgResult = pHronoDatabase->ReadRecord(&strEntry, (iPage - 1) * iRecordsPerPage + i);
			if (false == fgResult)
				break;
			write_string(sockfd, strEntry.szLogin);
			write_string(sockfd, strEntry.szDevice);
			write_string(sockfd, strEntry.szHost);
			write_integer(sockfd, (int)strEntry.tStartTime);
			write_integer(sockfd, (int)strEntry.tFinalTime);
			write_integer(sockfd, (int)strEntry.tLoginTime);
			write_integer(sockfd, strEntry.iInputTraffic);
			write_integer(sockfd, strEntry.iOutputTraffic);
			write_integer(sockfd, strEntry.iTotalTraffic);
			unsigned char c;
			read_byte(sockfd, &c);
			if (c != NEXT_RECORD)
				break;
		}
		if (iPage == iPages) {
			time_t tTotalTime;
			int iTotalInputTraffic;
			int iTotalOutputTraffic;
			if (true == pHronoDatabase->GetTotals(&tTotalTime, &iTotalInputTraffic, &iTotalOutputTraffic)) {
				write_string(sockfd, "@total@");
				write_string(sockfd, "");
				write_string(sockfd, "");
				write_integer(sockfd, 0);
				write_integer(sockfd, 0);
				write_integer(sockfd, (int)tTotalTime);
				write_integer(sockfd, iTotalInputTraffic);
				write_integer(sockfd, iTotalOutputTraffic);
				write_integer(sockfd, iTotalInputTraffic + iTotalOutputTraffic);
				unsigned char c;
				read_byte(sockfd, &c);
			}
		}
		write_string(sockfd, "");
		lprintf("User %s got hrono dump for user(s) %s...\n", pszLogin, szTarget);
	}
	delete pHronoDatabase;
	return true;
}

bool action_dump_current(int sockfd, char *pszLogin)
{
	char *pszMessage;
	CCurrentDatabase *pCurrentDatabase;
	if (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pCurrentDatabase;
		return false;
	}
	SCurrentEntry strCurrentEntry;
	if (true == pCurrentDatabase->GetFirstRecord(&strCurrentEntry))
		do {
			write_string(sockfd, strCurrentEntry.szLogin);
			write_string(sockfd, strCurrentEntry.szDevice);
			write_string(sockfd, strCurrentEntry.szHost);
			write_string(sockfd, strCurrentEntry.szGateway);
			write_integer(sockfd, strCurrentEntry.tStartTime);
			write_integer(sockfd, strCurrentEntry.tFinalTime);
			write_integer(sockfd, strCurrentEntry.iFinalInputTraffic);
			write_integer(sockfd, strCurrentEntry.iFinalOutputTraffic);
			unsigned char c;
			read_byte(sockfd, &c);
			if (c != NEXT_RECORD)
				break;
		} while (true == pCurrentDatabase->GetNextRecord(&strCurrentEntry));
	write_string(sockfd, "");
	lprintf("User %s got current dump...\n", pszLogin);
	delete pCurrentDatabase;
	return true;
}

bool action_dump_users(int sockfd, char *pszLogin)
{
	char szTarget[32];

	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
// You need to be in the staff group to be able to see other users accounts
	if (0 != strcmp(szTarget, pszLogin) &&
		 false == check_staff(pszLogin)) {		
		lprintf("Unauthorized dump users request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszMessage;
	CUsersDatabase *pUsersDatabase;
	if (NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage, true))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		return false;
	}
	int iRecords = pUsersDatabase->GetCount(szTarget) + 1;
	int iRecordsPerPage = pConfig->GetInteger("Global", "records_per_page", 10);
	if (iRecordsPerPage < 1 || iRecordsPerPage > 50)
		iRecordsPerPage = 10;
	int iPages = iRecords / iRecordsPerPage;
	iPages += (iRecords % iRecordsPerPage ? 1 : 0);
	write_integer(sockfd, iRecords == 1 ? 0 : iPages);
	
	int iPage;
	if (1 != iRecords) {
		if (false == read_integer(sockfd, &iPage)) {
			write_string(sockfd, "");
			delete pUsersDatabase;
			return false;
		}
		if (iPage < 1 || iPage > iPages) {
			write_string(sockfd, "");
			delete pUsersDatabase;
			return false;
		}
		SUsersEntry strEntry;
		for (int i = 0; i < iRecordsPerPage; i++) {
			char *pszUser = pUsersDatabase->ReadRecord(&strEntry, (iPage - 1) * iRecordsPerPage + i);
			if ('\0' == *pszUser)
				break;
			write_string(sockfd, pszUser);
			write_integer(sockfd, (int)strEntry.tTime);
			write_integer(sockfd, strEntry.iTraffic);
			write_integer(sockfd, strEntry.iFlags);
			unsigned char c;
			read_byte(sockfd, &c);
			if (c != NEXT_RECORD)
				break;
		}
		if (iPage == iPages) {
			time_t tTotalTime;
			int iTotalTraffic;
			if (true == pUsersDatabase->GetTotals(&tTotalTime, &iTotalTraffic)) {
				write_string(sockfd, "@total@");						
				write_integer(sockfd, (int)tTotalTime);
				write_integer(sockfd, iTotalTraffic);
				write_integer(sockfd, 0);
				unsigned char c;
				read_byte(sockfd, &c);
			}
		}
		write_string(sockfd, "");
		lprintf("User %s got users dump for user(s) %s...\n", pszLogin, szTarget);
	}
	delete pUsersDatabase;
	return true;
}

bool action_set_user(int sockfd, char *pszLogin)
{
	char szTarget[32];
	int iTime;
	int iTraffic;
	int iDisconnect;
	int iDelete;

	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		 (false == read_integer(sockfd, &iTime)) ||
		 (false == read_integer(sockfd, &iTraffic)) ||
		 (false == read_integer(sockfd, &iDisconnect)) ||
		 (false == read_integer(sockfd, &iDelete))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
// You need to be in the staff group to be able to see other users accounts
	if (false == check_staff(pszLogin)) {		
		lprintf("Unauthorized set user request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszMessage;
	CUsersDatabase *pUsersDatabase;
	if (NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		return false;
	}
	if (false == pUsersDatabase->ExistsRecord(szTarget)) {
		lprintf("There is no account %s in attempt to set info for it...\n", szTarget);
		write_byte(sockfd, INVALID_USER);
		delete pUsersDatabase;
		return false;
	}
	if (false == pUsersDatabase->UpdateRecord(szTarget, (time_t)iTime, iTraffic, (iDelete << 1) | iDisconnect)) {
		lprintf("Error updating user's record for for %s...\n", szTarget);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		return false;
	}
	delete pUsersDatabase;
	lprintf("User %s set info for user %s...\n", pszLogin, szTarget);
	write_byte(sockfd, OK);
	return true;
}

bool action_get_user(int sockfd, char *pszLogin)
{
	SUsersEntry strUsersEntry;
	char szTarget[32];

	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
// You need to be in the staff group to be able to see other users accounts
	if (0 != strcmp(szTarget, pszLogin) && false == check_staff(pszLogin)) {		
		lprintf("Unauthorized get user request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		return false;
	}
	char *pszMessage1;
	char *pszMessage2;
	CUsersDatabase *pUsersDatabase = NULL;
	CCurrentDatabase *pCurrentDatabase = NULL;
	if ((NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage1, true))) ||
		 (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage2, true)))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	if (NULL != pszMessage1 || NULL != pszMessage2) {
		lprintf("%s\n", pszMessage1 != NULL ? pszMessage1 : pszMessage2);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	if (false == pUsersDatabase->ExistsRecord(szTarget)) {
		lprintf("There is no account %s in attempt to get info for it...\n", szTarget);
		write_byte(sockfd, INVALID_USER);
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	if (false == pUsersDatabase->ReadRecord(szTarget, &strUsersEntry)) {
		lprintf("Error reading users' record for for %s...\n", szTarget);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	time_t tCurrentTime = pCurrentDatabase->GetTime(szTarget);
	int iCurrentTraffic = pCurrentDatabase->GetTraffic(szTarget);
	delete pUsersDatabase;
	delete pCurrentDatabase;
	lprintf("User %s got info for user %s...\n", pszLogin, szTarget);
	write_byte(sockfd, OK);
	write_integer(sockfd, strUsersEntry.tTime);
	write_integer(sockfd, strUsersEntry.iTraffic);
	write_integer(sockfd, strUsersEntry.iFlags);
	write_integer(sockfd, (int)tCurrentTime);
	write_integer(sockfd, iCurrentTraffic);
	return true;
}

bool action_delete_user(int sockfd, int newsockfd, int iServerPort, char *pszLogin, char *pszPassword, CTokenizedString &cPasswdServers)
{
	char szTarget[32];

	if (-1 == read_string(sockfd, szTarget, sizeof(szTarget))) {
		write_byte(sockfd, TIMEOUT);
		close(newsockfd);
		return false;
	}
	if (false == check_staff(pszLogin)) {
		lprintf("Unauthorized delete request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		close(newsockfd);
		return false;
	}
	lprintf("Deleting account %s from passwdd servers...\n", szTarget);
	
	char *pszNextServer;
	do {
		if (false == passwdd_delete_user(newsockfd, szTarget)) {
			lprintf("Error deleting account %s by user %s from passwdd server for the above user...\n", szTarget, pszLogin);
			write_byte(sockfd, ERROR);
			close(newsockfd);
			return false;
		}
		close(newsockfd);
		pszNextServer = cPasswdServers.GetNextString();
		if (pszNextServer) {
			lprintf("Deleting account %s from passwdd server %s...\n", szTarget, pszNextServer);
			if (-1 == (newsockfd = connect_passwdd_server(pszNextServer, iServerPort))) {
				write_byte(sockfd, ERROR);
				return false;
			}
			if (false == passwdd_check_login(pszNextServer, iServerPort, newsockfd, pszLogin, pszPassword)) {
				write_byte(sockfd, LOGIN_FAILED);
				close(newsockfd);
				return false;
			}
		}
	} while (pszNextServer);
	char *pszMessage;
	CUsersDatabase *pUsersDatabase;
	if (NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error deleting user %s...\n", szTarget);
		write_byte(sockfd, ERROR);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		return false;
	}
	if (false == pUsersDatabase->ExistsRecord(szTarget)) {
		lprintf("There is no account %s in attempt to delete it...\n", szTarget);
		write_byte(sockfd, INVALID_USER);
		delete pUsersDatabase;
		return false;
	}
	if (false == pUsersDatabase->DeleteRecord(szTarget)) {
		lprintf("Error deleting account %s from %s...\n", szTarget, pszLogin);
		write_byte(sockfd, ERROR);
		delete pUsersDatabase;
		return false;
	}
	lprintf("Account deleted:\n");
	lprintf("  By: %s\n", pszLogin);
	lprintf("  Account name: %s\n", szTarget);
	write_byte(sockfd, OK);
	delete pUsersDatabase;
	return true;
}

bool action_add_user(int sockfd, int newsockfd, int iServerPort, char *pszLogin, char *pszPassword, CTokenizedString &cPasswdServers)
{
	char szTarget[32];
	char szGecos[128];
	int iTime;
	int iTraffic;
	int iDisconnect;
	int iDelete;
	
	if ((-1 == read_string(sockfd, szTarget, sizeof(szTarget))) ||
		 (-1 == read_string(sockfd, szGecos, sizeof(szGecos))) ||
		 (false == read_integer(sockfd, &iTime)) ||
		 (false == read_integer(sockfd, &iTraffic)) ||
		 (false == read_integer(sockfd, &iDisconnect)) ||
		 (false == read_integer(sockfd, &iDelete))) {
		write_byte(sockfd, TIMEOUT);
		close(newsockfd);
		return false;
	}
		
	if (strlen(szTarget) < 1 || strlen(szTarget) > 8) {
		lprintf("Invalid account name %s...\n", szTarget);
		write_byte(sockfd, ERROR);
		close(newsockfd);
		return false;
	}
	for (int q = 0; '\0' != szTarget[q]; q++)
  		if (!isalnum(szTarget[q])) {
			lprintf("Invalid account name %s...\n", szTarget);
			write_byte(sockfd, ERROR);
			close(newsockfd);
			return false;
		}
		
	if (false == check_staff(pszLogin)) {
		lprintf("Unauthorized add request from user %s...\n", pszLogin);
		write_byte(sockfd, NO_RIGHTS);
		close(newsockfd);
		return false;
	}
	char *pszMessage;
	CUsersDatabase *pUsersDatabase;
	if (NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);
		close(newsockfd);
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		close(newsockfd);
		delete pUsersDatabase;
		return false;
	}
	if (true == pUsersDatabase->ExistsRecord(szTarget)) {
		lprintf("User already exists: %s...\n", szTarget);
		write_byte(sockfd, INVALID_USER);
		close(newsockfd);
		return false;
	}
	if (false == pUsersDatabase->AddRecord(szTarget, (time_t)iTime, iTraffic, iDisconnect | (iDelete << 1))) {
		lprintf("Error adding account for %s from %s...\n", szTarget, pszLogin);
		write_byte(sockfd, ERROR);
		close(newsockfd);
		return false;
	}
	delete pUsersDatabase;
	lprintf("Account added:\n");
	lprintf("  By: %s\n", pszLogin);
	lprintf("  Account name: %s\n", szTarget);
	lprintf("  Gecos: %s\n", szGecos);
	lprintf("  Time quota: %d\n", iTime);
	lprintf("  Traffic quota: %d\n", iTraffic);
	lprintf("  Drop flag: %d\n", iDisconnect);
	lprintf("  Delete flag: %d\n", iDelete);
	lprintf("Adding account to passwdd servers...\n");
	
	char *pszNextServer;
	do {
		if (false == passwdd_add_user(newsockfd, szTarget, szGecos)) {
			lprintf("Error adding account to passwdd server for the above user...\n");
			write_byte(sockfd, ERROR);
			close(newsockfd);
			return false;
		}
		close(newsockfd);
		pszNextServer = cPasswdServers.GetNextString();
		if (pszNextServer) {
			lprintf("Adding account to passwdd server %s...\n", pszNextServer);
			if (-1 == (newsockfd = connect_passwdd_server(pszNextServer, iServerPort))) {
				write_byte(sockfd, ERROR);
				return false;
			}
			if (false == passwdd_check_login(pszNextServer, iServerPort, newsockfd, pszLogin, pszPassword)) {
				write_byte(sockfd, LOGIN_FAILED);
				close(newsockfd);
				return false;
			}
		}
	} while (pszNextServer);
	write_byte(sockfd, OK);
	return true;
}

bool action_login(int sockfd, char *pszLogin, bool fgTrusted)
{
	SCurrentEntry strCurrentEntry;
	char szDevice[UT_LINESIZE + 1];
	char szHost[CE_HOSTSIZE];
	char szGateway[CE_HOSTSIZE];

	if ((-1 == read_string(sockfd, szDevice, sizeof(szDevice))) ||
		 (-1 == read_string(sockfd, szHost, sizeof(szHost))) ||
		 (-1 == read_string(sockfd, szGateway, sizeof(szGateway)))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	if (!fgTrusted)
		if (-1 == pConfig->GetInteger("Blocked", szHost, -1)) {
			lprintf("Not blocked user trying to log in as %s...\n", pszLogin);
			write_byte(sockfd, INVALID_USER);
			return false;
		}
	
	char *pszMessage;
	CCurrentDatabase *pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage, true);
	if (NULL == pCurrentDatabase) {
		lprintf("Memory allocation error...\n");
		write_byte(sockfd, ERROR);				
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		write_byte(sockfd, ERROR);
		delete pCurrentDatabase;
		return false;
	}
	bool fgLogged = pCurrentDatabase->GetRecord(szHost, &strCurrentEntry);
	delete pCurrentDatabase;
	if (true == fgLogged) {
		if (0 == strcmp(pszLogin, strCurrentEntry.szLogin) && !fgTrusted) {
			write_byte(sockfd, ALREADY_LOGGED);
			return false;
		}
  		if (0 != UserDown(strCurrentEntry.szLogin, strCurrentEntry.szDevice,	strCurrentEntry.szHost, strCurrentEntry.szGateway)) {
			write_byte(sockfd, ERROR);
			return false;
		}
	}
	int iResult = UserUp(pszLogin, szDevice, szHost, szGateway);
	if (1 == iResult)
		write_byte(sockfd, ERROR);
	if (2 == iResult)
		write_byte(sockfd, INVALID_USER);
	if (3 == iResult)
		write_byte(sockfd, NO_RIGHTS);
	if (0 == iResult) {
		if (!fgTrusted)		// There are no blocking firewall rules
		  							// for dial-up users
			if (false == EnableHost(szHost, pConfig->GetInteger("Blocked", szHost, -1))) {
				lprintf("Error removing blocking firewall rule when logging as user %s...\n", pszLogin);
				write_byte(sockfd, ERROR);
				return false;
			}
		write_byte(sockfd, OK);
		return true;
	}
	return false;
}

bool action_logout(int sockfd, char *pszLogin, bool fgTrusted)
{
	char szDevice[UT_LINESIZE + 1];
	char szHost[CE_HOSTSIZE];
	char szGateway[CE_HOSTSIZE];

	if ((-1 == read_string(sockfd, szDevice, sizeof(szDevice))) ||
		 (-1 == read_string(sockfd, szHost, sizeof(szHost))) ||
		 (-1 == read_string(sockfd, szGateway, sizeof(szGateway)))) {
		write_byte(sockfd, TIMEOUT);
		return false;
	}
	if (!fgTrusted)
		if (false == DisableHost(szHost, pConfig->GetInteger("Blocked", szHost, -1))) {
			lprintf("Error inserting blocking firewall rule when logging as user %s...\n", pszLogin);
			write_byte(sockfd, ERROR);
			return false;
		}
	int iResult = UserDown(pszLogin, szDevice, szHost, szGateway);
	if (1 == iResult)
		write_byte(sockfd, ERROR);
	if (2 == iResult)
		write_byte(sockfd, INVALID_USER);
	if (0 == iResult) {
		if (!fgTrusted)
			if (false == DisableHost(szHost, pConfig->GetInteger("Blocked", szHost, -1))) {
				lprintf("Error inserting blocking firewall rule when logging as user %s...\n", pszLogin);
				write_byte(sockfd, ERROR);
				return false;
			}
		write_byte(sockfd, OK);
		return true;
	}
	return false;
}

bool check_staff(char *pszLogin)
{
	struct passwd *pwLogin;
	const struct group *grLogin;
	char *pszAllowed;
	
	pwLogin = getpwnam(pszLogin);
	gid_t iLoginGroup = pwLogin->pw_gid;

	CTokenizedString cAllowedUsers(pConfig->GetString("Global", "staff", ""), ",;");

	if (NULL == (grLogin = getgrgid(iLoginGroup))) {
		lprintf("invalid group for user %s", pszLogin);
		return false;
	}
	if ((pszAllowed = cAllowedUsers.GetFirstString()))
		do {
			pszAllowed = TrimBoth(pszAllowed, " \t\x0a\x0d");
			if ('@' == pszAllowed[0]) {
			  	if (0 == strcmp(pszAllowed + 1, grLogin->gr_name))
					return true;
			} else {
			   if (0 == strcmp(pszLogin, pszAllowed))
					return true;
			}
		} while ((pszAllowed = cAllowedUsers.GetNextString()));
	return false;
}

bool passwdd_check_login(char *pszServer, int iServerPort, int newsockfd, char *pszLogin, char *pszPassword)
{
	unsigned char c = '\0';
	read_byte(newsockfd, &c);	
	if (WELCOME != c)
		return false;
	write_string(newsockfd, pszLogin);

	char szFileName[256];
	char *pszResult;	
	char *pszKeyDirectory = pConfig->GetString("Passwdd", "key_directory", "/var/spool/passwdc");

	strncpy(szFileName, pszKeyDirectory, sizeof(szFileName));
	strncat(szFileName, "/", sizeof(szFileName));
	strncat(szFileName, pszServer, sizeof(szFileName));
	
	RSAStorePublicKey();
	if (false == RSALoadPublicKey(szFileName)) {
		if ((pszResult = get_key(pszServer, iServerPort, pszKeyDirectory))) {
			RSARetreivePublicKey();
			return false;
		}
		if (false == RSALoadPublicKey(szFileName)) {
			RSARetreivePublicKey();
			return false;
		}
	}

	write_password(newsockfd, pszPassword);
	
	RSARetreivePublicKey();
	
	read_byte(newsockfd, &c);
	if (GO_AHEAD != c)
		return false;

	return true;
}

bool passwdd_add_user(int newsockfd, char *pszTarget, char *pszGecos)
{
	unsigned char c = '\0';
	
	write_byte(newsockfd, ACTION_ADDUSER);
	write_string(newsockfd, pszTarget);				// login
	write_string(newsockfd, pszGecos);				// info
	write_string(newsockfd, "");						// directory
	write_string(newsockfd, "");						// shell
	write_integer(newsockfd, -1);						// uid
	write_integer(newsockfd, -1);						// gid
	
	read_byte(newsockfd, &c);
	
	if (c == OK)
		write_string(newsockfd, "");

	return c == OK;
}

bool passwdd_delete_user(int newsockfd, char *pszTarget)
{
	unsigned char c = '\0';
	
	write_byte(newsockfd, ACTION_DELETEUSER);
	write_string(newsockfd, pszTarget);
	
	read_byte(newsockfd, &c);
	
	if (OK == c)
		write_string(newsockfd, "");
	
	return c == OK || c == INVALID_USER;
}

int connect_passwdd_server(char *pszServer, int iServerPort)
{
	pszServer = TrimBoth(pszServer, " \t\x0a\x0d");
	
	int sockfd;
	
	if ((sockfd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		lprintf("socket error: %s\n", strerror(errno));
		return -1;
	}

	struct hostent *host = gethostbyname(pszServer);
	if (NULL == host) {
		lprintf("resolving error: %s\n", pszServer);
		close(sockfd);
		return -1;
	}
	
	struct sockaddr_in client;

	memset(&client, 0, sizeof(client));
	client.sin_family = AF_INET;
	client.sin_addr.s_addr = *((unsigned long int *)(host->h_addr));
	client.sin_port = htons(iServerPort);
	
	if (connect(sockfd, (struct sockaddr *)&client, sizeof(client)) < 0) {
		lprintf("connect error: %s\n", strerror(errno));
		close(sockfd);
		return -1;
	}
	
	return sockfd;
}