#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "poll.hpp"
#include "conf.hpp"
#include "logfile.hpp"
#include "users.hpp"
#include "current.hpp"
#include "firewall.hpp"
#include "account.hpp"

extern CConfigurationFile *pConfig;

bool poll(void)
{
	SCurrentEntry strCurrentEntry;
	CCurrentDatabase *pCurrentDatabase = NULL;
	time_t tLoginTime;
	time_t tLogoutTime;
	char *pszMessage = NULL;
	int iInputTraffic;
	int iOutputTraffic;
	
	if (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage))) {
		lprintf("Memory allocation error polling current database...\n");
		return false;
	}
	if (NULL != pszMessage) {
		lprintf("%s\n", pszMessage);
		delete pCurrentDatabase;
		return false;
	}
	
	char **pRecordsToDelete = NULL;
	char **pNewRecordsToDelete = NULL;
	int iRecordsToDelete = 0;
	
// Update each entry in the current database	
	if (pCurrentDatabase->GetFirstRecord(&strCurrentEntry))
		do {
			iInputTraffic = GetTraffic(strCurrentEntry.szHost, INPUT_TRAFFIC);
			iOutputTraffic = GetTraffic(strCurrentEntry.szHost, OUTPUT_TRAFFIC);
			tLoginTime = strCurrentEntry.tStartTime;
			tLogoutTime = time(NULL);
	
			if ((-1 == iInputTraffic) || (-1 == iOutputTraffic)) {
				lprintf("Missing accounting firewall rule for host %s. Emergency down...\n", strCurrentEntry.szHost);
				EmergencyUserDown(&strCurrentEntry);
				pNewRecordsToDelete = (char **)realloc(pRecordsToDelete, (iRecordsToDelete + 1) * sizeof(char *));
				char *pszHost = strdup(strCurrentEntry.szHost);
				if (NULL == pNewRecordsToDelete || NULL == pszHost) {
					lprintf("Memory allocation error in scheduled polling current database...\n");
				} else {
					pRecordsToDelete = pNewRecordsToDelete;
					pRecordsToDelete[iRecordsToDelete] = pszHost;
					iRecordsToDelete += 1;
				}
			} else if (false == pCurrentDatabase->UpdateRecord(strCurrentEntry.szLogin,
																				strCurrentEntry.szDevice,
																				strCurrentEntry.szHost,
																				strCurrentEntry.szGateway,
																				tLogoutTime,
																				iInputTraffic,
																				iOutputTraffic)) {
				lprintf("Error updating current entry for user %s. Emergency down...\n", strCurrentEntry.szLogin);
				EmergencyUserDown(&strCurrentEntry);
				char *pszHost = strdup(strCurrentEntry.szHost);
				if (NULL == pNewRecordsToDelete || NULL == pszHost) {
					lprintf("Memory allocation error in scheduled polling...\n");
				} else {
					pRecordsToDelete = pNewRecordsToDelete;
					pRecordsToDelete[iRecordsToDelete] = pszHost;
					iRecordsToDelete += 1;
				}
			} 
		} while (pCurrentDatabase->GetNextRecord(&strCurrentEntry));

// Delete selected records
	for (int i = 0; i < iRecordsToDelete; i++) {
		pCurrentDatabase->DeleteRecord(pRecordsToDelete[i]);
		if (0 == strcmp("*", pRecordsToDelete[i]))
			// LAN user
			DisableHost(pRecordsToDelete[i], pConfig->GetInteger("Local", "port_allow", -1));
		else if (0 == strcmp("-", pRecordsToDelete[i]))
			// Blocked user
			DisableHost(pRecordsToDelete[i], pConfig->GetInteger("Blocked", pRecordsToDelete[i], -1));
		else
			// Dial-up user
			;
		delete pRecordsToDelete[i];
	}
	
	delete pRecordsToDelete;
	delete pCurrentDatabase;
	
	return true;
}

bool expired(void)
{
	SCurrentEntry strCurrentEntry;
	CUsersDatabase *pUsersDatabase = NULL;
	CCurrentDatabase *pCurrentDatabase = NULL;
	char *pszMessage1 = NULL;
	char *pszMessage2 = NULL;
	
	if ((NULL == (pUsersDatabase = new CUsersDatabase(pConfig, &pszMessage1))) ||
		 (NULL == (pCurrentDatabase = new CCurrentDatabase(pConfig, &pszMessage2)))) {
		lprintf("Memory allocation error polling current database...\n");
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	if (NULL != pszMessage1 || NULL != pszMessage2) {
		lprintf("%s\n", pszMessage1 ? pszMessage1 : pszMessage2);
		delete pUsersDatabase;
		delete pCurrentDatabase;
		return false;
	}
	
	char **pRecordsToDelete = NULL;
	char **pNewRecordsToDelete = NULL;
	int iRecordsToDelete = 0;
	
// Update each entry in the current database	
	if (pCurrentDatabase->GetFirstRecord(&strCurrentEntry))
		do {
			if (true == pUsersDatabase->ExpiredCheck(strCurrentEntry.szLogin,
																  pCurrentDatabase->GetTime(strCurrentEntry.szLogin),
																  pCurrentDatabase->GetTraffic(strCurrentEntry.szLogin))) {
				lprintf("Expired acount for user %s. Logging off...\n", strCurrentEntry.szLogin, strCurrentEntry.szHost);
				pNewRecordsToDelete = (char **)realloc(pRecordsToDelete, (iRecordsToDelete + 3) * sizeof(char *));
				char *pszLogin = strdup(strCurrentEntry.szLogin);
				char *pszDevice = strdup(strCurrentEntry.szDevice);
				char *pszHost = strdup(strCurrentEntry.szHost);
				char *pszGateway = strdup(strCurrentEntry.szGateway);
				if (NULL == pNewRecordsToDelete ||
					 NULL == pszLogin ||
					 NULL == pszDevice ||
					 NULL == pszHost) {
					lprintf("Memory allocation error in scheduled polling current database...\n");
				} else {
					pRecordsToDelete = pNewRecordsToDelete;
					pRecordsToDelete[iRecordsToDelete] = pszLogin;
					pRecordsToDelete[iRecordsToDelete + 1] = pszDevice;
					pRecordsToDelete[iRecordsToDelete + 2] = pszHost;
					pRecordsToDelete[iRecordsToDelete + 3] = pszGateway;
					iRecordsToDelete += 4;
				}
			} 
		} while (pCurrentDatabase->GetNextRecord(&strCurrentEntry));

	delete pCurrentDatabase;
	delete pUsersDatabase;

// Delete selected records
	for (int i = 0; i < (iRecordsToDelete / 4); i++) {
		UserDown(pRecordsToDelete[i * 4],
					pRecordsToDelete[i * 4 + 1],
					pRecordsToDelete[i * 4 + 2],
					pRecordsToDelete[i * 4 + 3]);
		if (0 == strcmp("*", pRecordsToDelete[i * 4 + 1]))			// Static users have "*" for device
			// LAN user
			DisableHost(pRecordsToDelete[i * 4 + 2], pConfig->GetInteger("Local", "port_allow", -1));
		else if (0 == strcmp("-", pRecordsToDelete[i * 3 + 1]))	// Blocked users have "-" for device
			// Blocked user
			DisableHost(pRecordsToDelete[i * 4 + 2], pConfig->GetInteger("Blocked", pRecordsToDelete[i * 4 + 2], 0));
		else																		// Dial-up users have valid device
			// Dial-up user
			;
		delete pRecordsToDelete[i * 4];
		delete pRecordsToDelete[i * 4 + 1];
		delete pRecordsToDelete[i * 4 + 2];
		delete pRecordsToDelete[i * 4 + 3];
	}
	
	delete pRecordsToDelete;
	
	return true;
}